/*!
  Manage metric time series data
  Copyright (c) 2009, 2010, Oracle and/or its affiliates. All rights reserved.
*/
/*global window,poller,mvc */
(function() {

/*
 Data structures used by this module.

  metric:
  {
    on: "<object-name>",
    a: "<attribute-name>",
    type: "<type>",
    server: "<server>"
  }

  metric series info (returned by getMetricSeries):
  {
    active: <true|false>,
    metric: <metric>,
    checkForHarvested: <bool>,
    data: [[<time-stamp>,<value>],...],
    stats: {
      since: <time-stamp>,
      min: <n>,
      max: <n>,
      avg: <n>,
      stdev: <n>,
      count: <n>,
      sum: <n>,
      sumOfSquares: <n>
    },
    lastShift: <n>, // number of samples shifted out of series on last update
    harvested: {
      "<range>": { // range key format: MM/DD/YYYY HH:MM;M
        checkForHarvested: <bool>,
        data: [...],
        stats: {...}
      },...
    }
  }

*/

var MIN_SAMPLE_PERIOD = 10; // 10 seconds
var samplePeriod = MIN_SAMPLE_PERIOD * 1000; // default
var maxSamples = 100; // default
var metrics = {};
var running = false;
var timerID = null;
var startPollTime = null;
var nextPollTime = null;
var changeListeners = [];
var harvestedTimer = null;
var harvestedReqCount = 0;

function makeMetricKey(metric) {
  return metric.server + ":" + metric.on + ":" + metric.a;
}

function fireChange(type) {
  var l = changeListeners;
  var i;
  for (i = 0; i < l.length; i++) {
    l[i](type);
  }
}

/*
 * Gather active metrics due to be sampled
 * Return null if no metrics active
 * The structure of active metrics is nested maps: server, object-name, attribute.
 */
function gatherActiveMetrics() {
  var m, k, o, onKey, server, serverKey;
  var am = {};
  var mcount = 0;
  for (k in metrics) {
    if (metrics.hasOwnProperty(k)) {
      m = metrics[k];
      if (m.active) {
        mcount += 1;
        serverKey = m.metric.server;
        onKey = m.metric.on;
        server = am[serverKey];
        if (!server) {
          server = am[serverKey] = {};
        }
        o = server[onKey];
        if (!o) {
          o = server[onKey] = {};
        }
        o[m.metric.a] = 0;
      }
    }
  }
  if (mcount > 0) {
    return am;
  }
  return null;
}

/*
 * Gather metrics that need harvested metric data filled in group results by range
 * Return null if none
 * The structure of active metrics is nested maps: server, object-name, attribute.
 */
function gatherHarvestedMetrics(current) {
  var m, k, range, h;
  var hm = {};
  var mcount = 0;
  
  function addMetricRange(range, m) {
    var o, onKey, server, serverKey, hmr;

    mcount += 1;
    serverKey = m.metric.server;
    onKey = m.metric.on;
    hmr = hm[range];
    if (!hmr) {
      hmr = hm[range] = {};
    }
    server = hmr[serverKey];
    if (!server) {
      server = hmr[serverKey] = {};
    }
    o = server[onKey];
    if (!o) {
      o = server[onKey] = {};
    }
    o[m.metric.a] = 0;
  }

  for (k in metrics) {
    if (metrics.hasOwnProperty(k)) {
      m = metrics[k];
      if (current) {
        if (m.checkForHarvested) {
          addMetricRange("current", m);
        }
      } else {
        h = m.harvested;
        for (range in h) {
          if (h.hasOwnProperty(range)) {
            if (h[range].checkForHarvested) {
              addMetricRange(range, m);
            }
          }
        }
      }
    }
  }
  if (mcount > 0) {
    return hm;
  }
  return null;
}

function scheduleForNextInterval() {
  var delta;
  var now = new Date().getTime();
  while (nextPollTime < now) {
    nextPollTime += samplePeriod;
  }
  delta = (nextPollTime - now) + 10; // 10 is fudge factor
  timerID = setTimeout(poller, delta);
}

function requestMetricValues(am, process) {
  var r;

  mvc.ajax("GET", "data/diagnostics/currentMetricValues", function(result) {
    if (!result) {
      fireChange("stop");
      running = false;
      return;
    }
    r = result.body;

    process(r);
  }, {metrics: am});
}

function requestHarvestedMetricValues(range, hmr, process) {
  var r, s;
  var data = {metrics: hmr};

  if (range == "current") {
    data.durationMinutes = Math.ceil(samplePeriod * maxSamples / 1000 / 60);
  } else {
    s = range.split(";");
    data.start = s[0] + ":00";
    data.durationMinutes = s[1];
    harvestedReqCount += 1;
  }

  mvc.ajax("GET", "data/diagnostics/harvestedMetricValues", function(result) {
    if (!result) {
      fireChange("stop");
      running = false;
      return;
    }
    r = result.body;
    process(range, r);
  }, data);
}

function updateStats(stats, time, value) {
  if (stats.since === null || time < stats.since) {
    stats.since = time;
  }
  stats.count += 1;
  stats.sum += value;
  stats.sumOfSquares += (value * value);
  stats.avg = stats.sum / stats.count;
  stats.stdev = Math.sqrt((stats.sumOfSquares / stats.count) - (stats.avg * stats.avg));
  if (stats.min === undefined || value < stats.min) {
    stats.min = value;
  }
  if (stats.max === undefined || value > stats.max) {
    stats.max = value;
  }
}

function updateMetric(metric, sampleTime, value) {
  var stats = metric.stats;
  var data = metric.data;
  var lastSampleTime;
  if (data.length > 0) {
    lastSampleTime = data[data.length - 1][0];
    if (sampleTime - lastSampleTime > (2 * samplePeriod)) {
      // insert a gap if too long between samples
      data.push([lastSampleTime + samplePeriod, null]);
    }
  }
  data.push([sampleTime, value]);
  updateStats(stats, sampleTime, value);
}

function updateMetrics(metric, range, values) {
  var o, i, value, copyFrom, len;
  var sampleTime;
  var period = null;
  var lastSampleTime = null;

  if (range == "current") {
    o = metric;
  } else {
    o = metric.harvested[range];
  }

  for (i = 0; i < values.length; i++) {
    value = values[i];
    sampleTime = value[0];

    if (lastSampleTime) {
      if (period && sampleTime - lastSampleTime > (2 * period)) {
        // insert a gap if too long between samples
        values.splice(i, 0, [lastSampleTime + period, null]);
        i += 1;
      }
      period = sampleTime - lastSampleTime;
    }
    lastSampleTime = sampleTime;
    updateStats(o.stats, sampleTime, value[1]);
  }

  len = o.data.length;
  if (len === 0) {
    copyFrom = values.length - maxSamples;
    copyFrom = copyFrom < 0 ? 0 : copyFrom;
    o.data = values.slice(copyFrom);
  } else {
    copyFrom = values.length - maxSamples - len;
    copyFrom = copyFrom < 0 ? 0 : copyFrom;
    o.data = values.slice(copyFrom).concat(o.data);
  }
  o.checkForHarvested = false;
}

function processResults(r) {
  var mr = r.metrics;
  var st = r.sampleTime;
  var on, metric, attrs, a, key, m, extra, server, serverMetrics;

  for (server in mr) {
    if (mr.hasOwnProperty(server)) {
      serverMetrics = mr[server];
      for (on in serverMetrics) {
        if (serverMetrics.hasOwnProperty(on)) {
          attrs = serverMetrics[on];
          for (a in attrs) {
            if (attrs.hasOwnProperty(a)) {
              metric = {server: server, on: on, a: a};
              key = makeMetricKey(metric);
              m = metrics[key];
              if (m && m.active) {
                updateMetric(m, st, attrs[a]);
                extra = m.data.length - maxSamples;
                m.lastShift = extra > 0 ? extra : 0;
                if (extra == 1) {
                  m.data.shift();
                } else if (extra > 1) {
                  m.data = m.data.slice(m.data.length - maxSamples);
                }
              }
            }
          }
        }
      }
    }
  }
  // do it again on next polling interval
  scheduleForNextInterval();

  // let listeners know about new data
  fireChange("newData");
}

function processHarvestedResults(range, r) {
  var mr = r.metrics;
  var on, metric, attrs, a, key, m, extra, server, serverMetrics;

  for (server in mr) {
    if (mr.hasOwnProperty(server)) {
      serverMetrics = mr[server];
      for (on in serverMetrics) {
        if (serverMetrics.hasOwnProperty(on)) {
          attrs = serverMetrics[on];
          for (a in attrs) {
            if (attrs.hasOwnProperty(a)) {
              metric = {server: server, on: on, a: a};
              key = makeMetricKey(metric);
              m = metrics[key];
              if (m && attrs[a]) {
                updateMetrics(m, range, attrs[a]);
              }
            }
          }
        }
      }
    }
  }

  // let listeners know about new data
  if (range == "current") {
    fireChange("newData");
  } else {
    harvestedReqCount -= 1;
    if (harvestedReqCount === 0) {
      fireChange("harvestedData");
    }
  }
}

function poller() {
  var am, hm, range;

  if (!running) {
    return;
  }
  am = gatherActiveMetrics();
  hm = gatherHarvestedMetrics(true);
  if (am === null && hm === null) {
    fireChange("stop");
    running = false;
    return;
  }

  if (am) {
    requestMetricValues(am, processResults);
  }
  if (hm) {
    for (range in hm) {
      if (hm.hasOwnProperty(range)) {
        requestHarvestedMetricValues(range, hm[range], processHarvestedResults);
      }
    }
  }
}

function startPoller() {
  if (running) {
    return;
  }
  fireChange("start");
  if (nextPollTime === null) {
    nextPollTime = new Date().getTime();
    // round up to next second
    nextPollTime = Math.floor((nextPollTime + 1000) / 1000) * 1000;
    startPollTime = nextPollTime;
  }
  scheduleForNextInterval();
  running = true;
}

function getHarvested() {

  if (harvestedTimer !== null) {
    return;
  }

  harvestedTimer = setTimeout(function() {
    var range;
    var hm = gatherHarvestedMetrics(false);

    harvestedTimer = null;
    if (hm === null) {
      return;
    }
    for (range in hm) {
      if (hm.hasOwnProperty(range)) {
        requestHarvestedMetricValues(range, hm[range], processHarvestedResults);
      }
    }
  }, 1000);

}


function propCount(obj) {
  var count = 0, p;
  for (p in obj) {
    if (obj.hasOwnProperty(p)) {
      count += 1;
    }
  }
  return count;
}

function emptyStats() {
  return { since: null, count: 0, sum: 0, sumOfSquares: 0 };
}

function emptyMetric(metric) {
  return { metric: metric, views: {}, data: [], harvested: {}, stats: emptyStats() };
}

window.wldfDM = {
  /*
   * This is the minimum amount of time in seconds
   * that samples will be polled. 
   */
  minSamplePeriod: MIN_SAMPLE_PERIOD,

  /*
   * Set the sample/polling period to, sec, number of seconds.
   * If less than minSamplePeriod then minSamplePeriod will be used.
   */
  setSamplePeriod: function(sec) {
    if (sec < MIN_SAMPLE_PERIOD) {
      sec = MIN_SAMPLE_PERIOD;
    }
    samplePeriod = sec * 1000;
  },

  /*
   * Get the sample/polling period. The return value is in seconds.
   */
  getSamplePeriod: function() {
    return samplePeriod / 1000;
  },

  /*
   * Set the maximum number of samples that will be kept for any metric.
   */
  setMaxSamples: function(max) {
    maxSamples = max;
  },

  /*
   * Get the maximum number of samples that will be kept for any metric.
   */
  getMaxSamples: function() {
    return maxSamples;
  },

  /*
   * Add a metric (if not alread present) and get harvested data for the given range.
   * OK to call more than once for the same metric, range pair.
   * This does not affect polling
   */
  registerRange: function(metric, range) {
    var key = makeMetricKey(metric);
    var m, hr;
    // validate range MM/DD/YYYY HH:MM;d
    if (!/^\d\d\/\d\d\/\d\d\d\d \d\d:\d\d;\d*$/.test(range)) {
      throw "Invalid range " + range; 
    }
    m = metrics[key];
    if (!m) {
      m = metrics[key] = emptyMetric(metric);
    }
    hr = m.harvested[range];
    if (!hr) {
      hr = m.harvested[range] = { count: 0, checkForHarvested: true, data: [], stats: emptyStats() };
      getHarvested();
    }
    hr.count += 1;
  },

  /*
   * Remove the given range for the given metric.
   * Call once for each time registerRange was called with same args.
   */
  unregisterRange: function(metric, range) {
    var key = makeMetricKey(metric);
    var m, hr;

    m = metrics[key];
    if (m) {
      hr = m.harvested[range];
      if (hr) {
        hr.count -= 1;
        if (hr.count === 0) {
          delete m.harvested[range];
        }
      }
    }
  },

  /*
   * Add a metric (if not alread present) and start polling it. This makes the metric active for the given view.
   */
  register: function(metric, view) {
    var key = makeMetricKey(metric);
    var m, v;
    if (!metrics[key]) {
      metrics[key] = emptyMetric(metric);
    }
    m = metrics[key];
    v = m.views[view];
    m.views[view] = v ? v + 1 : 1;
    if (m.active === undefined) {
      m.checkForHarvested = true;
    }
    m.active = true;
    startPoller();
  },

  /*
   * Mark the metric not active in the view and stop polling for the metric if this is the last
   * active view.
   */
  unregister: function(metric, view) {
    var key = makeMetricKey(metric);
    var m, v;
    m = metrics[key];
    if (m) {
      v = m.views[view];
      if (v) {
        v = m.views[view] -= 1;
        if (v === 0) {
          delete m.views[view];
          if (propCount(m.views) === 0) {
            m.active = false;
          }
        }
      }
    }
  },

  /*
   * Binds the function, fn, to the change event. 
   * When the poller starts or stops the function is called with argument "start" and "stop" respectively. 
   * When current metric data changes the function is called with argument "newData".
   * When new harvested metric data is added the function is called with argument "harvestedData".
   * The function can then call getMetricSeries to get series info for each metric of interest.
   */ 
  bind: function(fn) {
    var l = changeListeners;
    var i;
    for (i = 0; i < l.length; i++) {
      if (l[i] === fn) {
        return;
      }
    }
    l.push(fn);
  },

  /*
   * Un-binds the function, fn, from the change event.
   */   
  unbind: function(fn) {
    var l = changeListeners;
    var i;
    for (i = 0; i < l.length; i++) {
      if (l[i] === fn) {
        l.splice(i,1); // remove fn from list
        break; // there should only be one
      }
    }
  },

  /*
   * Remove all data for all metrics
   */
  clear: function() {
    var m, k;
    for (k in metrics) {
      if (metrics.hasOwnProperty(k)) {
        m = metrics[k];
        m.data = [];
        if (m.active) {
          m.checkForHarvested = true;
        }
        m.stats = emptyStats();
      }
    }
    fireChange("newdata");
  },

  /*
   * Reset stats for all metrics
   */
  resetStats: function() {
    var m, k;
    for (k in metrics) {
      if (metrics.hasOwnProperty(k)) {
        m = metrics[k];
        m.stats = emptyStats();
      }
    }
  },

  /*
   * Return true if there is any data stored for any metrics
   */
  hasData: function() {
    var m, k;
    for (k in metrics) {
      if (metrics.hasOwnProperty(k)) {
        m = metrics[k];
        if (m.data && m.data.length > 0) {
          return true;
        }
      }
    }
    return false;
  },

  /* 
   * Returns metric series info for the given metric or null if the metric is not found.
   */
  getMetricSeries: function(metric) {
    var key = makeMetricKey(metric);
    var m;
    if (metrics[key]) {
      return metrics[key];
    } else {
      return null;
    }
  }
};

})();
