/*!
  UI widget for Tree view
  Copyright (c) 2010 Oracle and/or its affiliates. All rights reserved.
*/
/*global jQuery,mvc */

(function($) {

  var VK_LEFT = 37,
    VK_UP = 38,
    VK_RIGHT = 39,
    VK_DOWN = 40,
    VK_HOME = 36,
    VK_END = 35,
    VK_PGUP = 33,
    VK_PGDN = 34,
    VK_ENTER = 14,
    VK_RETURN = 13,
    VK_ESCAPE = 27,
    VK_DEL = 46,
    VK_F2 = 113;

  var CLASS_TREEVIEW = "treeView";

  var EVENT_CHANGE = "change";

  var DATA_LAST_FOCUSED = "lastFocused";

  var NODE_LABEL_SLCTR ="span.nodelabel",
    TREE_ITEM_SLCTR = "span.treeItem",
    SELECTED_CLASS = "selected",
    SELECTED_SLCTR = ".treeItem.selected",
    FOCUSED_CLASS = "focused",
    EXPANDABLE_CLASS = "expandable",
    COLLAPSABLE_CLASS = "collapsable",
    PROCESSING_CLASS = "processing",
    LEAF_CLASS = "leaf";

  function getIdFromNode($n) {
    var id = $n.get(0).id;
    return id.substring(id.lastIndexOf("_") + 1);
  }

  function renderTreeItemContent(node, out, nodeAdapter) {
    if (nodeAdapter.renderIcon) {
      nodeAdapter.renderIcon(node, out);
    }
    out.markup("<span class='nodelabel' tabIndex='-1'>");
    nodeAdapter.renderLabel(node, out);
    out.markup("</span>");
  }

  $.widget("ui.treeView", {
    _init: function() {
      var self = this;
      var $ctrl = this.element;
      var o = this.options;
      var nodeAdapter = o.nodeAdapter;
      var out = mvc.htmlBuilder();
      var rootNode, ul, $li;

      o.treeMap = {}; // mapping of li node id to node
      o.nextNodeId = 0;

      if (o.collapsibleRoot === false) {
        o.expandRoot = true;
      }

      // should the treeView contain a menu/toolbar? external for now
      out.markup("<div class='treeContainer' style='position:relative;overflow:auto;height:100%;'><ul>");
      rootNode = nodeAdapter.root(); //get the single root node
      if (o.showRoot) {
        this._renderNode(rootNode, out);
      } else {
        if (nodeAdapter.hasChildren(rootNode)) {
          this._renderChildren(rootNode, out);
        }
      }

      out.markup("</ul></div>");
      $ctrl.html(out.toString()).addClass(CLASS_TREEVIEW);

      if (o.expandRoot) {
        $li = $ctrl.find("ul:first li:first");
        if ($li.length > 0) {
          this._expandNode($li);
        }
      }

      $ctrl.bind("click.treeView", function(e) {
        var $target = $(e.target);
        var $parent, $node, $prevSel;

        if ($target.is(".hitarea")) {
          $parent = $target.parent();
          if ($parent.hasClass(EXPANDABLE_CLASS)) {
            self._expandNode($parent);
          } else {
            self._collapseNode($parent);
          }
          return false;
        } else {
          $node = $target.closest(TREE_ITEM_SLCTR);
          if ($node.length > 0) {
            self.selectNode($node, false);
            return false;
          }
        }
      }).bind("keydown.treeView", function(e) {
        var kc = e.keyCode;
        var $target, lastFocused, nh, page, $li;

        // ignore if target is the input for add/rename
        if (e.altKey || e.target.nodeName == "INPUT") {
          return;
        }
        if (e.keyCode == VK_PGUP || e.keyCode == VK_PGDN) {
          nh = $ctrl.find(".nodelabel:visible:first").outerHeight() || 24;
          $li = $ctrl.find("li:visible:first");
          nh += parseInt($li.css("margin-top"), 10) + parseInt($li.css("margin-bottom"), 10);
          page = Math.floor($ctrl.find(".treeContainer").get(0).clientHeight / nh) - 1;
        }
        if (kc == VK_HOME) {
          $ctrl.find(".treeItem:visible:first").each(function() { // at most once
            self.selectNode($(this), true, true);
          });
          return false;
        } else if (kc == VK_END) { 
          $ctrl.find(".treeItem:visible:last").each(function() { // at most once
            self.selectNode($(this), true, true);
          });
          return false;
        } else if (kc == VK_DOWN) {
          self._traverseDown(1);
          return false;
        } else if (kc == VK_UP) {
          self._traverseUp(1);
          return false;
        } else if (kc == VK_PGDN) {
          self._traverseDown(page);
          return false;
        } else if (kc == VK_PGUP) {
          self._traverseUp(page);
          return false;
        } else if (kc == VK_LEFT) {
          // If the focussed node is collapsible, collapse it.
          lastFocused = self._getData(DATA_LAST_FOCUSED);
          if (lastFocused) {
            $target = $(lastFocused).parent().parent();
            if ($target.hasClass(COLLAPSABLE_CLASS)) {
              self._collapseNode($target);
            } else {
              // If it is not collapsible, focus parent.
              $target.parent().prev(TREE_ITEM_SLCTR).each(function() { // at most once
                self.selectNode($(this), true, true);
              });
            }
          }
          return false;
        } else if (kc == VK_RIGHT) {
          // If the focused node is not a leaf, expand or move to descendant
          lastFocused = self._getData(DATA_LAST_FOCUSED);
          if (lastFocused) {
            $target = $(lastFocused).parent().parent();
            if ($target.hasClass(EXPANDABLE_CLASS)) {
              self._expandNode($target);
            } else if ($target.hasClass(COLLAPSABLE_CLASS)) {
              $target.find("ul:first li:first .treeItem").each(function() { // at most once
                self.selectNode($(this), true, true);
              });
            }
          }
          return false;
        } else if (kc == VK_F2) {
          lastFocused = self._getData(DATA_LAST_FOCUSED);
          if (lastFocused) {
            self._rename($(lastFocused).parent());
          }
        } else if (kc == VK_DEL) {
          lastFocused = self._getData(DATA_LAST_FOCUSED);
          if (lastFocused) {
            self._delete($(lastFocused).parent());
          }
        }
      });

      ul = $ctrl.find("ul:first").get(0);
      mvc.bindFocus(ul, function(e) {
        var $node = $(e.target).closest(NODE_LABEL_SLCTR);
        if ($node.length === 0) {
          return;
        }
        $node.addClass(FOCUSED_CLASS);
        self._setFocusable($node);
      });
      mvc.bindBlur(ul, function(e) {
        var $target = $(e.target).closest(NODE_LABEL_SLCTR);
        $target.removeClass(FOCUSED_CLASS);
      });

      // Set initial focus to first node
      self._setData(DATA_LAST_FOCUSED, null);
      $ctrl.find("ul:first .nodelabel:visible:first").each(function() { // at most once
        this.tabIndex = 0;
        self._setData(DATA_LAST_FOCUSED, this);
      });

    },

    selectNode: function($node, focus, delayTrigger) {
      var $ctrl = this.element;
      var self = this;
      var o = this.options;
      var $prevSel = $ctrl.find(SELECTED_SLCTR);
      var $li, $lbl;

      if ($prevSel.length > 0) {
        $prevSel.removeClass(SELECTED_CLASS);
      }
      $node.addClass(SELECTED_CLASS);

      // make sure parents expanded
      $li = $node.parent().parents("li:first");
      while ($li.length > 0 && $li.hasClass("expandable")) {
        this._expandNode($li);
        $li = $li.parents("li:first");
      }

      $lbl = $node.find(NODE_LABEL_SLCTR);
      if (focus) {
        mvc.setFocus($lbl.get(0), 0);
      } else {
        this._setFocusable($lbl);
      }
      if ($prevSel.get(0) !== $node.get(0)) {
        // use a timer to make sure the focus happens first and also throttle 
        // rapid changes from keyboard navigation.
        if (o.triggerTimerId) {
          clearTimeout(o.triggerTimerId);
          o.triggerTimerId = null;
        }
        o.triggerTimerId = setTimeout(function() {
          o.triggerTimerId = null;
          self._trigger(EVENT_CHANGE, 0);
        }, delayTrigger ? 350 : 1);
      }
    },

    // return jquery object with 0 or more selected treeItem elements
    selection: function() {
      var $ctrl = this.element;
      return $ctrl.find(SELECTED_SLCTR);
    },

    getNodes: function($nodes) {
      var $ctrl = this.element;
      var o = this.options;
      var nodes = [];
      $nodes.each(function() {
        nodes.push(o.treeMap[getIdFromNode($(this).closest("li"))]);
      });
      return nodes;
    },

    selectionNodes: function() {
      return this.getNodes(this.selection());
    },

    find: function($parentNode, match, depth) {
      var self = this;
      var $ctrl = this.element;
      var o = this.options;
      var $children, $target;
      var $node = null;
      var node;
      depth = depth || 1;

      if (!$parentNode && !o.showRoot) {
        $children = $ctrl.find("ul:first > li");
      } else {
        $target = $parentNode ? $parentNode.parent() : $ctrl.find("ul:first > li");
        this._addChildrenIfNeeded($target);
        $children = $target.find("ul:first > li");
      }
      $children.each(function(i) {
        node = o.treeMap[getIdFromNode($(this))];
        if (match(node)) {
          $node = $(this).children(TREE_ITEM_SLCTR);
          return false;
        }
      });
      if ($node === null && depth > 1) {
        $children.each(function(i) {
          $node = self.find($(this).children(TREE_ITEM_SLCTR), match, depth - 1);
          if ($node !== null) {
            return false;
          }
        });
      }
      return $node;
    },

    addNode: function($parentNode, name, context) {
      var self = this;
      var $ctrl = this.element;
      var o = this.options;
      var idPrefix = ($ctrl.get(0).id || "") + "_";
      var addId = idPrefix + "new";
      var nodeAdapter = o.nodeAdapter;
      var $ul, $newNode, parent, input;
      var completed = false;
      var lastFocused = self._getData(DATA_LAST_FOCUSED);

      function cancel() {
        $newNode.remove();
        self._makeLeafIfNeeded($parentNode);
        mvc.setFocus(lastFocused, 0);
      }

      function complete(newName) {
        if (completed) {
          return;
        }
        completed = true;
        nodeAdapter.addNode(parent, newName, context, function(child, index) {
          var $node, out;

          if (child === false) {
            // try again
            completed = false;
            input = $newNode.find("input").val(name).get(0);
            mvc.setFocus(input, 0);
            input.select();
            return;
          }
          if (child) {
            $newNode.remove();
            out = mvc.htmlBuilder();
            self._renderNode(child, out);
            if (index >= $ul.children("li").length) {
              $ul.append(out.toString());
            } else {
              $ul.children("li").eq(index).before(out.toString());
            }
            $node = $ul.children("li").eq(index);
            self.selectNode($node.children(TREE_ITEM_SLCTR), true);
          } else {
            cancel();
          }
        });
      }

      function addInput() {
        $ul.append("<li><span id='" + addId + "' class='treeItem'><span class='nodelabel'><input type='text'></span></span></li>");
        $newNode = $ul.find("#" + addId).parent();
        input = $newNode.find("input").val(name).keydown(function(e) {
          if (e.keyCode == VK_ENTER || e.keyCode == VK_RETURN) {
            complete($(this).val());
            return false;
          } else if (e.keyCode == VK_ESCAPE) {
            setTimeout(function() {cancel();}, 10);
            return false;
          }
        }).blur(function(e) {
          complete($(this).val());
        }).get(0);
        mvc.setFocus(input, 0);
        input.select();
      }

      if ($parentNode === null) {
        parent = nodeAdapter.root();
        if (o.showRoot) {
          $parentNode = $ctrl.find("ul:first > li");
        } else {
          $ul = $ctrl.find("ul:first");
          addInput();
          return;
        }
      } else {
        parent = o.treeMap[getIdFromNode($parentNode.parent())];
      }
      self._makeParentIfNeeded($parentNode);
      this._expandNode($parentNode.parent(), function() {
        $ul = $parentNode.next("ul");
        addInput();
      });
    },

    renameNode: function($node) {
      var self = this;
      var $ctrl = this.element;
      var o = this.options;
      var nodeAdapter = o.nodeAdapter;
      var node, input, oldName;
      var $target = $node.parent();
      var nodeId = getIdFromNode($target);
      var completed = false;
      var out = mvc.htmlBuilder();

      function cancel() {
        out.clear();
        renderTreeItemContent(node, out, nodeAdapter);
        $node.html(out.toString());
        self.selectNode($node, true);
      }

      function complete(newName) {
        if (completed) {
          return;
        }
        completed = true;
        if (newName == oldName) {
          cancel();
          return;
        }
        nodeAdapter.renameNode(node, newName, function(renamedNode, index) {
          var oldIndex;
          var $ul, $children;

          if (renamedNode === false) {
            // try again
            completed = false;
            input = $node.find("input").val(oldName).get(0);
            mvc.setFocus(input, 0);
            input.select();
            return;
          }
          if (renamedNode) {
            out.clear();
            renderTreeItemContent(renamedNode, out, nodeAdapter);
            $node.html(out.toString());
            o.treeMap[nodeId] = renamedNode; // update map in case node changed
            $ul = $target.parent();
            $children = $ul.children("li");
            oldIndex = $children.index($target);
            if (oldIndex != index) {
              if (index > oldIndex) {
                index += 1;
              }
              if (index >= $children.length) {
                $ul.append($target);
              } else {
                $children.eq(index).before($target);
              }
            }
            self.selectNode($node, true);
            // the DOM node didn't change so selectNode won't fire the change event - force it
            self._trigger(EVENT_CHANGE, 0); 
          } else {
            cancel();
          }
        });
      }

      node = o.treeMap[nodeId];
      oldName = nodeAdapter.getName(node);
      $node.html("<input type='text'>");
      input = $node.find("input").val(oldName).keydown(function(e) {
        if (e.keyCode == VK_ENTER || e.keyCode == VK_RETURN) {
          complete($(this).val());
          return false;
        } else if (e.keyCode == VK_ESCAPE) {
          setTimeout(function() {cancel();}, 10);
          return false;
        }
      }).blur(function(e) {
        complete($(this).val());
      }).get(0);
      mvc.setFocus(input, 0);
      input.select();
    },

    deleteNode: function($node) {
      var o = this.options;
      var $parent = $node.parents("ul").eq(0).prev();
      var id = getIdFromNode($node.parent());
      var $li, $next, next;
      var $lbl = $node.children(NODE_LABEL_SLCTR);
      var focused = $lbl.hasClass(FOCUSED_CLASS);
      var thisLastFocused = this._getData(DATA_LAST_FOCUSED) == $lbl.get(0);
      var selected = $node.hasClass(SELECTED_CLASS);

      delete o.treeMap[id];
      $li = $node.parent();
      if (selected || thisLastFocused) {
        // select next closest node
        $next = $li.next();
        if ($next.length === 0) {
          $next = $li.prev();
          if ($next.length > 0) {
            if ($next.hasClass(COLLAPSABLE_CLASS)) {
              $next = $next.find("li:visible:last");
            }
          } else {
            $next = $li.parent().parent("li");
          }
        }
        if (thisLastFocused) {
          if ($next.length > 0) {
            next = $next.find(".treeItem > .nodelabel").get(0);
            next.tabIndex = 0;
            this._setData(DATA_LAST_FOCUSED, next);
          } else {
            this._setData(DATA_LAST_FOCUSED, null);
          }
        }
        if ($next.length > 0) {
          if (selected) {
            this.selectNode($next.children(TREE_ITEM_SLCTR), focused);
          }
        }
      }
      $li.remove();
      this._makeLeafIfNeeded($parent);
    },

    destroy: function() {
      var $ctrl = this.element;

      $ctrl.unbind(".treeView");
      $ctrl.html("").removeClass(CLASS_TREEVIEW);
      $.widget.prototype.destroy.apply(this, arguments); // default destroy
    },

    _renderNode: function(node, out) {
      var o = this.options;
      var nodeAdapter = o.nodeAdapter;
      var $ctrl = this.element;
      var idPrefix = ($ctrl.get(0).id || "") + "_";
      var hasChildren, nextId, nodeClass;

      nextId = o.nextNodeId;
      o.treeMap[nextId] = node;
      o.nextNodeId += 1;

      if (nextId === 0 && o.showRoot && !o.collapsibleRoot) {
        hasChildren = false; // probably does but don't want hit area
        nodeClass = EXPANDABLE_CLASS;
      } else {
        hasChildren = nodeAdapter.hasChildren(node);
        nodeClass = hasChildren ? EXPANDABLE_CLASS : LEAF_CLASS;
      }

      out.markup("<li id='").attr(idPrefix + nextId).markup("' class='").attr(nodeClass).markup("'>");
      if (hasChildren === null || hasChildren === true) {
        // if have children or not sure if there are children then display the hit area
        out.markup("<div class='hitarea'/>");
      }
      out.markup("<span class='treeItem'>");
      renderTreeItemContent(node, out, nodeAdapter);
      out.markup("</span>");
      // do lazy rendering - don't add children until expanded
      out.markup("</li>");
    },

    _renderChildren: function(node, out, fn, $target) {
      var self = this;
      var o = this.options;
      var nodeAdapter = o.nodeAdapter;
      var len;
      function doit() {
        var i;
        for (i = 0; i < len; i++) {
          self._renderNode(nodeAdapter.child(node, i), out);
        }
        if (fn) {
          fn(true);
        }
      }
      len = nodeAdapter.childCount(node);
      if (len === null) {
        if (fn) {
          //give feedback
          $target.removeClass(COLLAPSABLE_CLASS).addClass(PROCESSING_CLASS);
          nodeAdapter.fetchChildNodes(node, function(status) {
            if (status) {
              len = nodeAdapter.childCount(node);
              doit();
            } else {
              fn(false);
            }
            // remove feedback
            $target.removeClass(PROCESSING_CLASS).addClass(COLLAPSABLE_CLASS);
          });
        }
      } else {
        doit();
      }
    },

    _makeParentIfNeeded: function($parent) {
      if ($parent && $parent.prev("div").length === 0) {
        $parent.parent().removeClass(LEAF_CLASS);
        $parent.before("<div class='hitarea'></div>");
        $parent.after("<ul></ul>");
      }
    },
  
    _makeLeafIfNeeded: function($parent) {
      if ($parent && $parent.next("ul").find("li").length === 0) {
        $parent.parent().removeClass(EXPANDABLE_CLASS).removeClass(COLLAPSABLE_CLASS).addClass(LEAF_CLASS);
        $parent.prev(".hitarea").remove();
        $parent.next("ul").remove();
      }
    },

    // Add children nodes to the tree without expanding
    // Will not work with lazy rendered nodes
    // $target is the li of the parent node to add to
    _addChildrenIfNeeded: function($target) {
      var o = this.options;
      var nodeAdapter = o.nodeAdapter;
      var node = o.treeMap[getIdFromNode($target)];
      var $ul, out;

      $ul = $target.children("ul");
      if ($ul.length > 0 || $target.hasClass(LEAF_CLASS)) {
        return; // a leaf or already added so nothing to do
      } else {
        out = mvc.htmlBuilder();
        out.markup("<ul>");
        this._renderChildren(node, out);
        out.markup("</ul>");
        $target.append(out.toString()).children("ul").hide();
      }
    },

    // $target is the li of the parent node to expand
    _expandNode: function($target, fn) {
      var o = this.options;
      var nodeAdapter = o.nodeAdapter;
      var node = o.treeMap[getIdFromNode($target)];
      var $ul, out;
      $target.removeClass(EXPANDABLE_CLASS).addClass(COLLAPSABLE_CLASS);
      $ul = $target.children("ul");
      if ($ul.length > 0 && nodeAdapter.childCount(node) !== null) {
        $ul.show(); // allready rendered so show it
        if (fn) {
          fn();
        }
      } else {
        $ul.remove(); // remove if any
        out = mvc.htmlBuilder();
        out.markup("<ul>");
        this._renderChildren(node, out, function(status) {
          if (status) {
            out.markup("</ul>");
            $target.append(out.toString());
            if (fn) {
              fn();
            }
          }
        }, $target);
      }
    },

    // $target is the li of the parent node to collapse
    _collapseNode: function($target) {
      $target.removeClass(COLLAPSABLE_CLASS).addClass(EXPANDABLE_CLASS);
      if ($target.find(SELECTED_SLCTR).length > 0) {
        this.selectNode($target.children(TREE_ITEM_SLCTR), true);
      }
      $target.children("ul").hide();
    },

    _delete: function($node) {
      var self = this;
      var o = this.options;
      var nodeAdapter = o.nodeAdapter;
      var node = o.treeMap[getIdFromNode($node.parent())];

      if (nodeAdapter.allowDelete(node)) {
        nodeAdapter.deleteNode(node, function(status) {
          if (status) {
            self.deleteNode($node);
          }
        });
      }
    },

    _rename: function($node) {
      var o = this.options;
      var nodeAdapter = o.nodeAdapter;
      var node = o.treeMap[getIdFromNode($node.parent())];

      if (nodeAdapter.allowRename(node)) {
        this.renameNode($node);
      }
    },

    _traverseDown: function(count) {
      var $target, $next, i;
      var lastFocused = this._getData(DATA_LAST_FOCUSED);

      if (!lastFocused) {
        return;
      }
      $target = $(lastFocused).parent().parent();
      for (i = 0; i < count; i++) {
        // First try the child li, then sibling li, finally parent's sibling if any.
        if ($target.hasClass(COLLAPSABLE_CLASS)) {
          $target = $target.find("ul:first li:first"); 
        } else {
          // Look for next sibling, if not found, move up and find next sibling.
          $next = $target.next();
          if ($next.length > 0) {
            $target = $next;
          } else {
            $next = $target.parents("ul").eq(0).parents("li").next("li");
            if ($next.length === 0) {
              break;
            }
            $target = $next;
          }
        }
      }
      if ($target.length > 0) {
        this.selectNode($target.children(TREE_ITEM_SLCTR), true, true);
      }
    },

    _traverseUp: function(count) {
      var $target, $prev, i;
      var lastFocused = this._getData(DATA_LAST_FOCUSED);

      if (!lastFocused) {
        return;
      }
      $target = $(lastFocused).parent().parent();
      for (i = 0; i < count; i++) {
        // First try previous last child, then previous, finally parent if any
        $prev = $target.prev();
        if ($prev.length > 0) {
          if ($prev.hasClass(COLLAPSABLE_CLASS)) {
            $target = $prev.find("li:visible:last");
          } else {
            $target = $prev;
          }
        } else {
          $prev = $target.parent().parent("li");
          if ($prev.length === 0) {
            break;
          }
          $target = $prev;
        }
      }
      if ($target.length > 0) {
        this.selectNode($target.children(TREE_ITEM_SLCTR), true, true);
      }
    },

    _setFocusable: function($node) {
      var node = $node.get(0);
      var lastFocused = this._getData(DATA_LAST_FOCUSED);
      if (lastFocused && lastFocused !== node) {
        lastFocused.tabIndex = -1;
      }
      node.tabIndex = 0;
      this._setData(DATA_LAST_FOCUSED, node);
    }

  });

  $.extend($.ui.treeView, {
    getter: ["selection", "selectionNodes", "getNodes", "find"],
    defaults: {
      showRoot: true, // if false the tree appears like a forest (multi-rooted)
      expandRoot: true, // if true the root node is initially expanded
      collapsibleRoot: true, // if false the root node cannot be collapsed (has no hit area)
      multiSelect: false, // TODO true not supported yet
      nodeAdapter: null, /* adapter object 
        {
          root: fn(), // returns root node
          getName: fn(n), // returns name of node. Used for editing during rename
          child: f(n, i), // return the ith child of node n
          childCount: fn(n), // returns number of children the node n has, null if the answer is not yet known
          hasChildren: fn(n), // returns true if the node has children, false if it does not and null if not yet known
          childrenAllowed: fn(n), returns true if node can have children.
          renderIcon: fn(n, out), // optional. render icon to go before label
          renderLabel: fn(n, out), // render node label markup to htmlBuilder out
          fetchChildNodes: fn(n, complete(status)),  // called to let the adapter fetch child nodes. May be called after
                                                    // child count returns null. The completion call back status is
                                                    // true for success (0 or more children were added) and false if there
                                                    // was an error
          addNode: fn(parent, name, context, fn(child, index)), // child is the node to add. If child is false try again.
                                                    // If child is null then add failed - node is removed.
                                                    // Index is the position to insert the node. 
          renameNode: fn(n, newName, fn(node, index)), // n is the node to rename with newName.
                                                    // In the call back node is the renamed node (mostlikely the same)
                                                    // If node is false try again.
                                                    // If node is null then rename failed return to previous value.
                                                    // Index is the new position to move the node to.
          deleteNode: fn(n, fn(status)), // Called in response to DEL key. Delete node n then call fn with true to delete
                                         // the tree node or false to cancel the delete.
          allowRename: fn(n), // return true if the node can be renamed 
          allowDelete: fn(n)  // return true if the node can be deleted
        }
        */
      triggerTimerId: null // internal use
    }
  });

})(jQuery);
