/*!
  Core MVC support
  Copyright (c) 2009, 2010, Oracle and/or its affiliates. All rights reserved.
*/
/*global window,$ */

// This is the singleton object for all Model View Controler methods
var mvc = {};

(function(mvc) {

var BUNDLE = "mvccore";
var highContrastMode = false;

/*
 * Anticipate the next version of JavaScript in a safe way
 */
if (typeof Object.create !== 'function') {
  Object.create = (function() {
    function F() {}
    return (function (o) {
      F.prototype = o;
      return new F();
    });
  })();
}

function parseQueryParams() {
  var q = location.search;
  var hash = q.indexOf("#");
  var params = {};

  if (hash >=0) {
    q = q.substring(hash + 1);
  }
  if (q.charAt(0) == "?") {
    q = q.substring(1);
  }
  q.replace(/([^=&]+)=([^&]*)/g, function(m, k, v) {
    var key = decodeURIComponent(k);
    var value = decodeURIComponent(v);
    if (params.hasOwnProperty(key)) {
      if ($.isArray(params[key])) {
        params[key].push(value);
      } else {
        params[key] = [params[key], value];
      } 
    } else {
      params[key] = value;
    }
  });
  return params;
}

mvc.getContextRoot = function() {
  var p = location.pathname.split("/");
  var cr = "";
  if (p.length > 1) {
    cr = "/" + p[1];
  }
  return cr;
};

/*
 * If firebug not available do support minimal debug logging functionality but only when debug param
 */
var queryParams = parseQueryParams();

if (!window.console) {
  if (queryParams.debug === "1") {
    $(document).ready(function() {
      $("body").append("<div id='consoleLog'><textarea rows='6' cols='120'></textarea></div>");
    });
  
    window.console = {
      log: function(m) {
        var $logBuffer = $("#consoleLog textarea");
        $logBuffer.val($logBuffer.val() + "\n" + m);
        // scroll to the bottom
        var logBuffer = $logBuffer.get(0);
        if (logBuffer) {
          logBuffer.scrollTop = logBuffer.scrollHeight;
        }
      }
    };
  } else {
    window.console = {
      log: function(m) { }
    };
  }
}

mvc.logError = function(e) {
  var message;
  if (e.constructor === ReferenceError) {
    message = "Reference error: " + e.message;
  } else if (e.constructor === Error) {
    message = "Error: " + e.message;
  } else if (e.constructor === RangeError) {
    message = "Range error: " + e.message;
  } else if (e.constructor === SyntaxError) {
    message = "Syntax error: " + e.message;
  } else if (e.constructor === TypeError) {
    message = "Type error: " + e.message;
  } else if (e.message) {
    message = e.message;
  } else if (typeof e === "string") {
    message = e;
  } else {
    message = "unknown error";
  }
  if (e.fileName) {
    message += " @ " + e.fileName;
  }
  if (e.lineNumber) {
    message += "(line " + e.lineNumber + ")";
  }
  if (console && console.log) {
    console.log(message);
  }
};

var severityToIcon = {
  "FATAL": "fatal",
  "ERROR": "error",
  "WARNING": "warn",
  "CONFIRMATION": "confirm",
  "INFORMATION": "info"
};
var severityToText = null;

function defaultReportStatus(xhrStatus, xhrStatusText, messages) {
  var i, m;
  var out = mvc.htmlBuilder();
  var error = "ERROR";

  function addMessage(severity, message, details) {
    out.markup("<li><div><span class='icon ").attr(severityToIcon[severity])
      .markup("'></span><span class='screenReadersOnly'>").content(severityToText[severity]).markup(" </span>")
      .content(message).markup("</div>");
    if (details) {
      out.markup("<div>").content(mvc.getMessage(BUNDLE, "message.details") + " ")
        .content(details).markup("</div>");
    }
    out.markup("</li>");
  }

  if (severityToText === null) {
    severityToText = {
      "FATAL": mvc.getMessage(BUNDLE, "message.severity.fatal"),
      "ERROR": mvc.getMessage(BUNDLE, "message.severity.error"),
      "WARNING": mvc.getMessage(BUNDLE, "message.severity.warning"),
      "SUCCESS": mvc.getMessage(BUNDLE, "message.severity.confirmation"),
      "NORMAL": mvc.getMessage(BUNDLE, "message.severity.information")
    };
  }
  out.markup("<ul class='messageList'>");
  if (xhrStatus === 0 || xhrStatus == 12029) {
    addMessage(error, mvc.getMessage(BUNDLE, "message.status.connect"));
  } else if (xhrStatus == 404) {
    addMessage(error, mvc.getMessage(BUNDLE, "message.status.404"));
  } else if (xhrStatus == 500) {
    addMessage(error, mvc.getMessage(BUNDLE, "message.status.500"));
  } else if (xhrStatus < 200 || xhrStatus > 299) {
    addMessage(error, mvc.format(BUNDLE, "message.status.other", xhrStatus, xhrStatusText));
  } else {
    if (messages) {
      for (i = 0; i < messages.length; i++) {
        m = messages[i];
        addMessage(m.severity, m.message, m.details);
      }
    }
  }
  out.markup("</ul>");
  mvc.doMessageDialog(mvc.getMessage(BUNDLE, "message.dialog.title"), out.toString());
}

function isLoginPage(data) {
  return data.indexOf("<!DOCTYPE HTML") >= 0 && data.indexOf('action="' + mvc.getContextRoot() + '/j_security_check"') > 0;
}

var busyWait = 200;
var busyLinger = 1000;
var asyncTimer = null;
var activeXHRCount = 0;
var CSRFtoken = null;
var CSRF_HEADER = "X-WriteToken";
var defaultAjaxOptions = {
  processing: null, // fn(busy) called with busy true when time to show prossing active indication and false when time to stop
  sessionExpired: null, // fn() called when the session has expired
  reportStatus: defaultReportStatus
};
var ajaxOptions = defaultAjaxOptions;

function addQueryParam(url, param, value) {
  return url + (url.indexOf("?") >= 0 ? "&" : "?") + param + "=" + encodeURIComponent(value);
}

mvc.ajaxOptions = function(options) {
  ajaxOptions = $.extend({}, defaultAjaxOptions, options);
};

// fn and data are optional but you usually have at least one of them
mvc.ajax = function(method, url, fn, data) {
  var getOverPost = false;
  var savedXhr = null;

  if (!data) {
    data = null;
  }
  if (!$.isFunction(fn)) {
    if (fn !== null && fn !== undefined) {
      data = fn;
    }
    fn = null;
  }
  if (data) {
    // convert it to a JSON string
    data = JSON.stringify(data);
  }
  if (method == "MULTI_DELETE") {
    method = "POST";
    url = addQueryParam(url, "_method", "MULTI_DELETE");
  } else if (method == "GET" && data) {
    if (data.length > 1500) { // 1500 max for the query should keep the total URL under 2000
      method = "POST";
      url = addQueryParam(url, "_method", "GET");
      getOverPost = true;
    } else {
      url = addQueryParam(url, "_json", data);
      data = null;
    }
  }

  var start = (new Date()).getTime();
  var options = {
    type: method,
    url: url,
    data: data,
    processData: false,
    dataType: "json",
    global: false,
    beforeSend: function(xhr) {
      var pos;
      var w;

      savedXhr = xhr;
      // CSRF processing
      if ((method == "POST" && !getOverPost) || method == "PUT" || method == "DELETE") {
        if (CSRFtoken !== null) {
          xhr.setRequestHeader(CSRF_HEADER, CSRFtoken);
        }
      }
      //console.log("--> before send: " + activeXHRCount);
      // if the request is quick don't bother showing any spinner but after busyWait do
      if (activeXHRCount === 1 && asyncTimer === null) {
        asyncTimer = setTimeout(function() {
          asyncTimer = null;
          //console.log("--> processing start " + (asyncTimer === null));
          if (ajaxOptions.processing) {
            ajaxOptions.processing(true);
          }
        }, busyWait);
      }
    },
    dataFilter: function(data, type) {
      if (isLoginPage(data)) {
        // Its the login page, the session must have expired
        return { sessionExpired: true };
      }
      return data;
    },
    error: function(xhr, status, ex) {
      //console.log("--> error: " + status + " " + xhr.status + " " + xhr.statusText);

      ajaxOptions.reportStatus(xhr.status, xhr.statusText);
      if (fn) {
        try {
          fn(null); // null result indicates error
        } catch (ex2) {
          mvc.logError(ex2);
        }
      }
    },
    success: function(result, status) {
      var end = (new Date()).getTime();
      var out, authFn;
      //console.log("--> success: " + (result.sessionExpired ? "expired" : status) + " xhr status: " + savedXhr.status + " time: " + (end - start) + "ms.");

      // do generic status and message processing
      if (status == "success") {
        if (result.sessionExpired) {
          if (ajaxOptions.sessionExpired) {
            ajaxOptions.sessionExpired();
          }
          result = null; // null indicates an error
        } else {

          if (savedXhr.status < 200 || savedXhr.status > 299 || result.messages.length > 0) {
            ajaxOptions.reportStatus(savedXhr.status, savedXhr.statusText, result.messages);
          }

          // CSRF processing
          if (method == "GET" || getOverPost) {
            CSRFtoken = savedXhr.getResponseHeader(CSRF_HEADER);
          }
        }
      }

      if (fn) {
        try {
          fn(result);
        } catch (ex) {
          mvc.logError(ex);
        }
      }
    },
    complete: function(xhr, status) {
      var elapsed = (new Date()).getTime() - start;

      function done() {
        //console.log("--> processing stop");
        if (ajaxOptions.processing) {
          ajaxOptions.processing(false);
        }
      }

      activeXHRCount -= 1;
      //console.log("--> complete: " + activeXHRCount + ", " + (asyncTimer === null) + ", " + status + " time: " + elapsed + "ms.");
      if (activeXHRCount === 0) {
        if (asyncTimer === null) {
          // the spinner is spinning so don't flash it
          if (elapsed < busyLinger) {
            setTimeout(function() {
              done();
            }, busyLinger - elapsed);
          } else {
            done();
          }
        } else {
          // the request(s) went quick no need for spinner
          clearTimeout(asyncTimer);
          asyncTimer = null;
        }
      }
      //console.log("--> complete: " + activeXHRCount + ", " + status + " time: " + elapsed + "ms.");
    }
  };
  if (method == "POST" || method == "PUT") {
    options.contentType = "application/json";
  }
  activeXHRCount += 1;
  $.ajax(options);

};

//
// HTML Builder
//

var slowStringConcat = ($.browser.msie && $.browser.version < 8);

var htmlEscape = function(map, re, s) {
  var escaped = null;
  s = "" + s; // make sure s is a string
  escaped = s.replace(re, function(ch) {
    return map[ch];
  });
  return escaped;
};

var attrMap = { "&": "&amp;", "<": "&lt;", ">": "&gt;", "'": "&apos;", '"': "&quot;" };
var attrRE = /&|<|>|'|"/g; 
mvc.escapeHTMLAttr = function(s) {
  return htmlEscape(attrMap, attrRE , s);
};

var contentMap = { "&": "&amp;", "<": "&lt;", ">": "&gt;" };
var contentRE = /&|<|>/g;
mvc.escapeHTMLContent = function(s) {
  return htmlEscape(contentMap, contentRE, s);
};
var htmlBuilderPrototype = {
  markup: function(t) {
    this.html += t;
    return this;
  },
  attr: function(name, value) {
    if (arguments.length === 1) { // name is optional
      value = name;
      name = null;
    }
    if (name) {
      this.html += " ";
      this.html += name;
      this.html += "='";
    }
    this.html += mvc.escapeHTMLAttr(value);
    if (name) {
      this.html += "'";
    }
    return this;
  },
  optionalAttr: function(name, value) {
    if (value && typeof value !== "object") {
      this.html += " ";
      this.html += name;
      this.html += "='";
      this.html += mvc.escapeHTMLAttr(value);
      this.html += "'";
    }
    return this;
  },
  optionalBoolAttr: function(name, value) {
    // must be boolean and must be true - not just truthy
    if (value === true) {
      this.html += " ";
      this.html += name;
    }
    return this;
  },
  content: function(t) {
    this.html += mvc.escapeHTMLContent(t);
    return this;
  },
  clear: function() {
    this.html = "";
  },
  toString: function() {
    return this.html;
  }
};
// older versions of IE (< 8) had very slow string concatination
// so using push and array.join is much faster
var htmlBuilderPrototypeUsingJoin = {
  markup: function(t) {
    this.html.push(t);
    return this;
  },
  attr: function(name, value) {
    if (arguments.length === 1) { // name is optional
      value = name;
      name = null;
    }
    if (name) {
      this.html.push(" ");
      this.html.push(name);
      this.html.push("='");
    }
    this.html.push(mvc.escapeHTMLAttr(value));
    if (name) {
      this.html.push("'");
    }
    return this;
  },
  optionalAttr: function(name, value) {
    if (value && typeof value !== "object") {
      this.html.push(" ");
      this.html.push(name);
      this.html.push("='");
      this.html.push(mvc.escapeHTMLAttr(value));
      this.html.push("'");
    }
    return this;
  },
  optionalBoolAttr: function(name, value) {
    // must be boolean and must be true - not just truthy
    if (value === true) {
      this.html.push(" ");
      this.html.push(name);
    }
    return this;
  },
  content: function(t) {
    this.html.push(mvc.escapeHTMLContent(t));
    return this;
  },
  clear: function() {
    this.html = [];
  },
  toString: function() {
    return this.html.join("");
  }
};
mvc.htmlBuilder = function() {
  var that = Object.create(slowStringConcat ? htmlBuilderPrototypeUsingJoin : htmlBuilderPrototype);
  that.clear();
  return that;
};

//
// View
//
var accessibilityMode = false;

mvc.setAccessibilityMode = function(mode) {
  accessibilityMode = !!mode;
};

mvc.getAccessibilityMode = function() {
  return accessibilityMode;
};

// Let screen readers know that the DOM has changed (must also set focus)
// JAWS at least will update the virtual buffer when innerHTML is assigned to 
mvc.domChanged = function() {
  var $vbUpdate = $("#vbupdate");
  if ($vbUpdate.length === 0) {
    $("body").append("<div id='vbupdate' style='display:none;'>x<div>");
    $vbUpdate = $("#vbupdate");
  }
  $vbUpdate.innerHTML = "x";
};

// helper for setting focus
mvc.setFocus = function(e, tabIndex) {
  tabIndex = tabIndex || 0;
  e.tabIndex = tabIndex; // IE needs this before focus
  // this reportedly helps some screen readers
  setTimeout(function() { e.focus(); }, 0);
};

//
// bind delegating focus and blur handlers
//
mvc.bindFocus = function(elem, fn) {
  function wraphandler(e) {
    var event = $.event.fix(e);
    fn(event);
  }
  if (elem.addEventListener) {
    elem.addEventListener("focus", wraphandler, true);
  } else if (elem.attachEvent) {
    elem.attachEvent("onfocusin", wraphandler);
  }
};
mvc.bindBlur = function(elem, fn) {
  function wraphandler(e) {
    var event = $.event.fix(e);
    fn(event);
  }
  if (elem.addEventListener) {
    elem.addEventListener("blur", wraphandler, true);
  } else if (elem.attachEvent) {
    elem.attachEvent("onfocusout", wraphandler);
  }
};

//
// I18n
//
var bundles = {};

var localeSettings = {
  locale: "en",
  numberFormat: {
    decimalSeparator: ".",
    groupingSeparator: ",", // null if no grouping desired
    groupingSize: 3, // null if no grouping desired
    minusSign: "-"
  },
  dateTimeFormat: {
    amPmStrings: [ 'AM', 'PM' ],
    shortMonths: [ 'Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec' ],
    dateTimePattern: 'M/d/yy h:mm:ss a',
    datePattern: 'M/d/yy',
    timePattern: 'h:mm:ss a'
  }
};

mvc.setLocaleSettings = function(settings) {
  localeSettings = $.extend(true, localeSettings, settings);
};

mvc.addMessageBundle = function(name, bundle) {
  bundles[name] = bundle;
};

function getMessage(bundleName, key) {
  var bundle = bundles[bundleName];
  var result;
  if (bundle) {
    result = bundle[key];
    if (result) {
      return result;
    }
  }
  return null;
}

mvc.getMessage = function(bundleName, key) {
  var msg;
  if ($.isArray(bundleName)) {
    $.each(bundleName, function(i, bn) {
      msg = getMessage(bn, key);
      if (msg) {
        return false;
      }
    });
  } else {
    msg = getMessage(bundleName, key);
  }
  return msg === null ? key : msg;
};

mvc.format = function(bundleName, patternKey) {
  var pattern = mvc.getMessage(bundleName, patternKey);
  var args = [pattern].concat(Array.prototype.slice.call(arguments, 2));
  return mvc.formatMessage.apply(this, args);
};

mvc.formatMessage = function(pattern) {
  var re = /\{([0-9]*)\}/g;
  var args;
  if ($.isArray(arguments[1])) {
    args = arguments[1];
  } else {
    args = Array.prototype.slice.call(arguments, 1);
  }
  var count = 0;
  var result = pattern.replace(re, function(m, p1) {
    var n = parseInt(p1, 10);
    count++;
    if (n >= args.length) {
      throw "format(" + pattern + "): too few arguments";
    }
    return mvc.escapeHTMLContent(args[n]);
  });
  if (count < args.length) {
    mvc.logError("Warning: format(" + pattern + "): too many arguments. Expecting " + count + ", got " + args.length);
  }
  return result;
};

mvc.formatNumber = function(n, decPlaces) {
  var result;
  var nf = localeSettings.numberFormat;
  var str = (decPlaces === null || decPlaces === undefined) ? n.toString(10) : n.toFixed(decPlaces); 
  if (str.indexOf("e") >= 0) {
    return str;
  }
  if (str.charAt(0) == "-") {
    str = str.substring(1);
  }
  var i = str.indexOf(".");
  if (i == -1) {
    i = str.length;
    result = "";
  } else {
    result = nf.decimalSeparator + str.substring(i + 1);
  }
  if (nf.groupingSeparator && nf.groupingSize) {
    do {
      i -= nf.groupingSize;
      if (i > 0) {
        result = nf.groupingSeparator + str.substr(i, nf.groupingSize) + result;
      } else if (i <= 0) {
        result = str.substr(0, nf.groupingSize + i) + result;
      }
    } while (i > 0);
  } else {
    result = str.substring(0, i) + result;
  }
  if (n < 0) {
    result = nf.minusSign + result;
  }
  return result;
};

mvc.formatInteger = function(n) {
  return mvc.formatNumber(n, 0);
};

function zeroPad(n, pn) {
  n = "" + n;
  pn = pn - n.length;
  if (pn < 0) {
    pn = 0;
  }
  return "00000000".substr(0, pn) + n; 
}

mvc.getDateTimeDisplayFormat = function(pattern, localePattern) {
  var df = localeSettings.dateTimeFormat;
  pattern = pattern || df.dateTimePattern;
  localePattern = localePattern || df.localizedDateTimePattern;
  var quoted = false;
  var str = "";
  var i, j, c, lc;

  for (i = 0; i < pattern.length; i++) {
    c = pattern.charAt(i);
    lc = localePattern.charAt(i);
    if (!quoted && /[a-z,A-Z]/.test(c)) {
      j = i + 1;
      while (j < pattern.length && pattern.charAt(j) == c) {
        j += 1;
      }
      i = j - 1;

      switch (c) {
      case "'":
        if (i + 1 < pattern.length && pattern.charAt(i + 1) == "'") {
          str += "'";
        } else {
          quoted = true;
        }
        break;
      case 'y':
      case 'M':
      case 'd':
      case 'h':
      case 'H':
      case 'm':
      case 's':
        str += lc.toLowerCase();
        break;
      case 'a':
        str += df.amPmStrings[0].toLowerCase() + "/" + df.amPmStrings[1].toLowerCase();
        break;
      case 'K':
      case 'k':
        if (lc.toLowerCase() == "k") {
          str += "h";
        } else {
          lc.toLowerCase();
        }
        break;
      }
    } else {
      if (c == "'") {
        quoted = false;
      } else {
        str += c;
      }
    }
  }
  return str;
};

// Java style date pattern formatting. Not supported: G, w, W, D, F, E, S, z, Z, K, k
// options: {pattern: <pattern-string>, fullYear: <bool>, excludeAmPm: <bool>, forceHours: <h|H>}
mvc.formatDateTime = function(d, options) {
  options = options || {};
  var df = localeSettings.dateTimeFormat;
  var pattern = options.pattern || df.dateTimePattern;
  var quoted = false;
  var str = "";
  var year = "" + d.getUTCFullYear();
  var hours = d.getUTCHours();
  var ampm = hours < 12 ? 0 : 1;
  var i, j, c, plen;

  for (i = 0; i < pattern.length; i++) {
    c = pattern.charAt(i);
    if (!quoted && /[a-z,A-Z]/.test(c)) {
      plen = 1;
      j = i + 1;
      while (j < pattern.length && pattern.charAt(j) == c) {
        j += 1;
        plen += 1;
      }
      i = j - 1;

      if ("hH".indexOf(c) >= 0 && options.forceHours && "hH".indexOf(options.forceHours) >= 0) {
        c = options.forceHours;
      }
      switch (c) {
      case "'":
        if (i + 1 < pattern.length && pattern.charAt(i + 1) == "'") {
          str += "'";
        } else {
          quoted = true;
        }
        break;
      case 'y':
        str += (options.fullYear || plen > 2) ? year : year.substring(2);
        break;
      case 'M':
        if (plen >= 3) {
          str += df.shortMonths[d.getUTCMonth()];
        } else {
          str += zeroPad(d.getUTCMonth() + 1, plen);
        }
        break;
      case 'd':
        str += zeroPad(d.getUTCDate(), plen);
        break;
      case 'a':
        if (!options.excludeAmPm) {
          str += df.amPmStrings[ampm];
        }
        break;
      case 'h':
        if (hours > 12) {
          hours = hours - 12;
        } else if (hours === 0) {
          hours = 12;
        }
        str += zeroPad(hours, plen);
        break;
      case 'H':
        str += zeroPad(hours, plen);
        break;
      case 'm':
        str += zeroPad(d.getUTCMinutes(), plen);
        break;
      case 's':
        str += zeroPad(d.getUTCSeconds(), plen);
        break;
      }
    } else {
      if (c == "'") {
        quoted = false;
      } else {
        str += c;
      }
    }
  }
  return str;
};

mvc.formatDate = function(d, options) {
  options = options || {};
  if (!options.pattern) {
    options.pattern = localeSettings.dateTimeFormat.datePattern;
  }
  return mvc.formatDateTime(d, options);
};

mvc.formatTime = function(d, options) {
  options = options || {};
  if (!options.pattern) {
    options.pattern = localeSettings.dateTimeFormat.timePattern;
  }
  return mvc.formatDateTime(d, options);
};

// get array of important parts split on separator chars. Include am/pm strings because they may contain separators
// when strict want each special character
function splitDate(dstr, strict, ampm) {
  var a = [];
  var i = 0;
  var lo = 0;
  var splitRE = new RegExp("(" + ampm[0] + ")|(" + ampm[1] + ")|[-\\:\\./ \\,]+", "ig");

  dstr.replace(splitRE, function(m, am, pm, o, s) {
    if (o > lo) {
      a[i] = s.substring(lo, o);
      i++;
    }
    if (am || pm || strict) {
      a[i] = m;
      i++;
    }
    lo = o + m.length;
  });
  if (lo < dstr.length) {
    a[i] = dstr.substring(lo);
  }
  return a;
}

// Java style date pattern parsing. Not supported: G, w, W, D, F, E, S, z, Z, K, k
// options: {pattern: <pattern-string>, strict: <bool>}
mvc.parseDateTime = function(str, options) {
  options = options || {};
  var df = localeSettings.dateTimeFormat;
  var pattern = options.pattern || df.dateTimePattern;
  var ampm = df.amPmStrings;
  var parts = splitDate(str, options.strict, ampm);
  var i, j, pi, c, part, yearBase;
  var months = df.shortMonths;
  var year, month = null, day, hour = 0, minute = 0, second = 0, qs;
  var hourOffset = null;
  var quoted = false;

  pi = 0;
  for (i = 0; i < pattern.length; i++) {
    c = pattern.charAt(i);

    if (pi >= parts.length) {
      break;
    }
    part = parts[pi];

    if (!quoted && /[a-z,A-Z]/.test(c)) {
      j = i + 1;
      while (j < pattern.length && pattern.charAt(j) == c) {
        j += 1;
      }
      i = j - 1;

      switch (c) {
      case "'":
        if (i + 1 < pattern.length && pattern.charAt(i + 1) == "'") {
          qs += "'";
        } else {
          qs = "";
          quoted = true;
        }
        break;
      case 'y':
        year = parseInt(part, 10);
        if (part.length == 2) {
          yearBase = Math.floor((new Date()).getUTCFullYear() / 100) * 100;
          if (year > 50) {
            yearBase -= 100;
          }
          year = yearBase + year;
        }
        if (isNaN(year) || year < 1970) {
          return null;
        }
        break;
      case 'M':
        part = part.toLowerCase();
        for (j = 0; j < months.length; j++) {
          if (months[j].toLowerCase() == part) {
            month = j;
          }
        }
        if (month === null) {
          month = parseInt(part, 10) - 1;
        }
        if (isNaN(month) || month < 0 || month > 11) {
          return null;
        }
        break;
      case 'd':
        day = parseInt(part, 10);
        if (isNaN(day) || day < 0 || day > 31) {
          return null;
        }
        break;
      case 'a':
        part = part.toLowerCase();
        if (ampm[1].toLowerCase() == part) { // pm
          hourOffset = 12;
        } else if (ampm[0].toLowerCase() == part) { // am
          hourOffset = 0;
        } else if (!options.strict && /\d*/.test(part)) {
          pi -= 1; // ignore and put back
        } else if (options.strict) {
          return null;
        }
        break;
      case 'h': // 1-12
      case 'H': // 0-23
        hour = parseInt(part, 10);
        if (isNaN(hour)) {
          return null;
        }
        break;
      case 'm':
        minute = parseInt(part, 10);
        if (isNaN(minute)) {
          if (!options.strict) {
            minute = 0;
            pi -= 1; // ignore and put back
          } else {
            return null;
          }
        }
        break;
      case 's':
        second = parseInt(part, 10);
        if (isNaN(second)) {
          if (!options.strict) {
            second = 0;
            pi -= 1; // ignore and put back
          } else {
            return null;
          }
        }
        break;
      }
      if (c != "'") {
        pi += 1;
      }
    } else {
      if (quoted) {
        if (c == "'") {
          quoted = false;
          if (options.strict && part != qs) {
            return null;
          }
          pi += 1;
        } else {
          qs += c;
        }
      } else if (options.strict) {
        if (part != pattern.substring(i, i + part.length)) {
          return null;
        }
        i += part.length - 1;
        pi += 1;
      }
    }
  }
  if (options.strict && pi != parts.length) {
    return null;
  }
  if (hourOffset !== null) {
    hour = hour + hourOffset;
    if (hour == 24) {
      hour = 12;
    } else if (hour == 12) {
      hour = 0;
    }
  }
  var dt = new Date(Date.UTC(year, month, day, hour, minute, second));
  if (isNaN(dt)) {
    return null;
  }
  return dt;
};


//
// URL Template processing
//
// See: http://bitworking.org/projects/URI-Templates/spec/draft-gregorio-uritemplate-03.txt
// The above spec says that all values to be substituted are encoded and is very specific about 
// how. "every octet of the UTF-8 string that falls outside of ( unreserved ) MUST be percent-encoded"
// The JavaScript function encodeURIComponent should at least be in the spirit of the encoding
// required. The Mozilla documentation says that it encodes any chars outside:  
//   alphabetic, decimal digits, - _ . ! ~ * ' ( )
// This differs from unreserved by including ! * ' ()
//
var expRE = /\{[^\}]+\}/g;

function parseTemplate(uriTemplate) {
  var result = null;
  var exps = uriTemplate.match(expRE); // assert {} come in pairs and are not empty?
  var i, exp;

  function parse_exp(exp) {
    var a, i, vars, ve, v, def, s;
    var result = {};
    if (exp.indexOf("|") !== -1) {
      a = exp.split("|");
      // assert a.length = 3
      // assert a[0].substring(0,1) == "-" and the rest is only ALPHA
      result.op = a[0].substring(1);
      result.arg = a[1];
      // assert arg only contains (reserved / unreserved / pct-encoded)
      vars = a[2].split(",");
    } else {
      vars = [ exp ];
    }
    result.vars = [];
    for (i = 0; i < vars.length; i++) {
      ve = vars[i];
      s = ve.indexOf("=");
      if (s !== -1) {
        v = ve.substring(0,s);
        def = ve.substring(s + 1);
      } else {
        v = ve;
        def = null;
      }
      // assert v only contains (ALPHA / DIGIT)*(ALPHA / DIGIT / "." / "_" / "-" )
      // assert def only contains (unreserved / pct-encoded)
      result.vars.push({name: v,def: def});
    }
    return result;
  }

  if (exps) {
    result = [];
    for (i = 0; i < exps.length; i++) {
      exp = exps[i];
      exp = exp.substring(1, exp.length - 1);
      result.push(parse_exp(exp)); 
    }
  }
  return result;
}

var uriTemplatePrototype = {
  expand: function(args) {
    if (!this.exps) {
      return this.template;
    }
    // else
    var i = 0;
    var invalid = false;
    var exps = this.exps;
    var uri = this.template.replace(expRE, function(m) {
      var exp = exps[i];
      var v, vi, j, result, argVal;
      var op = exp.op;
      if (!op) {
        v = exp.vars[0];
        // assert there is exactly one item in vars
        result = encodeURIComponent(args[v.name]);
        if (!result) {
          result = v.def ? v.def : "";
        }
      } else if (op == "join") {
        result = "";
        for (vi = 0, j = 0; vi < exp.vars.length; vi++) {
          v = exp.vars[vi];
          argVal = args[v.name];
          if (argVal !== null && argVal !== undefined) {
            if (j > 0) {
              result += exp.arg;
            }
            j++;
            result += v.name + "=" + encodeURIComponent(args[v.name]);
          }
        }
      } else if (op == "prefix") {
        result = "";
        if (exp.vars.length != 1) {
          mvc.logError("Invalid number of variables for prefix operator.");
          invalid = true;
        } else {
          v = exp.vars[0];
          argVal = args[v.name];
          if (argVal !== null && argVal !== undefined) {
            if ($.isArray(argVal)) {
              $.each(argVal, function(j, vi) {
                result += exp.arg + vi;
              });
            } else {
              result = exp.arg + argVal;
            }
          }
        }
      } else {
        mvc.logError("Unknown uri template operation.");
        invalid = true;
      }
      i += 1;

      return result;
    });
    return invalid ? null : uri;
  }
};
mvc.uriTemplate = function(template) {
  var that = Object.create(uriTemplatePrototype);
  that.template = template;
  that.exps = parseTemplate(template);
  return that;
};

mvc.renderIcon = function(out, iconName) {
  out.markup("<span class='icon ").attr(iconName).markup("'></span>");
};

/*
  options:
  {
    id: ,
    styleClass: ,
    o.label: , 
    o.icon: ,
    o.value: ,
    o.nb: , // (no border)
    o.tooltip
  }
*/
mvc.renderButton = function(out, o) {
  var styleClass = o.styleClass;
  if (o.nb) {
    if (styleClass) {
      styleClass += "nb" + " " + styleClass;
    } else {
      styleClass = "nb";
    }
  }
  out.markup("<button type='button'").optionalAttr(o.value).optionalAttr("id", o.id).
    optionalAttr("class", styleClass).
    optionalAttr("title", o.tooltip).markup(">");
  if (o.icon) {
    mvc.renderIcon(out, o.icon);
  }
  if (o.label) {
    out.content(o.label);
  }
  out.markup("</button>");
};

mvc.layoutForm = function($e) {
  var maxWidth = 0;
  var pad, $p, w;
  var $labels = $e.find(".ctrlRow > label, .ctrlRow > .label, .valueRow > .label").not(".above");
  $labels.each(function() {
    if ($(this).prev().length === 0) {
      w = $(this).outerWidth();
      $p = $(this).parent();
      if ($p.hasClass(".ctrlRow") && $p.parent().hasClass(".ctrlRow")) {
        w += parseInt($p.css("marginLeft"), 10);
      }
      maxWidth = (w > maxWidth) ? w : maxWidth;
    }
  });
  $labels.each(function() {
    if ($(this).prev().length === 0) {
      pad = maxWidth - $(this).outerWidth();
      $p = $(this).parent();
      if ($p.hasClass(".ctrlRow") && $p.parent().hasClass(".ctrlRow")) {
        pad -= parseInt($p.css("marginLeft"), 10);
      }
      if (pad > 0) {
        pad += parseInt($(this).css("paddingRight"), 10);
        $(this).css("paddingRight", pad);
      }
    }
  });
  $e.find(".ctrl").each(function() {
    if ($(this).css("display") == "block") {
      var h = $(this).prevAll("label, .label").outerHeight();
      $(this).css({"position": "relative", "top": "-" + h + "px", "left": (maxWidth + parseInt($(this).css("marginLeft"), 10)) + "px"});
      $(this).parent().height($(this).outerHeight());
    }
  });
};


mvc.initIcons = function($context) {
  if (!highContrastMode) {
    return;
  }

  function forEachCSSRule(match, fn) {
    var sheets = document.styleSheets;
    var sheet, rules, rule, s, r, selector, baseURI;
    match = match.toLowerCase();
    for (s = 0; s < sheets.length; s++) {
      sheet = sheets[s];
      rules = sheet.cssRules ? sheet.cssRules : sheet.rules;
      for (r = 0; r < rules.length; r++) {
        rule = rules[r];
        selector = rule.selectorText.toLowerCase();
        if (selector.indexOf(match) >= 0) {
          baseURI = sheet.href;
          baseURI = baseURI.substring(0, baseURI.lastIndexOf("/") + 1);
          fn(rule, selector, baseURI);
        }
      }
    }
  }

  forEachCSSRule(".icon", function(r, selector, base) {
    var bgi = r.style.backgroundImage;
    if (bgi) {
      bgi = bgi.substring(4, bgi.length - 1);
      if (bgi.charAt(0) == "\"") {
        bgi = bgi.substring(1, bgi.length - 1);
      }
      if (bgi.indexOf("//") == -1) {
        bgi = base + bgi;
      }
      $(r.selectorText, $context).each(function() {
        // wrapper div is needed for FF2
        var img = "<div><img alt='' src='"+bgi+"' style='position:relative;'></div>";
        $(this).css({"background-image":"none","position":"relative"}).get(0).innerHTML = img;
      });
    }
  });
};

// A BLAF style note window
// There is just one note window possible at a time.
// Caller must make sure contents is clean
// x, y optional
var NOTE_FADE_TIME = 300;
var $currentNote = null;

mvc.getCurrentNote = function() {
  return $currentNote;
};

mvc.showNote = function(contents, $for, x, y) {
  var wr = $(window).width() + $(document).scrollLeft();
  var wt = $(document).scrollTop();
  var px, py, forw, forh, notew, noteh, noteloc, offset;
  var noteWasShowing;

  noteWasShowing = $currentNote !== null;
  
  if ($currentNote) {
    $currentNote.remove(); // remove previous note window if any
  }

  $currentNote = $("<div id='noteWindow' class='noteWindow'><div class='noteWrapper'><div class='dstr'></div><div class='dsbl'><div class='ds'><div class='noteContent'>" +
    contents + "</div><div class='callout'><img src='images/noteCone.png' alt=''></div></div></div></div></div>").css({
      position: 'absolute',
      top: -99999,
      left: 0
  }).appendTo("body");
  mvc.layoutForm($currentNote);
  if (x && y) {
    forw = 0;
    forh = 0;
  } else {
    offset = $for.offset();
    x = offset.left;
    y = offset.top;
    forw = $for.width();
    forh = $for.height();
  }
  notew = $currentNote.width() + 12;
  noteh = $currentNote.find(".noteWrapper").height();

  // default to top right
  noteloc = "tr";
  px = x + forw;
  py = y - noteh;

  // make sure it is on screen
  if (py - 10 < wt) {
    noteloc = "br";
    px = x + forw;
    py = y + forh;
  }
  if (px + 10 + notew > wr) {
    noteloc = "tl";
    px = x - notew;
    py = y - noteh;
  }
  if (py - 10 < wt) {
    noteloc = "bl";
    px = x - notew;
    py = y + forh;
  }
  $currentNote.addClass(noteloc).css({
    display: "none",
    width: notew,
    height: noteh,
    top: py,
    left: px
  }).fadeIn(noteWasShowing ? 0 : NOTE_FADE_TIME);
  if (noteloc == "bl" || noteloc == "br") {
    $currentNote.find(".callout").css({top: -noteh });
  }
  mvc.domChanged();
  return $currentNote;
};

mvc.hideNote = function() {
  if ($currentNote !== null && !$currentNote.is(":animated")) {
    $currentNote.fadeOut(NOTE_FADE_TIME, function() {
      $(this).remove();
      if ($currentNote && $(this).get(0) === $currentNote.get(0)) {
        $currentNote = null;
      }
    });
  }
};

mvc.formatNoteContent = function(bundle, c) {
  var i, item;
  var out = mvc.htmlBuilder();
  for (i = 0; i < c.length; i++) {
    item = c[i];
    if (item.separator) {
      out.markup("<div class='separator'></div>");
    } else if (item.value !== null) {
      out.markup("<div class='valueRow'><span class='label'>").
        content(mvc.getMessage(bundle, item.label)).
        markup(": </span>");
      if (item.value.length > 60) {
        out.markup("<div>");
      }
      if (item.markup) {
        out.markup(item.value);
      } else {
        out.content(item.value);
      }
      if (item.value.length > 60) {
        out.markup("</div>");
      }
      out.markup("</div>");
    } else {
      out.markup("<p>").
        content(mvc.getMessage(bundle, item.label)).
        markup("</p>");
    }
  }
  return out.toString();
};

/*
  Splitter
  TODO: support edge
  probably should be a widget to support programmatic position update
*/
mvc.splitter = function(o) {
  var out = mvc.htmlBuilder();
  var horiz = o.orientation == "horizontal";
  var $after = $(o.after);
  var left, top, cursor;
  var pos = horiz ? "left" : "top";
  var keyInc, keyDec;
  var timerId = null;

  function setPos($s, p, closed, delay) {
    if (p < 60) {
      p = 0;
      closed = true;
    }
    if (closed) {
      $s.addClass("closed");
      $s.find("button").attr("title", o.restoreText);
    } else {
      $s.removeClass("closed");
      $s.find("button").attr("title", o.collapseText);
    }
    $s.css(pos, p);
    if (delay) {
      if (timerId) {
        clearTimeout(timerId);
        timerId = null;
      }
      timerId = setTimeout(function() {
        timerId = null;
        o.change(p, closed);
      }, 300);
    } else {
      o.change(p, closed);
    }
  }

  function getPos() {
    if (horiz) {
      return $after.position().left + $after.outerWidth();
    } // else
    return $after.position().top + $after.outerHeight();
  }

  function getMaxPos() {
    var $c = $after.parent();
    var $s = $after.next().eq(0);
    if (horiz) {
      return $c.width() - $s.width() - 60;
    } // else
    return $c.height() - $s.height() - 60;
  }

  if (horiz) {
    left = getPos();
    top = 0;
    cursor = "e-resize";
    keyInc = $.ui.keyCode.RIGHT;
    keyDec = $.ui.keyCode.LEFT;
  } else {
    left = 0;
    top = getPos();
    cursor = "s-resize";
    keyInc = $.ui.keyCode.DOWN;
    keyDec = $.ui.keyCode.UP;
  }

  out.markup("<div class='").attr((horiz ? "splitterH" : "splitterV") + (o.closed ? " closed" : ""))
    .markup("'><div></div><button type='button' title='")
    .attr(o.closed ? o.restoreText : o.collapseText).markup("'></button></div>");

  $(out.toString()).insertAfter(o.after).css({
    position: "absolute",
    left: left,
    top: top
  }).draggable({
    axis: horiz ? "x" : "y",
    containment: "parent",
    cancel: "button",
    cursor: cursor,
    scroll: false,
    stop: function(e, ui) {
      var p = ui.position[pos];
      var max = getMaxPos();
      if (p >= max) {
        p = max;
      }
      setPos($(this), p, false);
    }
  }).hover(function() {
    $(this).addClass("ui-state-hover");
  }, function() {
    $(this).removeClass("ui-state-hover");
  }).find("button").click(function() {
    var $s = $(this).parent();
    var closed = !$s.hasClass("closed");
    var p = closed ? 0 : getPos();
    setPos($s, p, closed);
  }).focus(function(e) {
    $(this).parent().addClass("ui-state-focus");
  }).blur(function(e) {
    $(this).parent().removeClass("ui-state-focus");
  }).keydown(function(e) {
    var kc = e.keyCode;
    var p = null;
    var $s, max;

    if (kc == keyDec) {
      $s = $(this).parent();
      p = $s.position()[pos];
      p -= 10;
    } else if (kc == keyInc) {
      $s = $(this).parent();
      p = $s.position()[pos];
      if (p === 0) {
        p = getPos();
      } else {
        p += 10;
      }
      max = getMaxPos();
      if (p > max) {
        p = max;
      }
    }
    if (p !== null) {
      setPos($s, p, false, true);
      return false;
    }
  });
};

/*
  formHelper - moves data in and out of form fields

mvc.formHelper(fields, options)

Fields is an array of field items as follows: 
{
  id: <str>, // name of form control ID
  name: <str>, // only used for radio controls
  prop: <str>, // name of data property
  formatter: <{fn: <fn(value)>, args: []}>
  constraints: [{fn: <fn(value)>, args: []},...], 
                                   // an array of constraints functions and optional constraint arguments
                                   // fn can be a function or the name of a registered constraint
                                   // if there are no args the object can simply be replaced with the function
  type: number | string | boolean  // controls basic conversion when getting and setting the field. Default is string
}

During constraint checking the field items have the current value of the field available in the value property.
This is most useful for global constraint checking.

Options is an object with these properties:
{
  formContext: <$context>, // required, jQuery context that contains the form fields
  globalConstraints: <fn(fields)>, // optional global constraint checking function. this is options
                                   // only called if all field constraints pass
  fieldErrorHandler: <fn($field, reason)>, // optional error handler. this is field.
                                           // Default sets error class on $field and sets up
                                           // a note with the reason and event handlers to revalidate
  formErrorHandler: <fn(errors, reason)>   // optional form error handler. this is options
                                           // Default assumes a div with class "messages" that will 
                                           // be shown and have messages added. errors is a boolean if true there were errors.
  layout: <fn($dialog)> // optional custom dialog layout function
}

During global constraint checking the fields array has a find method added that will find a field item by prop

Field and global constraint functions return null for success (the constraint is met) and a reason string if the constraint
is not met explaining what is wrong.

methods:

  load(obj, prefix)
    loads the form fields with values from obj as specified by the form helper fields and options

  store(obj, prefix)
    stores values from the form fields into obj as specified by the form helper fields and options
  
  clear(prefix)
    clears any error state 

  layout($dialog)
    layout the dialog
    
  For all methods prefix is an optional string prefix to be used on all form field ids (or names).

Static method:

mvc.formHelper.addValidator(name, fn)

  TODO:
    default id to prop?
    accessibility
    custom read conversion
    custom write conversions
*/

var validators = {};
var formatters = {};

// This public validate function can be used outside a formHelper
// to do validation using the same constraint functions available to a formHelper.
// Put the value to validate in field.value.
// $field and handler are optional
mvc.validate = function(field, $field, handler) {
  var reason = null;
  var constraints = field.constraints;
  var i, constraint, fn, args;

  if (!constraints || ($field && $field.get(0).disabled)) {
    return true;
  }
  for (i = 0; i < constraints.length; i++) {
    constraint = constraints[i];
    args = [];
    if (typeof constraint === "string" || $.isFunction(constraint)) {
      fn = constraint;
    } else {
      fn = constraint.fn;
      if (constraint.args) {
        args = args.concat(constraint.args);
      }
    }
    fn = $.isFunction(fn) ? fn : validators[fn];
    args.unshift(field.value);
    reason = fn.apply(field, args);
    if (handler) {
      handler.call(field, $field, reason);
    }
    if (reason !== null) {
      break;
    }
  }
  return reason === null;
};

// this is the field
function defaultFieldErrorHandler($field, reason) {
  var message, value;
  var handler = arguments.callee;
  var self = this;
  $field.removeClass("error").unbind(".fieldError");
  if (reason) {
    message = reason;
    $field.addClass("error").bind("focus.fieldError", function() {
      mvc.showNote(message, $field, true);
    }).bind("blur.fieldError", function() {
      // revalidate
      value = $.trim($field.val());
      self.value = value;
      mvc.hideNote(true);
      setTimeout(function() { mvc.validate(self, $field, handler); }, 10);
    }).bind("mouseenter.fieldError", function() {
      mvc.showNote(message, $field, false);
    }).bind("mouseleave.fieldError", function() {
      mvc.hideNote(false);
    });
  }
}
// this is the formHelper options
function defaultFormErrorHandler(errors, reason) {
  var out = mvc.htmlBuilder();
  var self = this;
  var $firstError;
  if (errors) {
    out.markup("<h3>").content(mvc.getMessage(BUNDLE, "validation.dialog.messageTitle")).markup("</h3>");
    if (reason) {
      out.markup("<p>").content(reason).markup("</p>");
    }
    this.formContext.find(".messages").html(out.toString()).slideDown("slow", function() {
      $firstError = self.formContext.find(".error").eq(0);
      if ($firstError.length > 0) {
        mvc.setFocus($firstError.get(0)); 
      }
    });
  } else {
    this.formContext.find(".messages").html("").hide();
  }
}

function findField(p) {
  var fields = this;
  var i, field;

  for (i = 0; i < fields.length; i++) {
    field = fields[i];
    if (field.prop == p) {
      return field;
    }
  }
  return null;
}

var formHelperPrototype = {
  getField: function(field, prefix) {
    var $field;
    if (field.name) {
      $field = $("input:radio[name=" + prefix + field.name + "]");
    } else {
      $field = $("#" + prefix + field.id);
    }
    return $field.length > 0 ? $field : null;
  },
  load: function(obj, prefix) {
    var fields = this.fields;
    var i, field, value, $field, dateOpt;
    prefix = prefix || "";

    function formatOnBlur(f) {
      var fmtr, args, fn;

      fmtr = f.formatter;
      args = [];
      if (typeof fmtr === "string" || $.isFunction(fmtr)) {
        fn = fmtr;
      } else {
        fn = fmtr.fn;
        if (fmtr.args) {
          args = args.concat(fmtr.args);
        }
      }
      fn = $.isFunction(fn) ? fn : formatters[fn];
      args.unshift("");
      $field.blur(function() {
        args[0] = $(this).val();
        $(this).val(fn.apply(f, args));
      });
    }

    for (i = 0; i < fields.length; i++) {
      field = fields[i];

      value = obj[field.prop];
      if (value === null || value === undefined) {
        value = "";
      } else {
        if (field.type == "date") {
          dateOpt = {fullYear: true};
          if (field.formatter && field.formatter.args && field.formatter.args.length > 0) {
            dateOpt.pattern = field.formatter.args[0];
          }
          value = mvc.formatDateTime(value, dateOpt);
        } else if (field.type != "boolean") {
          value = value.toString();
        }
      }

      $field = this.getField(field, prefix);
      if ($field === null) {
        continue;
      }
      if ($field.get(0).type == "checkbox") {
        $field.get(0).checked = value;
      } else if ($field.get(0).type == "radio") {
        $field = $field.filter("[value=" + value + "]");
        if ($field.length > 0) {
          $field.get(0).checked = true;
        }
      } else {
        $field.val(value);
      }
      // format on change should only be used on text fields
      if (field.formatter) {
        formatOnBlur(field);
      }
    }
  },
  store: function(obj, prefix) {
    var self = this;
    var o = this.options;
    var fields = this.fields;
    var errors = false;
    var i, field, $field, value, dateOpt, reason = null;

    // implicit input field, prefix, implicit output $field
    function getValue() {
      var value;
      $field = self.getField(field, prefix);
      if ($field === null) {
        return null;
      }
      if ($field.get(0).type == "checkbox") {
        value = $field.get(0).checked;
      } else if ($field.get(0).type == "radio") {
        value = $field.filter(":checked").val();
      } else {
        value = $.trim($field.val());
      }
      return value;
    }

    prefix = prefix || "";

    for (i = 0; i < fields.length; i++) {
      field = fields[i];
      value = getValue();
      if (value === null) {
        continue;
      }
      // validate
      field.value = value;
      if (!mvc.validate(field, $field, o.fieldErrorHandler)) {
        errors = true;
      }
    }
    if (!errors && o.globalConstraints) {
      fields.find = findField;
      reason = o.globalConstraints.call(o, fields);
      if (reason !== null) {
        errors = true;
      }
    }
    o.formErrorHandler.call(o, errors, reason);
    if (!errors) {
      for (i = 0; i < this.fields.length; i++) {
        field = this.fields[i];
        value = getValue();
        if (value === null) {
          continue;
        }

        // default conversions
        if (field.type == "number") {
          if (value !== "") {
            value = value * 1;
          } else {
            value = null;
          }
        } else if (field.type == "date") {
          if (value === "") {
            value = null;
          } else {
            dateOpt = {fullYear: true};
            if (field.formatter && field.formatter.args && field.formatter.args.length > 0) {
              dateOpt.pattern = field.formatter.args[0];
            }
            value = mvc.parseDateTime(value, dateOpt);
          }
        } else if (field.type == "boolean") {
          if (value === "") {
            value = null;
          }
        }
        obj[field.prop] = value;
      }
    }
    return !errors;
  },
  clear: function(prefix) {
    var o = this.options;
    var fields = this.fields;
    var i, field, $field;
    prefix = prefix || "";

    for (i = 0; i < fields.length; i++) {
      field = fields[i];
      $field = this.getField(field, prefix);
      if ($field === null) {
        continue;
      }
      // clear validation errors
      if (field.constraints) {
        o.fieldErrorHandler.call(field, $field, null);
      }
      if (field.formatter) {
        $field.unbind("blur");
      }
    }
    o.formErrorHandler.call(o, false);
    mvc.hideNote();
  },
  layout: function($dialog) {
    var o = this.options;
    mvc.layoutForm($dialog);
    if (o.layout) {
      o.layout($dialog);
    }
  }
};
mvc.formHelper = function(fields, options) {
  var that = Object.create(formHelperPrototype);
  that.fields = fields;
  that.options = $.extend({}, {
    formContext: null, // jquery context that contains the form fields
    layout: null, // fn()
    globalConstraints: null,
    fieldErrorHandler: defaultFieldErrorHandler,
    formErrorHandler: defaultFormErrorHandler
  }, options);
  return that;
};
mvc.formHelper.addValidator = function(name, fn) {
  if (!validators[name]) {
    validators[name] = fn;
  } else {
    mvc.logError("Attempt to overwrite formHelper validator ignored.");
  }
};
mvc.formHelper.addFormatter = function(name, fn) {
  if (!formatters[name]) {
    formatters[name] = fn;
  } else {
    mvc.logError("Attempt to overwrite formHelper formatter ignored.");
  }
};
mvc.formHelper.addValidator("required", function(value) {
  if (value === undefined || value === null || value === "") {
    return mvc.getMessage(BUNDLE, "validation.required");
  }
  return null;
});
mvc.formHelper.addValidator("number", function(value) {
  if (value === "") {
    return null;
  }
  if (parseFloat(value) !== value * 1) {
    return mvc.getMessage(BUNDLE, "validation.number");
  }
  return null;
});
mvc.formHelper.addValidator("integer", function(value) {
  if (value === "") {
    return null;
  }
  if (parseInt(value, 10) !== value * 1) {
    return mvc.getMessage(BUNDLE, "validation.integer");
  }
  return null;
});
mvc.formHelper.addValidator("range", function(value, min, max) {
  if (value === "") {
    return null;
  }
  value = value * 1;
  if (isNaN(value)) {
    return mvc.getMessage(BUNDLE, "validation.number");
  }
  if (min !== null && max !== null) {
    if (value > max || value < min) {
      return mvc.format(BUNDLE, "validation.range", min, max);
    }
  } else if (max !== null) {
    if (value > max) {
      return mvc.format(BUNDLE, "validation.rangeMax", max);
    }
  } else if (min !== null) {
    if (value < min) {
      return mvc.format(BUNDLE, "validation.rangeMin", min);
    }
  }
  return null;
});
mvc.formHelper.addValidator("member", function(value, list) {
  var i;
  for (i = 0; i < list.length; i++) {
    if (list[i] == value) {
      return null;
    }
  }
  return mvc.format(BUNDLE, "validation.notMember");
});
mvc.formHelper.addValidator("datetime", function(value, pattern) {
  var dt;
  var options = {strict: true};

  if (value === "") {
    return null;
  }
  if (pattern) {
    options.pattern = pattern;
  }
  dt = mvc.parseDateTime(value, options);

  if (dt === null) {
    return mvc.getMessage(BUNDLE, "validation.datetime");
  }
  return null;
});
mvc.formHelper.addValidator("deltatime", function(value) {
  if (value === "") {
    return null;
  }
  // Look for [d ][d]d:[d]d
  if (/^(\d+ )?\d?\d:\d?\d$/.test(value) ) {
    return null;
  }
  return mvc.getMessage(BUNDLE, "validation.deltatime");
});

// pattern is optional date/time pattern
mvc.formHelper.addFormatter("datetime", function(value, pattern) {
  var dt;
  var options = {fullYear: true};

  if (pattern) {
    options.pattern = pattern;
  }

  dt = mvc.parseDateTime(value, options);
  if (dt === null) {
    return value;
  }

  var newValue;
  newValue = mvc.formatDateTime(dt, options);
  return newValue;
});
// returns value in format: days hours:minutes:seconds.ms
// up to resolution which is one of: m, s, ms
mvc.formHelper.addFormatter("deltatime", function(value, resolution) {
  var tmp;
  var ONESECOND = 1000;       // ms
  var ONEMINUTE = 60 * ONESECOND;
  var ONEHOUR = 60 * ONEMINUTE;
  var ONEDAY = 24 * ONEHOUR;
  var deltatime = 0;
  var newValue = "";

  var match = /^(\d+ )?(\d+:)?(\d+:)?(\d+)(.\d+)?$/.exec(value);
  if (!match) {
    return value;
  }

  if (match[1]) {
    deltatime += match[1] * ONEDAY;
  }
  if (match[2] && match[3]) {
    deltatime += parseInt(match[2], 10) * ONEHOUR + parseInt(match[3], 10) * ONEMINUTE + parseInt(match[4], 10) * ONESECOND;
    if (match[5]) {
      deltatime += match[5].substring(1) * 1;
    }
  } else if (match[2]) {
    deltatime += parseInt(match[2], 10) * ONEHOUR + parseInt(match[4], 10) * ONEMINUTE;
    if (match[5]) {
      return value;
    }
  } else {
    deltatime += parseInt(match[4], 10) * ONESECOND;
    if (match[5]) {
      deltatime += match[5].substring(1) * 1;
    }
  }

  resolution = resolution || "m";

  tmp = Math.floor(deltatime / ONEDAY);
  newValue += tmp;

  deltatime -= tmp * ONEDAY;
  tmp = Math.floor(deltatime / ONEHOUR);
  newValue += " " + zeroPad(tmp, 2);

  deltatime -= tmp * ONEHOUR;
  tmp = Math.floor(deltatime / ONEMINUTE);
  newValue += ":" + zeroPad(tmp, 2);
  if (resolution == "m") {
    return newValue;
  }

  deltatime -= tmp * ONEMINUTE;
  tmp = Math.floor(deltatime/ ONESECOND);
  newValue += ":" + zeroPad(tmp, 2);
  if (resolution == "s") {
    return newValue;
  }

  deltatime -= tmp * ONESECOND;
  newValue += "." + deltatime;
  return newValue;
});

//
// doModalDialog
//
/* 
  Options
  {
    title: <string>,
    formHelper: <obj>,
    formData: <obj>,
    okLabel: <string>, // optional default OK
    noCancel: <bool>, // optional default false
    ok: <fn()>, // optional
    close: <fn(cancel)>, // optional
    focusElemAfter: <domElem> // optional
  }
*/
mvc.doModalDialog = function($dialog, options) {
  var minh, minw;
  var fh = options.formHelper;
  var dlgButtons = {};
  var defaultButtonPressed = false;
  var defaultButton = options.okLabel || mvc.getMessage(BUNDLE, "button.ok.label");
  var defaultButtonIndex = 0;
  dlgButtons[defaultButton] = function() {
    if (fh.store(options.formData)) {
      defaultButtonPressed = true;
      if (options.ok) {
        try {
          options.ok();
        } catch (ex) {
          mvc.logError(ex);
        }
      }
      $(this).dialog("close");
    }
  };
  if (!options.noCancel) {
    dlgButtons[mvc.getMessage(BUNDLE, "button.cancel.label")] = function() {
      $(this).dialog("close");
    };
  }

  fh.load(options.formData);

  $dialog.addClass("ui-dialog-content").css({position:"absolute", width:"auto", height:"auto", top:"0", left:"-99999px", display:"block"});
  fh.layout($dialog);
  minw = $dialog.outerWidth() + 8;
  minh = $dialog.outerHeight() + 58; // estimated size of header and button area
  $dialog.css({position:"static", top:"0", left:"0", display:"none"});

  $dialog.dialog({
    modal: true,
    draggable: true,
    closeText: mvc.getMessage(BUNDLE, "button.close.label"),
    minWidth: minw,
    width: minw,
    minHeight: minh,
    title: options.title,
    buttons: dlgButtons,
    dragStart: function(e, ui) {
      mvc.hideNote();
    },
    resizeStart: function(e, ui) {
      mvc.hideNote();
    },
    open: function(e, ui) {
      mvc.hideNote();
      $.ui.dialog.overlay.resize();
      mvc.initIcons($dialog.parent().parent());
    },
    close: function(e, ui) {
      fh.clear();
      $(this).dialog("destroy").unbind("keypress.domodal");
      if (options.close) {
        try {
          options.close(defaultButtonPressed === false);
        } catch (ex2) {
          mvc.logError(ex2);
        }
      }
      if (options.focusElemAfter) {
        // put focus back where it was
        mvc.setFocus(options.focusElemAfter, 0);
      }
    }
  }).bind("keypress.domodal", function (e) {
    var type;
    if (e.keyCode == $.ui.keyCode.ENTER && !e.altKey && !e.ctrlKey) {
      type = $(e.target).attr("type");
      if (type == "text" || type == "password" || type == "checkbox" || type == "radio") {
        setTimeout(function() {
          $dialog.parent().find(".ui-dialog-buttons button").get(defaultButtonIndex).focus();
          dlgButtons[defaultButton].call($dialog.get(0));
        }, 1);
        return false;
      }
    }
  });
  mvc.domChanged();
};

mvc.doMessageDialog = function(title, contents, fnOrfocusElemAfter) {
  var h, w;
  var dlgButtons = {};
  var defaultButton = mvc.getMessage(BUNDLE, "button.ok.label");
  dlgButtons[defaultButton] = function() {
    $(this).dialog("close");
  };

  var $dialog = $("<div></div>").appendTo("body").hide();
  $dialog.html(contents);
  $dialog.css({position:"absolute", top:"0", left:"-99999px", display:"block"});
  w = $dialog.outerWidth() + 8;
  h = $dialog.outerHeight() + 58; // estimated size of header and button area
  $dialog.css({position:"static", top:"0", left:"0", display:"none"});

  $dialog.dialog({
    modal: true,
    draggable: true,
    closeText: mvc.getMessage(BUNDLE, "button.close.label"),
    minWidth: w,
    width: w,
    minHeight: h,
    title: title,
    buttons: dlgButtons,
    open: function(e, ui) {
      mvc.hideNote();
      $.ui.dialog.overlay.resize();
      mvc.initIcons($dialog.parent().parent());
    },
    close: function(e, ui) {
      $(this).dialog("destroy");
      $(this).remove();
      if (fnOrfocusElemAfter) {
        if ($.isFunction(fnOrfocusElemAfter)) {
          fnOrfocusElemAfter();
        } else {
          // put focus back where it was
          mvc.setFocus(fnOrfocusElemAfter, 0);
        }
      }
    }
  });
  mvc.domChanged();
};

//
// Controller Initialization
//
var resizeListeners = [];
var resizeTimerId = null;

function resize() {
  if (resizeTimerId) {
    clearTimeout(resizeTimerId);
    resizeTimerId = null;
  }
  resizeTimerId = setTimeout(function() {
    var i;

    resizeTimerId = null;
    for (i = 0; i < resizeListeners.length; i++) {
      resizeListeners[i].fn();
    }
  }, 200);
}

mvc.bindResize = function(fn, order) {
  resizeListeners.push({fn: fn, order: order || 10});
  resizeListeners.sort(function(a, b) { return a.order - b.order; });
};

mvc.unbindResize = function(fn) {
  var i;
  for (i = 0; i < resizeListeners.length; i++) {
    if (resizeListeners[i].fn === fn) {
      resizeListeners.splice(i,1);
    }
  }
};

$(document).ready(function () {

  $(window).resize(resize);
  resize();

  $("body").append("<p id='referenceTextSize' style='padding:0;position:absolute;top:0;left:-99999px;background-color:#878787;'>T</p>");
  var testcolor = $("#referenceTextSize").css("background-color").toLowerCase();
  if (testcolor != "#878787" && testcolor != "rgb(135, 135, 135)") {
    highContrastMode = true;
  }

  var refText = $("#referenceTextSize").get(0);
  var refTextHeight = refText.offsetHeight;
  setInterval(function() {
    var h = refText.offsetHeight; // don't use jQuery .height() here for performance reasons
    if (h != refTextHeight) {
      resize();
      refTextHeight = h;
    }
  }, 500);

});

})(mvc);
