/*!
  UI widget for menus including menu bars, popup menus and menu items.
  Copyright (c) 2009, 2010, Oracle and/or its affiliates. All rights reserved.
*/
/*
  The widget options object is a menu with the following recursive structure:
  
  menu:
   {
     menubar: <bool> // only valid on root menu
     items: [<menuItem>...]
   }
  
  menuItem:
  // menu item is one of the following forms (discriminated by the type property):
   { type: "separator" }
   { type: "subMenu", label: <text>, icon: <name>, menu: <menu>, disabled: <bool-or-fn> }
   { type: "toggle", label: <text>, accelerator: <text>, set: <fn>, get: <fn>, onLabel: <text>, offLabel: <text>, disabled: <bool-or-fn> }
     Either label or onLabel and offLabel are used.
   { type: "action", label: <text>, accelerator: <text>, action: <fn>, icon: <name>, disabled: <bool-or-fn> }
   { type: "radioGroup", set: <fn>, get: <fn>, choices: [
     { label: <text-key>, value: <any>, disabled: <bool>/fn },...
   ] }
  
  Menu items can have an id property and can be found by id.
  Menus and menu items can be updated while the menus is not active.
  When action function is called this is the menu item and the root menu (widget options) is passed as an argument. The second argument is 
  the element that focus would normally return to when the menu closes (or null if none). The action should return true if it will take
  responsibility for the focus.
  
*/
/*global window,jQuery,mvc,menuManager,placeMenu */
(function($) {

var VK_LEFT = 37,
    VK_UP = 38,
    VK_RIGHT = 39,
    VK_DOWN = 40,
    VK_ENTER = 13,
    VK_SPACE = 32,
    VK_TAB = 9,
    VK_ESC = 27;

var currentMenu = null;
var currentMenuBar = null; //TODO
var menuLauncher = null;
var actionTookFocus = false;
var menuStack = [];
var keepTracking = false;
var lastMouseX = 0;
var lastMouseY = 0;
var lastTrackingMouseX = 0;
var lastTrackingMouseY = 0;
var subMenuDelayID = null;
var lastMenu = null;
var lastItem = null;

function menuResize() {
  var i, $menu, $item;

  if (menuLauncher) {
    placeMenu(menuStack[0], false, $(menuLauncher));
    for (i = 1; i < menuStack.length; i++) {
      $menu = menuStack[i];
      $item = $menu.parents(".menuItem").eq(0);
      placeMenu($menu, true, $item);
    }
  } else {
    // TODO handle menu at x,y
  }
}

function startTrackingMenus() {
  $(window).bind("blur.menuTracking", function(e) {
    // TODO  menuManager.closeAll();
  });
  $("html").bind("mousedown.menuTracking", function(e) {
    var $target = $(e.target);
    if ($target.closest(".menu").length === 0 && $target.get(0) !== menuLauncher) {
      menuManager.closeAll();
    }
  }).bind("mousemove.menuTracking", function(e) {
      var $target, menu;

      if (lastTrackingMouseX == e.pageX && lastTrackingMouseY == e.pageY) {
        return;
      }
      lastTrackingMouseX = e.pageX;
      lastTrackingMouseY = e.pageY;
      $target = $(e.target);

      menu = $target.parents(".menu").get(0);
      if (!menu && lastMenu != menu) {
        mvc.setFocus(lastMenu, 0);
      }
      lastMenu = menu;
  });
  mvc.bindResize(menuResize, 100);
}

function stopTrackingMenus() {
  $(window).unbind(".menuTracking");
  $("html").unbind(".menuTracking");
  mvc.unbindResize(menuResize);
}

function layoutMenu($menu) {
  var maxLabelWidth = 0;
  var maxAccelWidth = 0; // TODO support accelerator text
  var $labels = $menu.find("ul:first").children("li").children("a, label, span.hSeparator, .label");
  var subMenuWidth = $menu.find(".subMenuCol").outerWidth();

  $labels.each(function() {
    var w = $(this).outerWidth();
    maxLabelWidth = (w > maxLabelWidth) ? w : maxLabelWidth;
  });

  $labels.each(function() {
    var pad = maxLabelWidth - $(this).outerWidth();
    if (pad > 0) {
      pad += parseInt($(this).css("paddingRight"), 10);
      if (this.className == "hSeparator") {
        pad += subMenuWidth - 2;
      }
      $(this).css("paddingRight", pad);
    }
  });
}

function placeMenu($menu, subMenu, x, y) {
  var wr = $(window).width() + $(document).scrollLeft();
  var wb = $(window).height() + $(document).scrollTop();
  var px, py, elemw, elemh, menuw, menuh, offset;
  var $elem;
  var isVisible = $menu.css("display") != "none";

  if (!isVisible) {
    $menu.css({
        display: "block",
        position: "absolute",
        top: -99999,
        left: 0
    });
  }
  layoutMenu($menu);

  if (typeof x == "number" && typeof y == "number") {
    elemw = 0;
    elemh = 0;
  } else {
    $elem = x;
    offset = $elem.offset();
    x = offset.left;
    y = offset.top;
    elemw = $elem.outerWidth();
    elemh = $elem.outerHeight();
  }
  menuw = $menu.width();
  menuh = $menu.height();

  if (subMenu) {
    // default to right top aligned
    px = x + elemw;
    py = y;
  
    // make sure it is on screen
    if (px + 10 + menuw > wr) {
      px = x - menuw;
    }
    if (py + 10 + menuh > wb) {
      px = x + elemw;
      py = y + elemh - menuh;
    }
    if (px + 10 + menuw > wr) {
      px = x - menuw;
      py = y + elemh - menuh;
    }
  } else {
    // default to under start aligned
    px = x;
    py = y + elemh;
  
    // make sure it is on screen
    if (px + 10 + menuw > wr) {
      px = x + elemw - menuw; 
    }
    if (py + 10 + menuh > wb) {
      px = x;
      py = y - menuh;
    }
    if (px + 10 + menuw > wr) {
      px = x + elemw - menuw;
    }
  }
  offset = $menu.offsetParent().offset();
  $menu.css({
    display: (isVisible ? "block" : "none"),
    top: py - offset.top,
    left: px - offset.left
  });
  if (!isVisible) {
    $menu.slideDown(200, function() {
      mvc.setFocus($menu.get(0), 0);
    });
  }
}

var menuManager = {
  openMenu: function($menu, $menuBar, x, y) {
    currentMenu = $menu;
    currentMenuBar = $menuBar;
    if (typeof x == "object" && x.get) {
      menuLauncher = x.get(0);
    } else {
      menuLauncher = null;
    }
    actionTookFocus = false;

    keepTracking = true;
    if (menuStack.length === 0) {
      startTrackingMenus();
    }
    this.closeAll();
    menuStack.push($menu);
    keepTracking = false;
    placeMenu($menu, false, x, y);
  },

  isCurrentMenu: function($menu) {
    return currentMenu && currentMenu.get(0) === $menu.get(0);
  },

  openSubMenu: function($item, $menu) {
    this.closeOpenSiblings($item);
    menuStack.push($menu);
    placeMenu($menu, true, $item);
    $item.addClass("opened");
  },

  closeOpenSiblings: function($item) {
    var $openSubMenu = $item.parent().find(".opened");
    while ($openSubMenu.length > 0) {
      this.closeLast(false);
      $openSubMenu = $item.parent().find(".opened");
    }
  },

  closeLast: function(focus) {
    var $menu, $item;
    if (menuStack.length > 0) {
      $menu = menuStack.pop();

      $item = $menu.parents(".menuItem").eq(0);
      if ($item.length > 0) {
        $item.removeClass("opened");
        if (focus) {
          mvc.setFocus($item.find("a, input").get(0), 0);
        }
      }
      $menu.hide();
      if (menuStack.length === 0) {
        // return focus to where it was before menu was opened
        if (menuLauncher && !actionTookFocus) {
          mvc.setFocus(menuLauncher, 0);
        }
        currentMenu = null;
        if (!keepTracking) {
          stopTrackingMenus();
        }
      }
    }
  },

  closeAll: function() {
    while (menuStack.length > 0) {
      this.closeLast(false);
    }
  }
};

function findItemById(menu, id) {
  var theItem = null;
  $.each(menu.items, function(i, item) {
    if (item.id == id) {
      theItem = item;
      return false;
    }
    if (item.type == "subMenu") {
      theItem = findItemById(item.menu, id);
      if (theItem !== null) {
        return false;
      }
    }
  });
  return theItem;
}

$.widget("ui.menu", {

  _init: function() {
    var self = this;
    var o = this.options;
    var $ctrl = this.element;

    function getChoiceFromId(id) {
      var choice = null;
      var index = id.search(/_c[0-9]+$/);
      if (index >= 0) {
        choice = id.substring(index + 2) * 1;
      }
      return choice;
    }

    function invokeItem(item, choice) {
      var type = item.type;

      if (item.disabled) {
        return false;
      }

      try {
        if (type == "toggle") {
          item.set(!item.get());
        } else if (type == "radioGroup") {
          item.set(item.choices[choice].value);
        } else if (type == "action") {
          actionTookFocus = item.action(o, menuLauncher) || false;
        } else if (type == "subMenu") {
          return false;
        }
      } catch(ex) {
        // ignore error so menu closes
        mvc.logError(ex);
      }
      return true;
    }

    $ctrl.attr("tabindex", -1).addClass(o.menubar ? "menubar" : "menu").hide();

    $ctrl.mouseup(function(e) {
      var $target = $(e.target).closest(".menuItem, .menuItemSep").eq(0);
      var item, type, choice, id;

      // ignore right mouse button up
      if (e.which === 3) { // using which is the jQuery way
        return;
      }

      if ($target.length > 0) {
        if ($target.hasClass("menuItem")) {
          id = $target.get(0).id;
          item = self._getMenuItemFromId(id);
          choice = getChoiceFromId(id);
          if (invokeItem(item, choice)) {
            menuManager.closeAll();
          }
        }
      } else {
        menuManager.closeAll();
      }
      return false;
    }).click(function(e) {
      return false; // don't want default click processing since it id done on mouse up
    }).mousemove(function(e) {
      var $target, $item, $menu;

      if (lastMouseX == e.pageX && lastMouseY == e.pageY) {
        return;
      }
      lastMouseX = e.pageX;
      lastMouseY = e.pageY;
      $target = $(e.target);

      $item = $target.parents(".menuItem").eq(0);
      if ($item.length > 0) {
        if (!$item.hasClass("disabled")) {
          if (!$item.hasClass("focused") && !$item.hasClass("opened")) {
            if (subMenuDelayID) {
              clearTimeout(subMenuDelayID);
            }
            menuManager.closeOpenSiblings($item);
            mvc.setFocus($item.find("a, input").get(0), 0);
            if ($item.find(".menu").length > 0 && !$item.hasClass("opened")) {
              subMenuDelayID = setTimeout(function() {
                subMenuDelayID = null;
                menuManager.openSubMenu($item, $item.find(".menu"));
              }, 400);
            }
          }
          return;
        }
      } else {
        $item = $target.parents(".menuItemSep").eq(0);
      }
      if ($item.length > 0 && $item.get(0) != lastItem) {
        if (subMenuDelayID) {
          clearTimeout(subMenuDelayID);
        }
        menuManager.closeOpenSiblings($item);
      }
      lastItem = $item.get(0);

      $menu = $target.closest(".menu").eq(0);
      if ($menu.length > 0 && !$menu.hasClass("focused")) {
        mvc.setFocus($menu.get(0), 0);
      }
    }).keydown(function(e) {
      var kc = e.keyCode;
      var $cur, $next;

      if (e.altKey) {
        return;
      }
      if ($ctrl.hasClass("focused")) {
        $cur = $ctrl;
      } else {
        $cur = $ctrl.find(".focused");
      }
      if (kc == VK_UP) {
        if ($cur.hasClass("menu")) {
          $next = $cur.find("ul:first").children(".menuItem:not(.disabled):last");
        } else {
          $next = $cur.prevAll(".menuItem:not(.disabled)").eq(0);
        }
        if ($next.length === 0) {
          $next = $cur.parent().children(".menuItem:not(.disabled):last");
        }
        mvc.setFocus($next.find("a, input").get(0), 0);
        return false;
      } else if (kc == VK_DOWN || kc == VK_TAB) {
        if ($cur.hasClass("menu")) {
          $next = $cur.find("ul:first").children(".menuItem:not(.disabled):first");
        } else {
          $next = $cur.nextAll(".menuItem:not(.disabled)").eq(0);
        }
        if ($next.length === 0) {
          $next = $cur.parent().children(".menuItem:not(.disabled):first");
        }
        mvc.setFocus($next.find("a, input").get(0), 0);
        return false;
      } else if (kc == VK_RIGHT) {
        $next = $cur.children(".menu");
        if ($next.length > 0) {
          menuManager.openSubMenu($cur, $next);
          $next = $next.find("ul:first").children(".menuItem:not(.disabled):first");
          mvc.setFocus($next.find("a, input").get(0), 0);
        } else {
          // TODO move in menu bar
        }
        return false;
      } else if (kc == VK_LEFT) {
        $next = $cur.parents(".menuItem"); 
        if ($next.length > 0) {
          menuManager.closeLast(true);
        } else {
          // TODO move in menu bar
        }
        return false;
      } else if (kc == VK_ESC) {
        menuManager.closeLast(true);
        return false;
      } else if (kc == VK_ENTER || kc == VK_SPACE) {
        return false; // ignore until key up (prevent default button handling in IE)
      }
    }).keyup(function(e) {
      var kc = e.keyCode;
      var $cur, item, choice, id;

      if (e.altKey) {
        return;
      }
      if ($ctrl.hasClass("focused")) {
        $cur = $ctrl;
      } else {
        $cur = $ctrl.find(".focused");
      }
      if (kc == VK_ENTER || kc == VK_SPACE) {
        id = $cur.get(0).id;
        item = self._getMenuItemFromId(id);
        choice = getChoiceFromId(id);
        if (item && invokeItem(item, choice)) {
          menuManager.closeAll();
        }
        return false;
      }
    }).focus(function(e) {
      $(this).addClass("focused");
    }).blur(function(e) {
      $(this).removeClass("focused");
    });

  },

  toggle: function(x, y) {
    var $ctrl = this.element;
    if (menuManager.isCurrentMenu($ctrl)) {
      menuManager.closeAll();
    } else {
      this.open(x, y);
    }
  },

  open: function(x, y) {
    var self = this;
    var $ctrl = this.element;
    var o = this.options;
    var idprefix = this.element.get(0).id || "";
    var out = mvc.htmlBuilder();

    this._renderMenu(out, idprefix, o);

    $ctrl.html(out.toString());
    mvc.initIcons($ctrl);

    $ctrl.find("div.menu").focus(function(e) {
      $(this).addClass("focused");
    }).blur(function(e) {
      $(this).removeClass("focused");
    });

    // TODO use delegation
    $ctrl.find(".menuItem").find("a, input").focus(function(e) {
      $(this).parents(".menuItem").eq(0).addClass("focused");
    }).blur(function(e) {
      $(this).parents(".menuItem").eq(0).removeClass("focused");
    });

    menuManager.openMenu($ctrl, null, x, y);

  },

  find: function (id) {
    var o = this.options;
    return findItemById(o, id);
  },

  // don't call while tracking menus
  destroy: function() {
    var o = this.options;
    $(this.element).html("").removeClass(o.menubar ? "menubar" : "menu");
    $.widget.prototype.destroy.call(this);
  },

  _getMenuItemFromId: function(id) {
    var o = this.options;
    var idParts = id.split("_");
    var path = [];
    var i, mi, item = null;

    i = idParts.length - 1;
    if (idParts[i].match(/^c[0-9]+$/)) {
      i--;
    }
    for (; i > 0; i--) {
      if (idParts[i].match(/^[0-9]+$/)) {
        path.unshift(idParts[i] * 1);
      } else {
        break;
      }
    }
    mi = o.items;
    for (i = 0; i < path.length; i++) {
      item = mi[path[i]];
      if (!item || (i < path.length -1 && !item.menu)) {
        return null;
      }
      if (item.menu) {
        mi = item.menu.items;
      }
    }
    return item;
  },

  _renderMenu: function(out, idprefix, menu) {
    var self = this;
    
    function separator() {
      out.markup("<li class='menuItemSep'><span class='statusCol'>&nbsp;</span><span class='hSeparator'></span></li>");
    }

    out.markup("<div class='dstr'></div><div class='dsbl'><div class='ds'><div class='menuContent'><ul>");
    $.each(menu.items, function(i, item) {
      var type = item.type;
      var state, label, itemClass;
      var id = idprefix + "_" + i;
      var forid = idprefix + "I" + i;

      itemClass = type == "separator" ? "menuItemSep" : "menuItem";
      if (item.disabled) {
        itemClass += " disabled";
      }

      if (type == "radioGroup") {
        // add separator before if there isn't one and not first menu item
        if (i > 0 && menu.items[i - 1].type != "separator") {
          separator();
        }
        state = item.get(); // state is current selected choice
        $.each(item.choices, function(j, choice) {
          var rid = id + "_c" + j;
          var rforid = forid + "_c" + j;
          var checked = state == choice.value;
          out.markup("<li id='").attr(rid).markup("' class='menuItem'>");

          // icon
          out.markup("<span class='statusCol'>");
          mvc.renderIcon(out, checked ? "menuBullet" : "");

          out.markup("<input type='radio' name='").attr(id).markup("C' id='").attr(rforid).markup("'").
            optionalBoolAttr("disabled", choice.disabled).optionalBoolAttr("checked", checked).markup(">");
          out.markup("&nbsp;</span>");
          // label
          if (choice.disabled) {
            out.markup("<span class='label'>").content(choice.label).markup("</span>"); // TODO acc add (disabled)
          } else {
            out.markup("<label for='").attr(rforid).markup("'>").content(choice.label).markup("</label>");
          }
          // TODO accel
          // no sub menu
          out.markup("<span class='subMenuCol'>&nbsp;</span>");
          out.markup("</li>");
        });
        // add separator after if there isn't one and not last menu item
        if (i < menu.items.length - 1 && menu.items[i + 1].type != "separator") {
          separator();
        }
      } else if (type == "separator") {
        separator();
      } else {
        out.markup("<li id='").attr(id).markup("' class='").attr(itemClass).markup("'>");
        if (type == "action") {
          // icon
          out.markup("<span class='statusCol'>");
          if (item.icon) {
            mvc.renderIcon(out, item.icon);
          }
          out.markup("&nbsp;</span>");
          // label
          if (item.disabled) {
            out.markup("<span class='label'>").content(item.label).markup("</span>");
          } else {
            out.markup("<a href=''>").content(item.label).markup("</a>");
          }
          // TODO accel
          // no sub menu
          out.markup("<span class='subMenuCol'>&nbsp;</span>");
        } else if (type == "toggle") {
          state = item.get();
          // icon
          out.markup("<span class='statusCol'>");
          if (item.label) {
            mvc.renderIcon(out, state ? "menuChecked" : "");
          }
          out.markup("<input type='checkbox' id='").attr(forid).markup("'").
            optionalBoolAttr("disabled", item.disabled).optionalBoolAttr("checked", state).markup(">");
          out.markup("&nbsp;</span>");
          // label
          label = item.label;
          if (!label) {
            label = state ? item.onLabel : item.offLabel;
          }
          if (item.disabled) {
            out.markup("<span class='label'>").content(label).markup("</span>"); // TODO acc add (disabled)
          } else {
            out.markup("<label for='").attr(forid).markup("'>").content(label).markup("</label>");
          }
          // TODO accel
          // no sub menu
          out.markup("<span class='subMenuCol'>&nbsp;</span>");
        } else if (type == "subMenu") {
          // icon
          out.markup("<span class='statusCol'>");
          if (item.icon) {
            mvc.renderIcon(out, item.icon);
          }
          out.markup("&nbsp;</span>");
          // label
          if (item.disabled) {
            out.markup("<span class='label'>").content(item.label).markup("</span>");
          } else {
            out.markup("<a href=''>").content(item.label).markup("</a>");
          }
          // sub menus don't have an accelerator
          // sub menu
          out.markup("<span class='subMenuCol'>");
          mvc.renderIcon(out, "subMenu");
          out.markup("&nbsp;</span><div class='menu'>");
          self._renderMenu(out, id, item.menu);
          out.markup("</div>");
        }
        out.markup("</li>");
      }
    });
    out.markup("</ul></div></div></div>");

  }

});


$.extend($.ui.menu, {
  version: "1.7",
  getter: "find",
  defaults: {
    menubar: true,
    items: []
  }
});

})(jQuery);
