/*!
  jQuery plugin for drawing gauges.
  Copyright (c) 2009, 2010, Oracle and/or its affiliates. All rights reserved.
*/
/*
  A gauge gives a two dimensional presentation to a set of scalar values
  including statistics about how they have changed over time (if available).

  Depends on flot, the color plugin contained in flot and the flot.markers plugin. Makes use of excanvas if present.
 
  Gauge data is an array of objects as follows:
  {
    value: <n>,
    avg: <n>, // optional
    stdev: <n>, // optional
    min: <n>, // optional
    max: <n>, // optional
    color: <color>,
    shape: <shape>, // shape name from the markers plugin
    fill: <bool>    // applies to marker shape
  }

  Options: 
  {
    type: "linearHorizontal", "linearVertical" or "radial",
    min: <n>, // optional
    max: <n>, // optional
    thresholdMin: <n>, // optional
    thresholdMax: <n>, // optional
    tickDecimals: <n>, // optional
    tickFormatter: <fn>, // function takes a number and returns a formated string representing that number
    labelMargin: <n>, // amount of space around labels 
    color: <color>, // chart forground color
    fillColor: <color>,  // color to fill markers with
    backgroundColor: <color>, // default is null which means transparent
    highlightColor: <color>, // shading color for outside threshold
    tickColor: <color>, // color of tick lines
    borderColor: <n>, // optional - default to color
    borderWidth: <n>,
    padding: <n>, // pixels padding inside canvas
    clickable: <bool>, // if true data value can be clicked on
    hoverable: <bool>, // if true data value can be hovered over 
    mouseActiveRadius: <n>, // defines hit area around a point
    highlightRadius: <n>
  }

  Events: gaugeclick and gaugehover events receive arguments pos and item

  pos is an object representing the mouse position at the time of the event
  { 
    pageX: <n>, // mouse page x
    pageY: <n>  // mouse page y
  }

  item is an object if the mouse was over or clicked on/near a data value or null otherwise
  {
    data: <data>, // the data object (described above) clicked on or hovered over
    index: <i> // index of data object in the array of gauge data
    px: <n>, // canvas x position of data value
    py: <n>, // canvas y position of data value 
    pageX: <n> // page x position of data value
    pageY: <n> // page y position of data value
   }

*/
/*global window,jQuery */
(function ($) {

var TYPE_VERTICAL = "linearVertical";
var TYPE_HORIZONTAL = "linearHorizontal";
var TYPE_RADIAL = "radial";

var defaultOptions = {
  type: TYPE_VERTICAL,
  min: null,
  max: null,
  thresholdMin: null,
  thresholdMax: null,
  tickFormatter: null,
  tickDecimals: null,
  labelMargin: 5,
  color: "#000000",
  fillColor: "#ffffff",
  backgroundColor: null,
  highlightColor: "#eeeeee",
  tickColor: "rgba(0,0,0,0.15)",
  borderColor: null,
  borderWidth: 2,
  padding: 4,
  clickable: false,
  hoverable: false,
  mouseActiveRadius: 10, // defines hit area around a point
  highlightRadius: 4
};

var L_MIN_WIDTH = 40;
var L_MAX_WIDTH = 100;
var Y_MARGIN = 8;
var DEFAULT_MIN = -1;
var DEFAULT_MAX = 1;


// 2 functions to convert from polar to cartesian coordinates
function polar2cartX(a, r) {
  return r * Math.cos(a);
}
function polar2cartY(a, r) {
  return r * Math.sin(a);
}

$.gauge = function($elem, data, options) {
  var self = {};
  var canvasWidth;
  var canvasHeight;
  var chartWidth;
  var chartHeight;
  var labelWidth;
  var labelHeight;
  var offsets = { top: 0, left: 0, bottom: 0, right: 0 };
  var canvas;
  var ctx;
  var overlay;
  var octx;
  var ticks;
  var eventHolder;
  var highlights = [];
  var redrawTimeout = null;

  // public properties
  self.options = $.extend({}, defaultOptions, options);
  self.data = data;

  // public methods
  self.getWidth = function() {
    return chartWidth;
  };

  self.getHeight = function() {
    return chartHeight;
  };

  self.offset = function () {
    var holderOffset = eventHolder.offset();
    holderOffset.left += offsets.left;
    holderOffset.top += offsets.top;
    return holderOffset;
  };

  self.getChartOffsets = function() {
    return offsets;
  };

  self.getCanvas = function() {
    return canvas;
  };

  self.getValuePos = function(i, v) {
    var o = this.options;
    var pos = {};
    var a, r, h;

    if (o.type == TYPE_HORIZONTAL) {
      pos.x = chartWidth - this.yp2c(v);
      pos.y = this.xp2c(i, 85);
    } else if (o.type == TYPE_VERTICAL) {
      pos.y = this.yp2c(v);
      pos.x = this.xp2c(i, 85);
    } else if (o.type == TYPE_RADIAL) {
      a = Math.PI - this.yp2c(v);
      r = this.xp2c(i, 15);
      h = chartHeight - Y_MARGIN;
      pos.x = polar2cartX(a, r) + h;
      pos.y = h - polar2cartY(a, r);
    }
    return pos;
  };

  // self.draw is public but set later

  //
  // implementation
  //

  function initOptions() {
    var o = self.options;
    if (!o.borderColor) {
      o.borderColor = o.color;
    }
  }

  function createCanvas() {
    canvasWidth = $elem.width();
    canvasHeight = $elem.height();

    // Can't use .html("") - IE needs the canvas wrapper ($elem) content cleared out with innerHTML to avoid mem leak
    // but we must also cleanup events added with jQuery.
    $elem.find("canvas").each(function(){
      $.event.remove(this);
      $.removeData(this);
    });
    $elem.get(0).innerHTML = "";

    if ($elem.css("position") == 'static') {
      $elem.css("position", "relative"); // for positioning labels and overlay
    }

    if (canvasWidth <= 0 || canvasHeight <= 0) {
      throw "Invalid dimensions for gauge: " + canvasWidth + "x" + canvasHeight;
    }

    if (window.G_vmlCanvasManager) {
      window.G_vmlCanvasManager.init_(document); // init excanvas if present
    }

    canvas = document.createElement('canvas');
    canvas.width = canvasWidth;
    canvas.height = canvasHeight;
    if (window.G_vmlCanvasManager) {
      canvas = window.G_vmlCanvasManager.initElement(canvas);
    }

    canvas = $(canvas).appendTo($elem).get(0);
    ctx = canvas.getContext("2d");

    // overlay canvas for drawing highlight
    overlay = document.createElement('canvas');
    overlay.width = canvasWidth;
    overlay.height = canvasHeight;
    if (window.G_vmlCanvasManager) {
      overlay = window.G_vmlCanvasManager.initElement(overlay);
    }

    overlay = $(overlay).appendTo($elem).css({ position: 'absolute', left: 0, top: 0 }).get(0);
    octx = overlay.getContext("2d");

  }

  // sets or updates options.min, options.max
  function processData() {
    var o = self.options;
    var cd = self.data;
    var min, max;

    min = o.min;
    if (min === null) {
      min = cd.length > 0 && cd[0].min ? cd[0].min : DEFAULT_MIN;
    }
    $.each(cd, function (i, data) {
      if (data.min < min) {
        min = data.min;
      }
    });

    max = o.max;
    if (max === null) {
      max = cd.length > 0 && cd[0].max ? cd[0].max : DEFAULT_MAX;
    }
    $.each(cd, function (i, data) {
      if (data.max > max) {
        max = data.max;
      }
    });

    o.min = min;
    o.max = max;
    if (o.min == o.max) {
      // can't allow the range to be zero
      o.max += 0.01;
      o.min -= 0.01;
    }
  }

  // round to nearby lower multiple of base
  function floorInBase(n, base) {
    return base * Math.floor(n / base);
  }
  
  // sets ticks this logic is from flot
  function initTicks() {
    var o = self.options;
    var numTicks;

    // estimate number of ticks
    if (o.type == TYPE_HORIZONTAL) {
       // heuristic based on the model a*sqrt(x) fitted to
       // some reasonable data points
      numTicks = 0.3 * Math.sqrt(canvasWidth);
    } else if (o.type == TYPE_VERTICAL) {
      numTicks = 0.3 * Math.sqrt(canvasHeight);
    } else { // type is radial
      numTicks = 0.3 * Math.sqrt(Math.PI * Math.min(canvasHeight, canvasWidth)); // circumference of 1/2 circle
    }

    var delta = (o.max - o.min) / numTicks;
    var size, formatter, magn, norm;

    // pretty rounding of base-10 numbers
    var maxDec = o.tickDecimals;
    var dec = -Math.floor(Math.log(delta) / Math.LN10);
    if (maxDec !== null && dec > maxDec) {
      dec = maxDec;
    }

    magn = Math.pow(10, -dec);
    norm = delta / magn; // norm is between 1.0 and 10.0
    
    if (norm < 1.5) {
      size = 1;
    } else if (norm < 3) {
      size = 2;
      // special case for 2.5, requires an extra decimal
      if (norm > 2.25 && (maxDec === null || dec + 1 <= maxDec)) {
        size = 2.5;
        ++dec;
      }
    } else if (norm < 7.5) {
      size = 5;
    } else {
      size = 10;
    }
    size *= magn;

    o.tickDecimals = Math.max(0, (maxDec !== null) ? maxDec : dec);

    formatter = function (v) {
      return v.toFixed(o.tickDecimals);
    };

    if (!$.isFunction(o.tickFormatter)) {
      o.tickFormatter = formatter;
    }

    // generate all possible ticks
    ticks = [];
    var start = floorInBase(o.min, size);
    var i = 0;
    var v = Number.NaN;
    var prev;
    do {
      prev = v;
      v = start + i * size;
      ticks.push({ v: v, label: o.tickFormatter(v) });
      i += 1;
    } while (v <= o.max && v != prev);
  }

  // sets labelWidth, labelHeight
  function measureLabels() {
    var o = self.options;
    var i, labels = [], l, $dummyDiv;

    if (o.type == TYPE_HORIZONTAL) {
      // to avoid measuring the widths of the labels, we
      // construct fixed-size boxes and put the labels inside
      // them, we don't need the exact figures and the
      // fixed-size box content is easy to center
      labelWidth = canvasWidth / (ticks.length > 0 ? ticks.length : 1);

      // measure x label heights
      for (i = 0; i < ticks.length; ++i) {
        l = ticks[i].label;
        if (l) {
          labels.push('<div class="tickLabel" style="float:left;width:' + labelWidth + 'px">' + l + '</div>');
        }
      }

      if (labels.length > 0) {
        $dummyDiv = $('<div style="position:absolute;top:-10000px;width:10000px;">' +
                    labels.join("") + '<div style="clear:left"></div></div>').appendTo($elem);
        labelHeight = $dummyDiv.height();
        $dummyDiv.remove();
      }
    } else if (o.type == TYPE_VERTICAL || o.type == TYPE_RADIAL) {
      // calculate y label dimensions
      for (i = 0; i < ticks.length; ++i) {
        l = ticks[i].label;
        if (l) {
          labels.push('<div class="tickLabel">' + l + '</div>');
        }
      }

      if (labels.length > 0) {
        $dummyDiv = $('<div style="position:absolute;top:-10000px;">' + labels.join("") + '</div>').appendTo($elem);
        labelWidth = $dummyDiv.width();
        labelHeight = $dummyDiv.find("div").height();
        $dummyDiv.remove();
      }
    }

    if (labelWidth === null) {
      labelWidth = 0;
    }
    if (labelHeight === null) {
      labelHeight = 0;
    }
  }

  function insertLabels() {
    var o = self.options;
    $elem.find(".tickLabels").remove();

    var html = ['<div class="tickLabels" style="color:' + o.color + '">'];

    function addLabels(labelGenerator) {
      for (var i = 0; i < ticks.length; ++i) {
        var tick = ticks[i];
        if (!tick.label || tick.v < o.min || tick.v > o.max) {
          continue;
        }
        html.push(labelGenerator(tick));
      }
    }

    var margin = o.borderWidth + o.labelMargin;

    if (o.type == TYPE_HORIZONTAL) {
      addLabels(function (tick) {
        return '<div style="position:absolute;top:' + (offsets.top + chartHeight + margin) + 
        'px;left:' + Math.round(offsets.left + chartWidth - self.yp2c(tick.v) - labelWidth / 2) + 
        'px;width:' + labelWidth + 'px;text-align:center" class="tickLabel">' + tick.label + "</div>";
      });
    } else if (o.type == TYPE_VERTICAL) {
      addLabels(function (tick) {
        return '<div style="position:absolute;top:' + Math.round(offsets.top + self.yp2c(tick.v) - labelHeight / 2) + 
        'px;right:' + (offsets.right + chartWidth + margin) + 
        'px;width:' + labelWidth + 'px;text-align:right" class="tickLabel">' + tick.label + "</div>";
      });
    } else if (o.type == TYPE_RADIAL) {
      addLabels(function (tick) {
        var a = Math.PI - self.yp2c(tick.v);
        var r = chartHeight - Y_MARGIN;
        var x = polar2cartX(a, r + margin);
        var l = x > 0 ? 0 : - labelWidth;
        return '<div style="position:absolute;top:' + Math.round(offsets.top + r - polar2cartY(a, r + margin) - labelHeight / 2) + 
        'px;left:' + Math.round(offsets.left + r + x + l) + 
        'px;width:' + labelWidth + 'px;text-align:center" class="tickLabel">' + tick.label + "</div>";
      });
    }
    html.push('</div>');

    $elem.append(html.join(""));
  }

  function initChart() {
    var o = self.options;
    var range = o.max - o.min;
    var count = self.data.length;
    var yscale, xwidth, xscale;
    var offset, width, height, radius;

    measureLabels();
    var margin = o.borderWidth + o.labelMargin;

    offsets.top = offsets.left = offsets.bottom = offsets.right = o.padding + o.borderWidth;

    if (o.type == TYPE_HORIZONTAL) {
      width = canvasHeight - offsets.top - offsets.bottom - labelHeight - margin;
      height = canvasWidth - offsets.left - offsets.right - (Y_MARGIN * 2);
    } else if (o.type == TYPE_VERTICAL) {
      width = canvasWidth - offsets.left - offsets.right - labelWidth - margin;
      height = canvasHeight - offsets.top - offsets.bottom - (Y_MARGIN * 2);
    } else if (o.type == TYPE_RADIAL) {
      width = canvasWidth - offsets.left - offsets.right - 2 * (labelWidth + margin);
      height = canvasHeight - offsets.top - offsets.bottom - Y_MARGIN - labelHeight - margin;
      radius = Math.min(height, width / 2);
    }

    if (count === 0) {
      count = 1; // calc xwidth as if there were one data value 
    }
    xwidth = (o.type == TYPE_RADIAL ? radius : width) / count;
    offset = 0;
    if (xwidth < L_MIN_WIDTH) {
      xwidth = L_MIN_WIDTH; // some data will get clipped
    } else if (xwidth > L_MAX_WIDTH) {
      xwidth = L_MAX_WIDTH;
      offset = (width - (count * xwidth)) / 2;
    }
    xscale = xwidth / 100;

    if (o.type == TYPE_HORIZONTAL || o.type == TYPE_VERTICAL) {
      yscale = height / range;

      self.yp2c = function(p) {
        return Y_MARGIN + ((o.max - p) * yscale);
      };

      self.xp2c = function(i, p) {
        return (i * xwidth) + p * xscale;
      };

      if (o.type == TYPE_HORIZONTAL) {
        offsets.top += offset;
        offsets.bottom += offset + labelHeight + margin;
      } else { // linearVertical
        offsets.left += offset + labelWidth + margin;
        offsets.right += offset;
      }
    } else if (o.type == TYPE_RADIAL) {
      yscale = Math.PI / range;
      // angle in rads
      self.yp2c = function(p) {
        return (p - o.min) * yscale;
      };
      xwidth = xwidth * 0.8;
      // radius
      self.xp2c = function(i, p) {
        return radius - (i * xwidth) - (p * xscale);
      };

      offsets.top += labelHeight + margin;
      offsets.left += labelWidth + margin;
      offsets.right += labelWidth + margin;
      if (radius < height) {
        offsets.top += height - radius;
      }
      if (radius < width / 2) {
        offsets.left += width / 2 - radius;
        offsets.right += width / 2 - radius;
      }
    }

    chartWidth = canvasWidth - offsets.right - offsets.left;
    chartHeight = canvasHeight - offsets.top - offsets.bottom;
  }

  // draw the triangle that marks min or max
  function drawMinMax(x, y, max) {
    var h = 6;
    var w = max ? -8 : 8;
    var yb = y + (max ? - h : h);
    ctx.beginPath();
    ctx.moveTo(x, y);
    ctx.lineTo(x - w, yb);
    ctx.lineTo(x + w, yb);
    ctx.lineTo(x, y);
    ctx.fill();
  }

  function drawGrid() {
    var o = self.options;
    var bw = o.borderWidth;
    var v, i, r, t, b, h;

    function radialPath(bw, r, margin) {
      var x1 = -bw / 2;
      var x2 = (r * 2) + (bw / 2);
      var b = r + bw + margin;
      ctx.moveTo(x1, r);
      ctx.lineTo(x1, b);
      ctx.lineTo(x2, b);
      ctx.lineTo(x2, r);
      ctx.arc(r, r, r + (bw / 2), 0, Math.PI, true);
    }

    function drawThreshold() {
      if (o.type == TYPE_HORIZONTAL) {
        ctx.fillRect(chartWidth - t, 0, -h, chartHeight);
      } else if (o.type == TYPE_VERTICAL) {
        ctx.fillRect(0, t, chartWidth, h);
      } else if (o.type == TYPE_RADIAL) {
        r = chartHeight - Y_MARGIN;
        ctx.beginPath();
        ctx.moveTo(r, r);
        ctx.arc(r, r, r, Math.PI + b, Math.PI + t, false);
        ctx.fill();
      }
    }

    ctx.save();
    ctx.translate(offsets.left, offsets.top);

    // background
    if (o.backgroundColor) {
      ctx.fillStyle = o.backgroundColor;
      if (o.type == TYPE_RADIAL) {
        ctx.beginPath();
        radialPath(bw ? bw : 1, chartHeight - Y_MARGIN, Y_MARGIN);
        ctx.fill();
      } else {
        ctx.fillRect(0, 0, chartWidth, chartHeight);
      }
    }

    // shade areas beyond the threshold
    if (o.thresholdMax !== null && o.thresholdMax < o.max) {
      ctx.fillStyle = o.highlightColor;
      t = self.yp2c(o.max);
      b = self.yp2c(o.thresholdMax);
      h = b - t;
      drawThreshold();
    }
    if (o.thresholdMin !== null && o.thresholdMin > o.min) {
      ctx.fillStyle = o.highlightColor;
      t = self.yp2c(o.thresholdMin);
      b = self.yp2c(o.min);
      h = b - t;
      drawThreshold();
    }

    // tick lines
    ctx.lineWidth = 1;
    ctx.strokeStyle = o.tickColor;
    ctx.beginPath();
    for (i = 0; i < ticks.length; ++i) {
      v = ticks[i].v;
      if (v < o.min || v > o.max) {
        continue;
      }
      if (o.type == TYPE_HORIZONTAL) {
        v = Math.floor(self.yp2c(v) + 0.5);
        ctx.moveTo(v, 0);
        ctx.lineTo(v, chartHeight);
      } else if (o.type == TYPE_VERTICAL) {
        v = Math.floor(self.yp2c(v) + 0.5);
        ctx.moveTo(0, v);
        ctx.lineTo(chartWidth, v);
      } else if (o.type == TYPE_RADIAL) {
        v = Math.PI - self.yp2c(v);
        r = chartHeight - Y_MARGIN;
        ctx.moveTo(r, r);
        ctx.lineTo(polar2cartX(v, r) + r, r - polar2cartY(v, r));
      }
    }
    ctx.stroke();

    // border
    if (bw) {
      ctx.lineWidth = bw;
      ctx.strokeStyle = o.borderColor;
      if (o.type == TYPE_RADIAL) {
        ctx.beginPath();
        radialPath(bw, chartHeight - Y_MARGIN, Y_MARGIN);
        ctx.stroke();
      } else {
        ctx.strokeRect(-bw / 2, -bw / 2, chartWidth + bw, chartHeight + bw);
      }
    }

    ctx.restore();
  }

  function drawLinearGauge(i, data) {
    var o = self.options;
    var dataColor;
    var yv, t, l, h, w, m, b, r;

    ctx.save();
    if (o.type == TYPE_HORIZONTAL) {
      ctx.rotate(Math.PI / 2); // 90 deg
      ctx.translate(offsets.top, - offsets.left - chartWidth);
    } else {
      ctx.translate(offsets.left, offsets.top);
    }

    m = self.xp2c(i, 50);
    ctx.lineWidth = o.borderWidth;
    ctx.strokeStyle = o.color;
    ctx.beginPath();
    ctx.moveTo(m, self.yp2c(o.min));
    ctx.lineTo(m, self.yp2c(o.max));
    ctx.stroke();

    if (data.value !== null) {
      dataColor = data.color;
      yv = self.yp2c(data.value);

      if (data.min !== undefined && data.max !== undefined) {
        ctx.lineWidth = 0;
        ctx.fillStyle = dataColor;
        drawMinMax(m, self.yp2c(data.max), true);
        drawMinMax(m, self.yp2c(data.min), false);
      }

      if (data.avg !== undefined && data.stdev !== undefined) {
        t = self.yp2c(data.avg + data.stdev / 2);
        b = self.yp2c(data.avg - data.stdev / 2);
        l = self.xp2c(i, 40);
        r = self.xp2c(i, 60);
        ctx.lineWidth = 1;
        ctx.strokeStyle = o.color;
        ctx.fillStyle = $.color.parse(dataColor).scale('a', 0.5).normalize().toString();
        ctx.fillRect(l, t, r - l, b - t);
        ctx.strokeRect(l, t, r - l, b - t);

        t = self.yp2c(data.avg);
        ctx.lineWidth = 2;
        ctx.strokeStyle = o.color;
        ctx.beginPath();
        ctx.moveTo(l + 1, t);
        ctx.lineTo(r - 1, t);
        ctx.stroke();
      }

      ctx.strokeStyle = dataColor;
      ctx.lineWidth = 2;
      ctx.beginPath();
      ctx.moveTo(self.xp2c(i, 25), yv);
      ctx.lineTo(self.xp2c(i, 75), yv);
      ctx.stroke();

      ctx.save();
      // fix transformations so that markers are drawn with normal orientation
      ctx.translate(self.xp2c(i, 85), yv);
      if (o.type == TYPE_HORIZONTAL) {
        ctx.rotate(-Math.PI / 2); // - 90
      }
      $.plot.markers.draw(ctx, data.shape, 0, 0, 4, 2, 0, dataColor,
        data.fill ? o.fillColor : dataColor);
      ctx.restore();
    }

    ctx.restore();
  }

  function drawRadialGauge(i, data) {
    var o = self.options;
    var dataColor;
    var r = chartHeight - Y_MARGIN;
    var rm, a, x1, x2, b, e;

    ctx.save(); // move to center of half circle and angles measure from negative X axis
    ctx.translate(offsets.left + r, offsets.top + r);
    ctx.scale(-1, -1);

    rm = self.xp2c(i, 50);
    ctx.lineWidth = o.borderWidth;
    ctx.strokeStyle = o.color;
    ctx.beginPath();
    ctx.arc(0, 0, rm, Math.PI, 0, true);
    ctx.stroke();

    if (data.value !== null) {
      dataColor = data.color;
      a = self.yp2c(data.value);

      if (data.min !== undefined && data.max !== undefined) {
        ctx.lineWidth = 0;
        ctx.fillStyle = dataColor;
        ctx.save(); // translate so that triangle base is perpendicular to arc
        b = self.yp2c(data.max);
        ctx.translate(polar2cartX(b, rm), polar2cartY(b, rm));
        ctx.rotate(Math.PI + b);
        drawMinMax(0, 0, true);
        ctx.restore();
        ctx.save(); // translate so that triangle base is perpendicular to arc
        b = self.yp2c(data.min);
        ctx.translate(polar2cartX(b, rm), polar2cartY(b, rm));
        ctx.rotate(Math.PI + b);
        drawMinMax(0, 0, false);
        ctx.restore();
      }

      if (data.avg !== undefined && data.stdev !== undefined) {
        b = self.yp2c(data.avg + data.stdev / 2);
        e = self.yp2c(data.avg - data.stdev / 2);
        x1 = self.xp2c(i, 40);
        x2 = self.xp2c(i, 60);
        ctx.lineWidth = 1;
        ctx.strokeStyle = o.color;
        ctx.fillStyle = $.color.parse(dataColor).scale('a', 0.5).normalize().toString();
        ctx.beginPath();
        ctx.moveTo(polar2cartX(b, x1), polar2cartY(b, x1));
        ctx.lineTo(polar2cartX(b, x2), polar2cartY(b, x2));
        ctx.arc(0, 0, x2, b, e, true);
        ctx.moveTo(polar2cartX(e, x2), polar2cartY(e, x2));
        ctx.lineTo(polar2cartX(e, x1), polar2cartY(e, x1));
        ctx.arc(0, 0, x1, e, b, false);
        ctx.fill();
        ctx.stroke();

        b = self.yp2c(data.avg);
        ctx.lineWidth = 2;
        ctx.strokeStyle = o.color;
        ctx.beginPath();
        ctx.moveTo(polar2cartX(b, x1), polar2cartY(b, x1));
        ctx.lineTo(polar2cartX(b, x2), polar2cartY(b, x2));
        ctx.stroke();
      }

      ctx.strokeStyle = dataColor;
      x1 = self.xp2c(i, 25);
      x2 = self.xp2c(i, 75);
      ctx.lineWidth = 2;
      ctx.beginPath();
      ctx.moveTo(polar2cartX(a, x1), polar2cartY(a, x1));
      ctx.lineTo(polar2cartX(a, x2), polar2cartY(a, x2));
      ctx.stroke();
      x1 = self.xp2c(i, 15);
      ctx.save();
      // fix transformations so that markers are drawn with normal orientation
      ctx.translate(polar2cartX(a, x1), polar2cartY(a, x1));
      ctx.scale(-1, -1);
      $.plot.markers.draw(ctx, data.shape, 0, 0, 4, 1, 0, dataColor,
        data.fill ? o.fillColor : dataColor);
      ctx.restore();
    }

    ctx.restore();
  }

  function draw() {
    var o = self.options;

    ctx.clearRect(0, 0, canvasWidth, canvasHeight);
    drawGrid();

    if (o.type == TYPE_RADIAL) {
      $.each(self.data, function (i, data) {
        drawRadialGauge(i, data);
      });
    } else {
      $.each(self.data, function (i, data) {
        drawLinearGauge(i, data);
      });
    }
  }

  // returns the data item the mouse is over, or null if none is found
  function findNearbyItem(mouseX, mouseY) {
    var o = self.options;
    var cd = self.data;
    var maxDistance = o.mouseActiveRadius;
    var foundItem = -1;
    var pos;

    $.each(cd, function(i, data) {
      var v = data.value;
      var dx, dy, dist, a, r, h;

      if (v === null) {
        return true;
      }

      pos = self.getValuePos(i, v);
      dx = Math.abs(pos.x - mouseX);
      dy = Math.abs(pos.y - mouseY);
      dist = Math.sqrt(dx * dx + dy * dy);
      if (dist < o.mouseActiveRadius) {
        foundItem = i;
        return false;
      }
    });

    if (foundItem >= 0) {
      return { data: cd[foundItem], index: foundItem, px: pos.x, py: pos.y };
    }
    return null;
  }

  function triggerClickHoverEvent(eventname, event) {
    var holderOffset = eventHolder.offset();
    var pos = { pageX: event.pageX, pageY: event.pageY };
    var canvasX = event.pageX - holderOffset.left - offsets.left;
    var canvasY = event.pageY - holderOffset.top - offsets.top;

    var item = findNearbyItem(canvasX, canvasY);

    if (item) {
      // fill in mouse pos for any listeners out there
      item.pageX = item.px + holderOffset.left + offsets.left;
      item.pageY = item.py + holderOffset.top + offsets.top;
    }

    $elem.trigger(eventname, [ pos, item ]);
  }

  function bindEvents() {
    var o = self.options;
    // include the canvas in the event holder too, because IE 7
    // sometimes has trouble with the stacking order
    eventHolder = $([overlay, canvas]);

    // bind events
    if (o.hoverable) {
      eventHolder.mousemove(function(e) {
        triggerClickHoverEvent("gaugehover", e);
      });
    }
    if (o.clickable) {
      eventHolder.click(function(e) {
        triggerClickHoverEvent("gaugeclick", e);
      });
    }
  }

  function drawOverlay() {
    var o = self.options;
    var i, index, data, a, r, x, y;
    var hRadius;

    redrawTimeout = null;

    // draw highlights
    octx.save();
    octx.clearRect(0, 0, canvasWidth, canvasHeight);
    if (o.type == TYPE_HORIZONTAL) {
      octx.rotate(Math.PI / 2); // 90 deg
      octx.translate(offsets.top, - offsets.left - chartWidth);
    } else if (o.type == TYPE_VERTICAL) {
      octx.translate(offsets.left, offsets.top);
    } else { // radial
      r = chartHeight - Y_MARGIN;
      octx.translate(offsets.left + r, offsets.top + r);
      octx.scale(-1, -1);
    }

    octx.lineWidth = o.highlightRadius;
    hRadius = 1.5 * o.highlightRadius;
    for (i = 0; i < highlights.length; ++i) {
      index = highlights[i];
      data = self.data[index];

      if (o.type == TYPE_RADIAL) {
        a = self.yp2c(data.value);
        r = self.xp2c(index, 15);
        x = polar2cartX(a, r);
        y = polar2cartY(a, r);
      } else {
        y = self.yp2c(data.value);
        x = self.xp2c(index, 85);
      }

      octx.strokeStyle = $.color.parse(data.color).scale('a', 0.5).toString();
      octx.beginPath();
      octx.arc(x, y, hRadius, 0, 2 * Math.PI, false);
      octx.stroke();

    }
    octx.restore();
  }

  function triggerRedrawOverlay() {
    if (!redrawTimeout) {
      redrawTimeout = setTimeout(drawOverlay, 30);
    }
  }

  function indexOfHighlight(index) {
    var i;
    for (i = 0; i < highlights.length; i++) {
      if (highlights[i] === index) {
        return i;
      }
    }
    return -1;
  }

  function highlight(index) {
    var i = indexOfHighlight(index);
    if (i == -1) {
      highlights.push(index);
      triggerRedrawOverlay();
    }
  }

  function unhighlight(index) {
    if (index === null || index === undefined) {
      highlights = [];
      triggerRedrawOverlay();
    }

    var i = indexOfHighlight(index);
    if (i != -1) {
      highlights.splice(i, 1);
      triggerRedrawOverlay();
    }
  }

  initOptions();
  createCanvas();
  processData();
  initTicks();
  initChart();
  insertLabels();
  draw();
  bindEvents();

  self.draw = draw;
  self.highlight = highlight;
  self.unhighlight = unhighlight;

  return self;
};

})(jQuery);
