/*!
  Flot plugin for drawing marker symbols in place of round points.
  Copyright (c) 2009, 2010, Oracle and/or its affiliates. All rights reserved.
*/
/*
  In the series use either markers or points but not both.

  Options for the plugin are:
  series: {
      markers: {
          show: <bool>, // default is false
          shape: <shape-name>, // Optional shape name listed below. If not present or null a generated default is used
          fill: <bool> // Optional. If true use fillColor else fill with line color. If not present or null a generated default is used
          fillColor: <color>, // default is "#ffffff"
          radius: <n>, // default is 4
      }
  }
  The shapes are:
    "circle", "diamond", "triangleUp", "triangleDown", "square", "bar", "halfCircleLeft", "halfCircleRight"
  They are assigned in this order first with fillColor then filled with the series color. Then repeat.

  The stroke color is series.color. The stroke width comes from series.line.lineWidth

*/
/*global jQuery*/
(function ($) {
  var options = {
    series: {
      markers: {
        show: false,
        // shape: generate by default from markerShapes
        // fill: generate by default
        radius: 4,
        fillColor: "#ffffff"
      }
    }
  };

  var shapes = {
    circle: function(ctx, x, y, radius) {
      ctx.arc(x, y, radius, 0, 2 * Math.PI, false);
    },
    square: function(ctx, x, y, radius) {
      var l = radius / Math.SQRT2;
      var xl = x - l, xr = x + l, yt = y - l, yb = y + l;
      ctx.moveTo(xl, yt);
      ctx.lineTo(xr, yt);
      ctx.lineTo(xr, yb);
      ctx.lineTo(xl, yb);
      ctx.lineTo(xl, yt);
    },
    triangleUp: function(ctx, x, y, radius) {
      var h = radius * Math.sin(Math.PI/6);
      var w = radius * Math.cos(Math.PI/6);
      var yb = y + h, yt = y - radius;
      ctx.moveTo(x, yt);
      ctx.lineTo(x + w, yb);
      ctx.lineTo(x - w, yb);
      ctx.lineTo(x, yt);
    },
    triangleDown: function(ctx, x, y, radius) {
      var h = radius * Math.sin(Math.PI/6);
      var w = radius * Math.cos(Math.PI/6);
      var yb = y + radius, yt = y - h;
      ctx.moveTo(x, yb);
      ctx.lineTo(x + w, yt);
      ctx.lineTo(x - w, yt);
      ctx.lineTo(x, yb);
    },
    diamond: function(ctx, x, y, radius) {
      var xl = x - radius, xr = x + radius, yt = y - radius, yb = y + radius;
      ctx.moveTo(x, yt);
      ctx.lineTo(xr, y);
      ctx.lineTo(x, yb);
      ctx.lineTo(xl, y);
      ctx.lineTo(x, yt);
    },
    halfCircleRight: function(ctx, x, y, radius) {
      ctx.arc(x, y, radius, Math.PI / 2, -Math.PI / 2, true);
      ctx.closePath();
    },
    halfCircleLeft: function(ctx, x, y, radius) {
      ctx.arc(x, y, radius, -Math.PI / 2, Math.PI / 2, true);
      ctx.closePath();
    },
    bar: function(ctx, x, y, radius) {
      var l = radius * 1.25;
      var w = radius * 0.4;
      var xl = x - w, xr = x + w, yt = y - l, yb = y + l;
      ctx.moveTo(xl, yt);
      ctx.lineTo(xr, yt);
      ctx.lineTo(xr, yb);
      ctx.lineTo(xl, yb);
      ctx.lineTo(xl, yt);
    }
    
  };
  var markerShapes = [ "circle", "diamond", "triangleUp", "triangleDown", "square", "bar", "halfCircleLeft", "halfCircleRight" ];

  function drawSeriesMarkers(ctx, series, plotOffset) {

    function plotPoints(datapoints, shape, radius, fillStyle, offset, axisx, axisy) {
      var points = datapoints.points,
          ps = datapoints.pointsize,
          drawShape = shapes[shape];

      for (var i = 0; i < points.length; i += ps) {
        var x = points[i], y = points[i + 1];
        if (x === null || x < axisx.min || x > axisx.max || y < axisy.min || y > axisy.max) {
            continue;
        }

        ctx.beginPath();
        drawShape(ctx, axisx.p2c(x), axisy.p2c(y) + offset, radius);
        if (fillStyle) {
          ctx.fillStyle = fillStyle;
          ctx.fill();
        }
        ctx.stroke();
      }
    }
      
    ctx.save();
    ctx.translate(plotOffset.left, plotOffset.top);

    var w;
    var lw = series.lines.lineWidth,
        sw = series.shadowSize,
        radius = series.markers.radius,
        shape = series.markers.shape;
    if (lw > 0 && sw > 0) {
      // draw shadow in two steps
      w = sw / 2;
      ctx.lineWidth = w;
      ctx.strokeStyle = "rgba(0,0,0,0.1)";
      plotPoints(series.datapoints, shape, radius, null, w + w/2,
                 series.xaxis, series.yaxis);

      ctx.strokeStyle = "rgba(0,0,0,0.2)";
      plotPoints(series.datapoints, shape, radius, null, w/2,
                 series.xaxis, series.yaxis);
    }

    ctx.lineWidth = lw;
    ctx.strokeStyle = series.color;
    var fill = series.markers.fill;
    var c = $.color.parse(fill ? series.markers.fillColor : series.color);
    plotPoints(series.datapoints, shape, radius, c.toString(), 0, series.xaxis, series.yaxis);
    ctx.restore();
  }

  
  function draw(plot, ctx) {
    var plotOffset = plot.getPlotOffset();
    var mi = 0;
    var mf = true;

    $.each(plot.getData(), function (i, series) {
      if (series.markers && series.markers.show) {
        if (series.markers.fill === undefined || series.markers.fill === null) {
          series.markers.fill = mf;
        }
        if (!series.markers.shape) {
          series.markers.shape = markerShapes[mi];
          mi += 1;
          if (mi >= markerShapes.length) {
            mi = 0;
            mf = !mf;
          }
        }
        drawSeriesMarkers(ctx, series, plotOffset);
      }
    });
  }

  function init(plot) {
    plot.hooks.draw.push(draw);
  }

  $.plot.plugins.push({
    init: init,
    options: options,
    name: 'markers',
    version: '1.0'
  });

  $.plot.markers = {
    draw: function(ctx, shape, x, y, radius, lineWidth, shadowWidth, colorStyle, fillStyle) {
      var w;

      function drawShape(x, y, fillStyle) {
        ctx.beginPath();
        shapes[shape](ctx, x, y, radius);
        if (fillStyle) {
          ctx.fillStyle = fillStyle;
          ctx.fill();
        }
        ctx.stroke();
      }

      if (lineWidth > 0 && shadowWidth > 0) {
        // draw shadow in two steps
        w = shadowWidth / 2;
        ctx.lineWidth = w;
        ctx.strokeStyle = "rgba(0,0,0,0.1)";
        drawShape(x, y + w + w/2, null);
        ctx.strokeStyle = "rgba(0,0,0,0.2)";
        drawShape(x, y + w/2, null);
      }
      ctx.lineWidth = lineWidth;
      ctx.strokeStyle = colorStyle;
      drawShape(x, y, fillStyle);
    }
  };
    
})(jQuery);
