/*!
  UI widget for a filtered list control. 
  Copyright (c) 2009, 2010, Oracle and/or its affiliates. All rights reserved.
*/
/*
  This control is currently specific to the needs of dashboard.
  Shortcuts were taken. This should be made up of a fancy list control or
  filtering should be part of a fancy list control. For now when label is null then 
  no filtering is done and it works like a somewhat fancy list control.
*/
/*global jQuery,mvc */
(function($) {

var VK_TAB = 9,
    VK_SPACE = 32,
    VK_PAGE_UP = 33,
    VK_PAGE_DOWN = 34,
    VK_END = 35,
    VK_HOME = 36,
    VK_LEFT = 37,
    VK_UP = 38,
    VK_RIGHT = 39,
    VK_DOWN = 40;

var FILTER_DELAY = 700;

var BUNDLE = "mvccore";

var DATA_LAST_FOCUSED = "lastFocused";
var DATA_FILTER = "filter";
var DATA_DRAG_INIT = "dragInit";

var CLASS_FILTERED_LIST = "filteredList";

var EVENT_CHANGE = "change";

var clsFocused = "focused";
var clsSelected = "selected";

var attrAriaSelected = "aria-selected";

$.widget("ui.filteredList", {

  _init: function() {
    var self = this;
    var o = this.options;
    var $ctrl = this.element;
    var id = $ctrl.get(0).id;
    var timerID = null;
    var out = mvc.htmlBuilder();
    var filterId = id + "_l";
    var ul;

    function filterNow() {
      if (timerID) {
        clearTimeout(timerID);
        timerID = null;
      }
      self._filter();
    }

    if (o.label) {
      o.filterPrompt = o.filterPrompt || mvc.getMessage(BUNDLE, "filter.prompt");
      o.clearButtonTitle = o.clearButtonTitle || mvc.getMessage(BUNDLE, "filter.clear.button");

      out.markup("<label for='").attr(filterId).markup("'>").content(o.label).markup(":</label>").
        markup("<div class='filter'><input id='").attr(filterId).markup("' type='text' class='prompt'>");
      mvc.renderButton(out, {id: id + "_c", icon: o.clearButtonIcon, nb: true, tooltip: o.clearButtonTitle});
      out.markup("</div>");
    }
    out.markup("<div role='listbox' aria-multiselectable='false' class='select'>").
      markup("<ul></ul></div>");

    $ctrl.html(out.toString()).addClass(CLASS_FILTERED_LIST);

    this._setData(DATA_FILTER, "");
    if (o.label) {
      this._promptMode(o.initialFilter === "");
    }
    this._populateList();
    if (o.label && o.initialFilter) {
      $ctrl.find(".filter :text").val(o.initialFilter);
      this._filter();
    }
    this.resize();

    if (o.label) {
      $ctrl.find(".filter button").click(function(e) {
        // clear filter
        self._clearFilter();
        var lastFocused = self._getData(DATA_LAST_FOCUSED); 
        if (lastFocused) {
          mvc.setFocus(lastFocused, 0);
        }
      });
  
      $ctrl.find(".filter :text").focus(function(e) {
        if ($(this).hasClass("prompt")) {
          self._promptMode(false);
        }
      }).blur(function(e) {
        filterNow();
        if ($(this).val() === "") {
          self._promptMode(true);
        }
      }).keydown(function(e) {
        var $li;
        if (e.keyCode == VK_DOWN) {
          filterNow();
          $li = $ctrl.find(".select li:visible:first");
          if ($li.length > 0) {
            self._selectItem($li);
          }
          return false;
        } else if (e.keyCode == VK_TAB) {
          // better to do this here rather than in blur because
          // we may be disabling the next tabstop
          filterNow();
          if ($(this).val() === "") {
            self._promptMode(true);
          }
          return; // let normal tab handling happen
        }
        // else normal key
        if (!timerID) {
          timerID = setTimeout(function() {
            self._filter();
            timerID = null;
          }, FILTER_DELAY);
        }
      });
    }

    // list functionality
    ul = $ctrl.find(".select ul").bind("keydown", function(e) {
      var offset = null;
      var $options, $li;
      var cur, next, lh, page;

      if (e.altKey) {
        return;
      }
      var kc = e.keyCode;
      if (kc == VK_PAGE_UP || kc == VK_PAGE_DOWN) {
        lh = $(this).find("li:visible:first").outerHeight();
        if (!lh || lh <= 0) {
          lh = 24;
        }
        page = ($(this).parent().height() / lh).toFixed() - 1;
      }
      if (kc == VK_SPACE) {
        e.preventDefault(); // some browsers use space to scroll and we don't want that
        return;
      } else if (kc == VK_UP || kc == VK_LEFT) {
        offset = -1;
      } else if (kc == VK_DOWN || kc == VK_RIGHT) {
        offset = 1;
      } else if (kc == VK_PAGE_UP) {
        offset = -page;
      } else if (kc == VK_PAGE_DOWN) {
        offset = page;
      } else if (kc == VK_END) {
        $li = $(this).find("li:visible:last");
        self._selectItem($li);
        return false;
      } else if (kc == VK_HOME) {
        $li = $(this).find("li:visible:first");
        self._selectItem($li);
        return false;
      }
      if (offset !== null) {
        $options = $(this).find("li:visible");
        if ($options.length < 1) {
          return false; // empty list
        }
        cur = $options.index($options.filter(".selected"));
        if (cur >= 0) {
          next = cur + offset;
        } else {
          cur = $options.index(self._getData(DATA_LAST_FOCUSED));
          if (cur >= 0) {
            next = cur;
            cur = null;
          }
        }
        if (cur == -1) {
          next = 0;
        }
        if (next < 0) {
          next = 0;
        } else if (next >= $options.length) {
          next = $options.length - 1;
        }
        if (next != cur) {
          $li = $options.eq(next);
          self._selectItem($li);
        }
        return false;
      }
    }).mousedown(function(e) {
      // this is only for left mouse button
      if (e.which != 1) {
        return;
      }
      var $item = $(e.target).closest("li");
      if ($item.length < 1) {
        return; // mouse event not on an item. Perhaps on scrollbar
      }
      self._selectItem($item);
      return false;
    }).mouseup(function(e) {
      // keep IE8 from showing a selection
      if (document.selection) {
        document.selection.empty();
      }
    }).get(0);
    
    mvc.bindFocus(ul, function(e) {
      var $target = $(e.target).closest("li");
      var target = $target.get(0);
      var lastFocused = self._getData(DATA_LAST_FOCUSED);
      if (lastFocused && lastFocused !== target) {
        lastFocused.tabIndex = -1;
      }
      $target.addClass(clsFocused);
      target.tabIndex = 0;
      self._setData(DATA_LAST_FOCUSED, target);
    });
    mvc.bindBlur(ul, function(e) {
      var $target = $(e.target);
      $target.removeClass(clsFocused);
    });

  },

  update: function(list, clearFilter) {
    var $ctrl = this.element;
    var o = this.options;

    o.list = list;
    this._populateList();
    if (o.label) {
      if (clearFilter) {
        this._promptMode(true);
      } else {
        this._setData(DATA_FILTER, "");
        this._filter();
      }
    }
  },

  resize: function() {
    function heightLessPadding($e) {
      return $e.innerHeight() - parseInt($e.css("paddingTop"), 10) - parseInt($e.css("paddingBottom"), 10);
    }
    function widthLessPadding($e) {
      return $e.innerWidth() - parseInt($e.css("paddingLeft"), 10) - parseInt($e.css("paddingRight"), 10);
    }
    var $ctrl = this.element;
    var o = this.options;
    var h = heightLessPadding($ctrl);
    var w, $text;
    if (o.label) {
      $text = $ctrl.find(".filter :text");
      w = widthLessPadding($ctrl.find(".filter"));
      $text.width(w - $ctrl.find(".filter button").outerWidth() - parseInt($text.css("paddingLeft"), 10) - parseInt($text.css("paddingRight"), 10) );
      h -= $ctrl.find(".filter").outerHeight(true) + $ctrl.find("label:first").outerHeight(true);
    }
    $ctrl.find(".select").height(h);
  },

  selection: function() {
    var $ctrl = this.element;
    var $item = $ctrl.find(".select li.selected");

    if ($item.length > 0) {
      return $item.find("input").val();
    }
    return null;
  },

  select: function(value, noFocus) {
    var $ctrl = this.element;
    var $item = null;

    $ctrl.find(".select li").each(function() {
      var curVal = $(this).find("input:first").val();
      if (curVal == value) {
        $item = $(this);
        return false;
      }
    });
    if ($item) {
      if ($item.css("display") == "none") {
        this._clearFilter();
      }
      // TODO if not giving it focus should at least scroll into view
      this._selectItem($item, noFocus);
      return $item;
    }
    return null;
  },

  destroy: function() {
    $(this.element).html("").removeClass(CLASS_FILTERED_LIST);
    $.widget.prototype.destroy.call(this);
  },

  _clearFilter: function() {
    this._promptMode(true);
    this._filter();
  },

  _promptMode: function(prompt) {
    var $ctrl = this.element;
    var o = this.options;

    if (prompt) {
      $ctrl.find(":text").addClass("prompt").val(o.filterPrompt);
      $ctrl.find("button").addClass("disabled").get(0).disabled = true;
    } else {
      $ctrl.find(":text").removeClass("prompt").val(this._getData(DATA_FILTER));
      $ctrl.find("button").removeClass("disabled").get(0).disabled = false;
    }
  },

  _selectItem: function($item, noFocus) {
    var $prev = this.element.find("ul li.selected"); 
    if ($prev.get(0) !== $item.get(0)) {
      $prev.removeClass(clsSelected).attr(attrAriaSelected, "false");
      $item.addClass(clsSelected).attr(attrAriaSelected, "true");
      this._trigger(EVENT_CHANGE, 0);
    }
    if (noFocus !== true) {
      mvc.setFocus($item.get(0), 0);
    }
  },

  _populateList: function() {
    var self = this;
    var $ctrl = this.element;
    var o = this.options;
    var adapter = o.itemAdapter;
    var list = o.list;
    var out = mvc.htmlBuilder();
    var i, item;

    this._setData(DATA_DRAG_INIT, false);

    for (i = 0; i < list.length; i++) {
      item = list[i];
      out.markup("<li role='option' tabindex='-1'").optionalAttr("class", o.itemLabel).markup(">");
      out.markup("<input type='hidden' value='").attr(adapter.value(item)).markup("'>");
      adapter.renderLabel(out, item);
      out.markup("</li>");
    }

    $ctrl.find(".select ul").html(out.toString()).find("li:first").each(function() { // at most 1
      this.tabIndex = 0;
      self._setData(DATA_LAST_FOCUSED, this);
    });
    this._enableDrag();
  },

  _filter: function() {
    var self = this;
    var $ctrl = this.element;
    var o = this.options;
    var filter, lastFilter, lcFilter;
    var $text = $ctrl.find(":text");
    var lostTabStop = false;
    var lostSelection = false;

    if ($ctrl.find(".select li:visible").length === 0) {
      lostTabStop = true;
    }

    lastFilter = this._getData(DATA_FILTER);
    if ($text.hasClass("prompt")) {
      filter = "";
    } else {
      filter = $.trim($text.val());
    }
    lcFilter = filter.toLowerCase();
    if (lastFilter != filter) {
      this._setData(DATA_FILTER, filter);

      $ctrl.find(".select li").each(function() {
        var $label = $(this).find(".label");
        var text = $label.text().toLowerCase();
        var text2 = null;
        if (o.filterOnTitle) {
          text2 = $label.attr("title").toLowerCase();
        }
        var match = text.indexOf(lcFilter) >= 0 || (text2 !== null && text2.indexOf(lcFilter) >= 0);
        $(this).toggle(match);
        if (!match) {
          if ($(this).hasClass(clsSelected)) {
            $(this).removeClass(clsSelected);
            lostSelection = true;
          }
          if (this.tabIndex === 0) {
            this.tabIndex = -1;
            lostTabStop = true;
          }
        }
      });
      if (lostTabStop) {
        $ctrl.find(".select li:visible:first").each(function() {
          this.tabIndex = 0;
          self._setData(DATA_LAST_FOCUSED, this);
        });
      }
      if (lostSelection) {
        this._trigger(EVENT_CHANGE, 0);
      }
    }
  },

  _enableDrag: function() {
    var $ctrl = this.element;
    var o = this.options;
    var $items;
    var dragInit = this._getData(DATA_DRAG_INIT);

    if (!o.draggable) {
      return;
    }
    if (o.enableDrag) {
      if (dragInit) {
        $items = $ctrl.find(".select ul li").draggable("enable");
      } else {
        $items = $ctrl.find(".select ul li").draggable(o.draggable);
        this._setData(DATA_DRAG_INIT, true);
      }
      $items.removeClass("ui-state-disabled");
    } else {
      $items = $ctrl.find(".select ul li").addClass("ui-state-disabled");
      if (dragInit) {
        $items.draggable("disable");
      }
    }
  },

  _setData: function(key, value) {
    $.widget.prototype._setData.call(this, key, value);
    if (key == 'enableDrag') {
      this._enableDrag();
    }
  }

});

/*
 * Default item adapter assumes items like { v: <value>, l: <label>, h:<help-title> }
 */
var defaultAdapter = {
  value: function(item) {
    return item.v;
  },
  renderLabel: function(out, item) {
    out.markup("<span class='label'").optionalAttr("title", item.h).markup(">").
      content(item.l).markup("</span>");
  }
};

$.extend($.ui.filteredList, {
  version: "1.7",
  getter: "selection",
  defaults: {
    label: null, // if null then no filtering
    filterPrompt: null, // defaults to message filter.prompt
    filterOnTitle: true, // if false only match filter on label text (text of element with class label), if true filter on both title and label text
    clearButtonIcon: "clearFilter",
    clearButtonTitle: null, // defaults to message filter.clear.button
    initialFilter: "",
    enableDrag: false,
    draggable: null, // draggable options to make list items draggable or null for no drag support?
    list: [], // list of items
    itemLabel: null,
    itemAdapter: defaultAdapter // { value: fn(item), renderLabel: fn(out, item) } // rendered label html must have a class of "label" 
  }
});

})(jQuery);
