/*!
  UI widget for dashboard view
  Copyright (c) 2009, 2010, Oracle and/or its affiliates. All rights reserved.
*/
/*global jQuery,mvc */
(function($) {

var VK_LEFT = 37,
    VK_UP = 38,
    VK_RIGHT = 39,
    VK_DOWN = 40;

var BUNDLE = "dashboard";

var EVENT_STATE_CHANGE = "statechange";

var CLASS_DASHBOARD_VIEW = "dashboardView";

function makePanelSelector(r, c) {
  return ".dashRow:eq(" + r + ") .dashPanel:eq(" + c + ")";
}

function makeRowSelector(r) {
  return ".dashRow:eq(" + r + ")";
}

function getViewMenu(id) {
  return $("#" + id + "ViewMenu");
}

function getRect($e) {
  var r = $e.offset();
  r.right = r.left + $e.width();
  r.bottom = r.top + $e.height();
  return r;
}

function isVisible($p, rc) {
  // panel $p is visible when it intersects container rectangle rc
  var rp = getRect($p);
  if (rp.left > rc.right || rp.right < rc.left || rp.top > rc.bottom || rp.bottom < rc.top) {
    return false;
  }
  return true;
}

$.widget("ui.dashboardView", {

  _init: function() {
    var self = this;
    var o = this.options;
    var $ctrl = this.element;
    var $scroller, scrollRect;
    var id = $ctrl.get(0).id;
    id = id ? id : "";
    var out = mvc.htmlBuilder();
    var state = o.state;
    var rows, r;
    var scrollTimerId = null;

    o.panelPosMap = {}; // map of dashPanel id to {r: <n>, c: <n>, <other book keeping>}
    o.nextPanelId = 0;
    o.lastCol = 0;

    rows = state.panels;
    out.markup("<div class='dashViewHeader'><h2>").content(state.name).markup("</h2><div class='headerBtns'>");
    mvc.renderButton(out, {id: id + "Menu", icon: "menu", nb: true, tooltip: mvc.getMessage(BUNDLE, "view.menu")});
    out.markup("</div></div><div class='dashPanelContainer' style='position:relative;overflow:auto;'><ul class='panels'>");
    for (r = 0; r < rows.length; r++) {
      this._renderRow(out, r);
    }
    out.markup("</ul></div>");
    $ctrl.html(out.toString()).addClass(CLASS_DASHBOARD_VIEW);
    if (state.activeState === undefined) {
      state.activeState = false;
    }
    this.resize(true);

    out.clear();
    out.markup("<div id='").attr(id + "ViewMenu").markup("'></div>");
    $(out.toString()).appendTo("body").hide();
    this._initMenu();

    $scroller = $ctrl.find(".dashPanelContainer");
    scrollRect = getRect($scroller);
    this._forEachPanel(function(p, $p) {
      if (isVisible($p, scrollRect)) {
        self._initDashPanel($p, p);
      } else {
        o.panelPosMap[$p.get(0).id].initNeeded = true;
      }
    });
    this.resize();

    $("#" + id + "Menu").click(function() {
      var $menu = getViewMenu(id);
      self._updateMenu($menu);
      $menu.menu("toggle", $(this));
    });

    $ctrl.bind("click.dashboardView", function(e) {
      var $target = $(e.target);
      var tabindex = $target.attr("tabindex") || $target.parent().attr("tabindex");

      // find the panel that contains the click
      var $panel = $target.closest(".dashPanel").eq(0);
      if ($panel.length > 0) {
        o.lastCol = o.panelPosMap[$panel.get(0).id].c;
        self._select($panel, tabindex === undefined);
      }
    });
    this._select($ctrl.find(".dashPanel:first"));

    $ctrl.bind("keydown.dashboardView", function(e) {
      var kc;
      var $cur, $next, next, pos, panel, w, h, t, b, l, r, sh, sw, scroller;
      var setCol = false;
      if (e.altKey || e.ctrlKey || $(e.target).is("input, select, textarea")) {
        return;
      }
      kc = e.keyCode;
      $cur = $ctrl.find(".dashPanel.selected");
      if (kc == VK_LEFT) {
        $next = $cur.prev(".dashPanel");
        if ($next.length === 0) {
          $next = $cur.parents(".dashRow").prev().find(".dashPanel:last");
        }
        if ($next.length === 0) {
          $next = $ctrl.find(".dashPanel:last");
        }
        setCol = true;
      } else if (kc == VK_RIGHT) {
        $next = $cur.next(".dashPanel");
        if ($next.length === 0) {
          $next = $cur.parents(".dashRow").next().find(".dashPanel:first");
        }
        if ($next.length === 0) {
          $next = $ctrl.find(".dashPanel:first");
        }
        setCol = true;
      } else if (kc == VK_UP) {
        $next = $cur.parents(".dashRow").prev();
        if ($next.length === 0) {
          $next = $ctrl.find(".dashRow:last");
        }
        $next = $next.find(".dashPanel");
        $next = $next.eq(Math.min(o.lastCol, $next.length - 1)); // stay in same col if possible
      } else if (kc == VK_DOWN) {
        $next = $cur.parents(".dashRow").next();
        if ($next.length === 0) {
          $next = $ctrl.find(".dashRow:first");
        }
        $next = $next.find(".dashPanel");
        $next = $next.eq(Math.min(o.lastCol, $next.length - 1)); // stay in same col if possible
      }
      if ($next && $next.length > 0) {
        pos = o.panelPosMap[$next.get(0).id];
        if (setCol) {
          o.lastCol = pos.c;
        }
        if (pos.initNeeded === true) {
          pos.initNeeded = false;
          panel = state.panels[pos.r][pos.c];
          self._initDashPanel($next, panel);
        }
        $cur.chart("unSelect");
        $next.chart("select", true);
        // scroll into view
        next = $next.get(0);
        scroller = $scroller.get(0);
        w = $next.width() + o.panelSpacing;
        h = $next.height() + o.panelSpacing;
        t = next.offsetTop - (o.panelSpacing / 2);
        b = t + h;
        l = next.offsetLeft - (o.panelSpacing / 2);
        r = l + w;
        sh = $scroller.get(0).clientHeight;
        sw = $scroller.get(0).clientWidth;
        if (h > sh || t < $scroller.scrollTop()) {
          scroller.scrollTop = t;
        } else if (b > $scroller.scrollTop() + sh) {
          scroller.scrollTop = t - Math.max(sh - h, 0);
        }
        if (w > sw || l < $scroller.scrollLeft()) {
          scroller.scrollLeft = l;
        } else if (r > $scroller.scrollLeft() + sw) {
          scroller.scrollLeft = l - Math.max(sw - w, 0);
        }
        return false;
      }
    });

    $scroller.scroll(function() {
      // delay action because some browsers fire to often
      if (scrollTimerId === null) {
        scrollTimerId = setTimeout(function() {
          scrollTimerId = null;
          self._initOrUpdatePanels();
        }, 10);
      }
    });

    if (!state.fromTemplate) {
      // d&d
      if (o.dropAccept) {
        $ctrl.droppable({
          accept: o.dropAccept,
          addClasses: false,
          hoverClass: "dropActive",
          drop: function(e, ui) {
            var $panel = self._newPanel("chart");
            $panel.chart("addMetricFromDraggable", ui.draggable);
          }
        });
      }
    }

    this._initRows($ctrl.find(".dashRow > ul"));
  },

  add: function(panel) {
    var $ctrl = this.element;
    var o = this.options;
    var panels = o.state.panels;
    var pos;
    var $panel, $row;
    var out = mvc.htmlBuilder();

    $panel = $ctrl.find(".dashPanel.selected");
    if ($panel.length > 0) {
      pos = $.extend({}, this._getPanelPosition($panel)); // make a copy
      // Insert new row after
      pos.r += 1;
      pos.c = 0;
      panels.splice(pos.r, 0, []);
    } else {
      // insert as new last row
      panels.push([]);
      pos = {r: panels.length - 1, c: 0};
    }
    panels[pos.r].splice(pos.c, 0, panel);
    this._renderRow(out, pos.r);
    if ($panel.length > 0) {
      $panel.parents(".dashRow").eq(0).after(out.toString());
    } else {
      $ctrl.find(".panels").append(out.toString());
    }
    $row = $ctrl.find(makeRowSelector(pos.r) + " > ul");
    this._initRows($row);
    this._reindexAll();
    $panel = $ctrl.find(makePanelSelector(pos.r, pos.c));
    this._initDashPanel($panel, panels[pos.r][pos.c]);
    this.resize();

    this._select($panel, true);

    this._trigger(EVENT_STATE_CHANGE, 0);
    return $panel;
  },

  remove: function($panel) {
    var $ctrl = this.element;
    var o = this.options;
    var panels = o.state.panels;
    var pos, selPos;
    var $row;
    var panel;

    $panel = $panel || $ctrl.find(".dashPanel.selected");
    if ($panel.length > 0) {
      pos = this._getPanelPosition($panel);
      selPos = $.extend({}, pos);
      panel = panels[pos.r][pos.c];

      // remove UI panel
      this._destroyDashPanel($panel, panel);
      $row = $panel.parent();
      delete o.panelPosMap[$panel.get(0).id];
      $panel.remove();
      // remove panel state
      panels[pos.r].splice(pos.c, 1);
      if (panels[pos.r].length === 0) {
        // remove row
        panels.splice(pos.r, 1);

        if (selPos.r >= panels.length) {
          selPos.r = panels.length - 1;
        }

        // remove view dashRow
        $row.parent().remove();
        this._reindexAll();
      } else {
        this._reindexRows([pos.r]);
      }
      this.resize();
      // after remove select panel in position removed adjusted so not off the end of row or col
      if (selPos.r >= 0) {
        if (selPos.c >= panels[selPos.r].length) {
          selPos.c = panels[pos.r].length - 1;
        }
        this._select(this.getPanel(selPos.r, selPos.c), true);
      }

      this._trigger(EVENT_STATE_CHANGE, 0);
    }
  },

  // returns array of {name: "name", r: <n>, c: <n>}.
  getPanels: function() {
    var self = this;
    var $ctrl = this.element;
    var o = this.options;
    var adapter = o.adapter;
    var panels = o.state.panels;
    var result = [];

    $ctrl.find(".dashPanel").each(function() {
      var pos = self._getPanelPosition($(this));
      var panel = panels[pos.r][pos.c];
      var name = adapter.getName(panel.state);
      result.push({name: name, r: pos.r, c: pos.c});
    });
    return result;
  },

  getPanel: function(r, c) {
    var $ctrl = this.element;
    return $ctrl.find(makePanelSelector(r, c));
  },

  start: function() {
    var $ctrl = this.element;
    var state = this.options.state;
    if (!state.activeState) {
      this._forEachPanel(function(p, $p) {
        if (p.type == "chart") {
          $.ui.chart.start(p.state, state.id);
        }
      });
      state.activeState = true;
    }
  },

  stop: function() {
    var $ctrl = this.element;
    var state = this.options.state;
    if (state.activeState) {
      this._forEachPanel(function(p, $p) {
        if (p.type == "chart") {
          $.ui.chart.stop(p.state, state.id);
        }
      });
      state.activeState = false;
    }
  },

  isActive: function() {
    return this.options.state.activeState;
  },

  resize: function(init) {
    var $ctrl = this.element;
    var o = this.options;
    var state = o.state;
    var viewH, viewW, rowH, panelW, panelH, maxW, h, w, p1, p2;
    var rows, cols, roff, i, j;
    var $panels, $panel, $rows;
    var $container = $ctrl.find(".dashPanelContainer");
    var $ul = $container.children("ul").eq(0);
    var spacing;
    var tall = false;

    // All panels have to (are assumed to) have the same border/margins on all sides and it doesn't change during life of this widget
    if (!o.panelSpacing) {
      $panel = $ctrl.find(".dashPanel:first");
      if ($panel.length > 0) {
        o.panelSpacing = (parseInt($panel.css("margin-top"), 10) + parseInt($panel.css("border-top-width"), 10)) * 2;
      }
    }
    spacing = o.panelSpacing || 12;

    if (!init) {
      // get min panel size TODO cache and only update if there is a change
      this._forEachPanel(function(p, $p) {
        if (!$p.hasClass("chart")) {
          return;
        }
        var minSize = $p.chart("getMinSize");
        minSize.h += spacing;
        minSize.w += spacing;
        if (minSize.h > o.minPanelHeight) {
          o.minPanelHeight = minSize.h;
        }
        if (minSize.w > o.minPanelWidth) {
          o.minPanelWidth = minSize.w;
        }
      });
    }
    rows = state.panels;
    p1 = $ctrl.position();
    p2 = $container.position();
    viewH = $ctrl.height() - (p2.top - p1.top) - 2; // -2 is to leave room for outline
    viewW = $ctrl.width() - 2;
    $container.css({width: viewW, height: viewH});
    rowH = viewH / rows.length;
    if (rowH < o.minPanelHeight) {
      rowH = o.minPanelHeight;
      $ul.height(rowH * rows.length);
    }
    if (rowH * rows.length > viewH) {
      viewW -= 20; // make room for V scrollbar
      tall = true;
    }

    maxW = viewW;
    for (i = 0; i < rows.length; i++) {
      cols = rows[i];
      w = o.minPanelWidth * cols.length;
      if (w > maxW) {
        maxW = w;
      }
    }
    if (maxW > viewW && !tall) {
      rowH -= 20 / rows.length; // make room for H scrollbar
      if (rowH < o.minPanelHeight) {
        rowH = o.minPanelHeight;
        $ul.height(rowH * rows.length);
      }
    }
    this._initOrUpdatePanels(); // new panels could become visible on resize

    panelH = rowH - spacing;
    $panels = $ctrl.find(".dashPanel");
    $rows = $ctrl.find(".dashRow > ul");
    roff = 0;
    for (i = 0; i < rows.length; i++) {
      cols = rows[i];
      $rows.eq(i).height(rowH);
      panelW = Math.floor(maxW / cols.length);
      panelW -= spacing;
      for (j = 0; j < cols.length; j++) {
        $panel = $panels.eq(roff + j);
        h = $panel.height();
        w = $panel.width();
        if (h != panelH || w != panelW) {
          $panel.css({height: panelH, width: panelW});
          if (!init) {
            if ($panel.hasClass("chart")) {
              $panel.chart("resize");
            }
          }
        }
      }
      roff += cols.length;
    }
    $ul.width(maxW);

  },

  update: function() {
    var o = this.options;
    var adapter = o.adapter;
    var scrollerRect = getRect(this.element.find(".dashPanelContainer"));

    this._forEachPanel(function(p, $p) {
      if (isVisible($p, scrollerRect)) {
        adapter.updatePanel($p, p.type, p.state);
      } else {
        o.panelPosMap[$p.get(0).id].updateNeeded = true;
      }
    });
  },

  destroy: function() {
    var $ctrl = this.element;
    var id = $ctrl.get(0).id;
    id = id ? id : "";

    $ctrl.unbind(".dashboardView");
    $ctrl.droppable("destroy");
    $ctrl.html("").removeClass(CLASS_DASHBOARD_VIEW);
    getViewMenu(id).remove();
    $.widget.prototype.destroy.call(this);
  },

  _initOrUpdatePanels: function() {
    var self = this;
    var o = this.options;
    var adapter = o.adapter;
    var $scroller = this.element.find(".dashPanelContainer");
    var scrollRect = getRect($scroller);

    this._forEachPanel(function(p, $p) {
      var id = $p.get(0).id;
      var init = o.panelPosMap[id].initNeeded === true;
      var update = o.panelPosMap[id].updateNeeded === true;

      if ((init || update) && isVisible($p, scrollRect)) {
        if (init) {
          o.panelPosMap[id].initNeeded = false;
          self._initDashPanel($p, p);
        }
        if (update) {
          o.panelPosMap[id].updateNeeded = false;
          adapter.updatePanel($p, p.type, p.state);
        }
      }
    });
  },

  _select: function($panel, setfocus) {
    var $ctrl = this.element;
    if ($panel.length > 0 && !$panel.hasClass("selected")) {
      $ctrl.find(".dashPanel.selected").chart("unSelect");
      $panel.chart("select", setfocus);
    }
  },

  _genChartName: function(name) {
    var adapter = this.options.adapter;
    var newName = name;
    var count = 0;
    var match, i, len, node;

    function matchName(panel) {
      if (adapter.getName(panel.state) == newName) {
        match = true;
        newName = name + " " + count;
        return false;
      }
    }

    do {
      count += 1;
      match = false;
      this._forEachPanel(matchName);
    } while (match);
    return newName;
  },

  _newPanel: function(type) {
    var adapter = this.options.adapter;
    var name = this._genChartName(mvc.getMessage(BUNDLE,"view.newChartName"));
    return this.add({type: type, state: adapter.newPanelState(type, name)});
  },

  _initDashPanel: function($e, panel) {
    var self = this;
    var adapter = this.options.adapter;
    adapter.initPanel($e, panel.type, panel.state, {
      getPanels: function() {
        return self.getPanels();
      },
      getPanel: function(r, c) {
        return self.getPanel(r, c);
      },
      getActiveView: function() {
        var state = self.options.state;
        if (state.activeState) {
          return state.id;
        }
        return null;
      }
    });
  },

  _destroyDashPanel: function($e, panel) {
    var $ctrl = this.element;
    var o = this.options;
    var active = o.state.activeState;

    if (panel.type == "chart") {
      if (active) {
        $e.chart("stop", o.state.name);
      }
      $e.chart("destroy");
    }
  },

  // $row is the ul under the dashRow, size is optional
  _resizeRow: function($row, size) {
    var $ctrl = this.element;
    var o = this.options;
    var panelW, maxW, w;
    var $ul = $ctrl.children(".dashPanelContainer > ul").eq(0);
    var $panels = $row.children(".dashPanel");

    size = size || $panels.length;
    maxW = $row.width();
    panelW = maxW / size;
    if (panelW < o.minPanelWidth) {
      panelW = o.minPanelWidth;
      w = panelW * size;
      if (w > maxW) {
        maxW = w;
      }
    }
    panelW -= o.panelSpacing || 12;

    $panels.each(function() {
      $(this).width(panelW).chart("resize");
    });
    $ul.width(maxW);
    return panelW;
  },

  _moveUp: function() {
    var $ctrl = this.element;
    var panels = this.options.state.panels;
    var fromPos, toPos, $panel, $row;

    $panel = $ctrl.find(".dashPanel.selected");
    if ($panel.length > 0) {
      fromPos = this._getPanelPosition($panel);
      if (fromPos.r === 0) {
        return;
      }
      toPos = $.extend({}, fromPos);
      toPos.r -= 1;
      $row = $ctrl.find(makeRowSelector(toPos.r) + " > ul");
      if (toPos.c >= panels[toPos.r].length) {
        toPos.c = panels[toPos.r].length - 1;
        $row.append($panel);
      } else {
        $row.find(".dashPanel").eq(toPos.c).before($panel);
      }
      this._moved(fromPos, toPos);
    }
  },

  _moveDown: function() {
    var $ctrl = this.element;
    var panels = this.options.state.panels;
    var fromPos, toPos, $panel, $row;

    $panel = $ctrl.find(".dashPanel.selected");
    if ($panel.length > 0) {
      fromPos = this._getPanelPosition($panel);
      if (fromPos.r == panels.length - 1) {
        return;
      }
      toPos = $.extend({}, fromPos);
      toPos.r += 1;
      $row = $ctrl.find(makeRowSelector(toPos.r) + " > ul");
      if (toPos.c >= panels[toPos.r].length) {
        toPos.c = panels[toPos.r].length - 1;
        $row.append($panel);
      } else {
        $row.find(".dashPanel").eq(toPos.c).before($panel);
      }
      this._moved(fromPos, toPos);
    }
  },

  _moveLeft: function() {
    var $ctrl = this.element;
    var panels = this.options.state.panels;
    var fromPos, toPos, $panel, $row;

    $panel = $ctrl.find(".dashPanel.selected");
    if ($panel.length > 0) {
      fromPos = this._getPanelPosition($panel);
      if (fromPos.c === 0) {
        return;
      }
      toPos = $.extend({}, fromPos);
      toPos.c -= 1;
      $row = $ctrl.find(makeRowSelector(toPos.r) + " > ul");
      $row.find(".dashPanel").eq(toPos.c).before($panel);
      this._moved(fromPos, toPos);
    }
  },

  _moveRight: function() {
    var $ctrl = this.element;
    var panels = this.options.state.panels;
    var fromPos, toPos, $panel, $row;

    $panel = $ctrl.find(".dashPanel.selected");
    if ($panel.length > 0) {
      fromPos = this._getPanelPosition($panel);
      if (fromPos.c >= panels[fromPos.r].length - 1) {
        return;
      }
      toPos = $.extend({}, fromPos);
      toPos.c += 1;
      $row = $ctrl.find(makeRowSelector(toPos.r) + " > ul");
      $row.find(".dashPanel").eq(toPos.c).after($panel);
      this._moved(fromPos, toPos);
    }
  },

  _moveAbove: function() {
    var $ctrl = this.element;
    var panels = this.options.state.panels;
    var fromPos, toPos, $panel, $row;
    var out = mvc.htmlBuilder();

    $panel = $ctrl.find(".dashPanel.selected");
    if ($panel.length > 0) {
      fromPos = $.extend({}, this._getPanelPosition($panel));
      toPos = $.extend({}, fromPos);
      toPos.c = 0;
      fromPos.r += 1;
      panels.splice(toPos.r, 0, []);

      this._renderRow(out, toPos.r);
      $row = $ctrl.find(makeRowSelector(toPos.r));
      $row.before(out.toString());
      $row = $ctrl.find(makeRowSelector(toPos.r) + " > ul");
      this._initRows($row);
      $row.append($panel);
      this._moved(fromPos, toPos);
    }
  },

  _moveBelow: function() {
    var $ctrl = this.element;
    var panels = this.options.state.panels;
    var fromPos, toPos, $panel, $row;
    var out = mvc.htmlBuilder();

    $panel = $ctrl.find(".dashPanel.selected");
    if ($panel.length > 0) {
      fromPos = this._getPanelPosition($panel);
      toPos = $.extend({}, fromPos);
      toPos.c = 0;
      toPos.r += 1;
      panels.splice(toPos.r, 0, []);

      this._renderRow(out, toPos.r);
      $row = $ctrl.find(makeRowSelector(fromPos.r));
      $row.after(out.toString());
      $row = $ctrl.find(makeRowSelector(toPos.r) + " > ul");
      this._initRows($row);
      $row.append($panel);
      this._moved(fromPos, toPos);
    }
  },

  // a panel moved
  _moved: function(fromPos, toPos) {
    var $ctrl = this.element;
    var o = this.options;
    var panels = o.state.panels;
    var rows, p, i, $row;

    p = panels[fromPos.r][fromPos.c];
    panels[fromPos.r].splice(fromPos.c,1);
    panels[toPos.r].splice(toPos.c, 0, p);
    if (panels[fromPos.r].length === 0) {
      // remove row
      panels.splice(fromPos.r, 1);
      // remove view dashRow
      $(makeRowSelector(fromPos.r)).remove();
      this._reindexAll();
    } else if (panels[toPos.r].length == 1) {
      this._reindexAll();
    } else {
      rows = [fromPos.r];
      if (fromPos.r != toPos.r) {
        rows.push(toPos.r);
      }
      this._reindexRows(rows);
    }
    this.resize();

    this._trigger(EVENT_STATE_CHANGE, 0);
  },

  _renderRow: function(out, r) {
    var $ctrl = this.element;
    var id = $ctrl.get(0).id || "";
    var o = this.options;
    var rows = o.state.panels;
    var cols, c, nextId;

    out.markup("<li class='dashRow'><ul>");
    cols = rows[r];
    for (c = 0; c < cols.length; c++) {
      nextId = id + "_" + o.nextPanelId;
      o.panelPosMap[nextId] = {r: r, c: c};
      o.nextPanelId += 1;

      out.markup("<li class='dashPanel' id='").attr(nextId).markup("'></li>");
    }
    out.markup("</ul></li>");
  },

  // $rows is the ul element under one or more .dashRow elements
  _initRows: function($rows) {
    var $ctrl = this.element;
    var self = this;

    $rows.sortable({
      connectWith: ".dashRow ul",
      handle: ".dashPanelHeader",
      cursor: "move",
      placeholder: "dashPlaceholder",
      forcePlaceholderSize: true,
      helper: function(e, item) {
        var h = $(item).clone().children(":not(.dashPanelHeader)").remove().end();
        return h.get(0);
      },
      opacity: 0.7,
      tolerance: "pointer",
      over: function(e, ui) {
        var $row = $(this);
        var w;
        if (this != ui.sender.get(0)) {
          w = self._resizeRow($row, $row.children(".dashPanel").length + 1);
          // resize the item also to fit in this row
          ui.helper.width(w);
          ui.placeholder.width(w);
        }
      },
      out: function(e, ui) {
        var $row = $(this);
        var w;
        if (this != ui.sender.get(0) && ui.helper) {
          w = self._resizeRow($row);
        }
      },
      update: function(e, ui) {
        var toPos = {r: $ctrl.find(".dashRow").index($(this).parent()),
          c: $(this).children(".dashPanel").index(ui.item)};
        var fromPos;
  
        if (toPos.c == -1) {
          return; // this is an update before remove - wait for the update after receive
        }
        fromPos = self._getPanelPosition(ui.item);
        self._moved(fromPos, toPos);
      }
    });
  },

  _updateMenu: function($menu) {
    var self = this;
    var $ctrl = this.element;
    var o = this.options;
    var adapter = o.adapter;
    var state = o.state;
    var panels = state.panels;
    var views = o.getViews();
    var $panel = $ctrl.find(".dashPanel.selected");
    var pos, rows, cols;
    var noAddDelete = state.fromTemplate === true;

    function addViewMenuItems(item, copy) {
      var i, v;
      var count = 0;
      
      function moveCopyViewAction() {
        var pos = self._getPanelPosition($panel);
        var panel = panels[pos.r][pos.c];
        var name;

        if (copy) {
          // clone
          panel = $.extend(true, {}, panel);
        }
        if (this.value == state.id) {
          name = self._genChartName(adapter.getName(panel.state));
          adapter.setName(panel.state, name);
          self.add(panel);
        } else {
          o.addPanelToView(panel, this.value, function(success) {
            if (success) {
              if (!copy) {
                self.remove($panel);
              }
            }
          });
        }
      }

      item.menu.items = [];
      for (i = 0; i < views.length; i++) {
        v = views[i];
        if (!copy && v.value == state.id) {
          continue;
        }
        count += 1;
        item.menu.items.push({ type: "action", label: v.name, value: v.value, action: moveCopyViewAction });
      }
      item.disabled = (count === 0);
    }

    rows = state.panels.length;
    if ($panel.length && rows > 0) {
      pos = this._getPanelPosition($panel);
      cols = state.panels[pos.r].length;
      

      $menu.menu("find", "deleteChart").disabled = noAddDelete;
      $menu.menu("find", "newChart").disabled = noAddDelete;

      addViewMenuItems($menu.menu("find", "moveTo"), false);
      addViewMenuItems($menu.menu("find", "copyTo"), true);

      $menu.menu("find", "up").disabled = (pos.r === 0);
      $menu.menu("find", "down").disabled = (pos.r === rows - 1);
      $menu.menu("find", "left").disabled = (pos.c === 0);
      $menu.menu("find", "right").disabled = (pos.c === cols - 1);
      $menu.menu("find", "above").disabled = (cols <= 1);
      $menu.menu("find", "below").disabled = (cols <= 1);
    } else {
      $.each(["deleteChart", "moveTo", "copyTo", "up", "down", "left", "right", "above", "below"], function(i, id) {
        $menu.menu("find", id).disabled = true;
      });
    }
  },

  _initMenu: function() {
    var self = this;
    var $ctrl = this.element;
    var id = $ctrl.get(0).id;
    id = id ? id : "";
    var $menu = getViewMenu(id);

    var viewMenu = {
      menubar: false,
      items: [
        { id: "newChart", type: "action", label: mvc.getMessage(BUNDLE, "view.menu.newChart"), action: function() { self._newPanel("chart"); return true; } },
        { id: "deleteChart", type: "action", label: mvc.getMessage(BUNDLE, "view.menu.deleteChart"), action: function() { self.remove(); } },
        { type: "subMenu", label: mvc.getMessage(BUNDLE, "view.menu.move"), menu: { items: [ 
          { id: "up", type: "action", label: mvc.getMessage(BUNDLE, "view.menu.moveUp"), action: function() { self._moveUp(); } },
          { id: "down", type: "action", label: mvc.getMessage(BUNDLE, "view.menu.moveDown"), action: function() { self._moveDown(); } },
          { id: "left", type: "action", label: mvc.getMessage(BUNDLE, "view.menu.moveLeft"), action: function() { self._moveLeft(); } },
          { id: "right", type: "action", label: mvc.getMessage(BUNDLE, "view.menu.moveRight"), action: function() { self._moveRight(); } },
          { id: "above", type: "action", label: mvc.getMessage(BUNDLE, "view.menu.moveAbove"), action: function() { self._moveAbove(); } },
          { id: "below", type: "action", label: mvc.getMessage(BUNDLE, "view.menu.moveBelow"), action: function() { self._moveBelow(); } }
        ] } },
        { id: "moveTo", type: "subMenu", label: mvc.getMessage(BUNDLE, "view.menu.moveTo"), menu: { items: [ ] } },
        { id: "copyTo", type: "subMenu", label: mvc.getMessage(BUNDLE, "view.menu.copyTo"), menu: { items: [ ] } }
      ]
    };
    $menu.menu(viewMenu);
  },

  _reindexAll: function() {
    var $ctrl = this.element;
    var o = this.options;

    $ctrl.find("ul:first .dashRow").each(function(r) {
      $(this).find("ul > .dashPanel").each(function(c) {
        var pos = o.panelPosMap[this.id];
        pos.r = r;
        pos.c = c;
      });
    });
  },

  // rows is an array of row indexes
  _reindexRows: function(rows) {
    var $ctrl = this.element;
    var o = this.options;
    var i, r;

    function update(c) {
      var pos = o.panelPosMap[this.id];
      pos.r = r;
      pos.c = c;
    }

    for (i = 0; i < rows.length; i++) {
      r = rows[i];
      $(makeRowSelector(r)).find("ul > .dashPanel").each(update);
    }
  },

  _getPanelPosition: function($panel) {
    var o = this.options;
    return  o.panelPosMap[$panel.get(0).id];
  },

  // call fn for each panel giving the panel, $panel, row and colum
  // fn can return false to break out of loop
  _forEachPanel: function(fn) {
    var $ctrl = this.element;
    var $panels = $ctrl.find(".dashPanel");
    var o = this.options;
    var rows = o.state.panels;
    var r, c, roff, cols, $panel;

    roff = 0;
    for (r = 0; r < rows.length; r++) {
      cols = rows[r];
      for (c = 0; c < cols.length; c++) {
        $panel = $panels.eq(roff + c);
        if (fn(cols[c], $panel, r, c) === false) {
          return;
        }
      }
      roff += cols.length;
    }
  }

});

$.extend($.ui.dashboardView, {
  version: "1.7",
  getter: ["isActive", "getPanels", "getPanel", "add"],
  defaults: {
    state: {}, // { name: "", id: "", panels: [], activeState: false, fromTemplate: }
    dropAccept: null,
    getViews: null, // fn returns array of {name: <view-name>, value: <view-id>}
    addPanelToView: null, // fn(panel, viewId, fn) adds panel to view with given id when done fn is called with true for success
    adapter: null, // {
                   //   newPanelState fn(type, name), // returns panel state object
                   //   getName: fn(state), // returns name of panel
                   //   setName: fn(state, name),
                   //   initPanel: fn($e, type, state, containerFunctions),
                   //   updatePanel: fn($e, type, state),
                   // }
    minPanelWidth: 300, // can be updated based on the needs of panels
    minPanelHeight: 200 // can be updated based on the needs of panels
  }
});

})(jQuery);
