/*!
  Dashboard entry point
  Copyright (c) 2009, 2010, Oracle and/or its affiliates. All rights reserved.
*/
/*
  Implements overall page functionality, initialization, view list, and metric browser.
*/
/*global window,jQuery,wls,mvc,wldfDM,insertView,initViewTemplates,selectView,updateViewListAndMainActions,split,stopView,deleteView,getDescriptionForMetrics */
(function($) {

var BUNDLE = "dashboard";

var VK_ENTER = 13;

var DELAY_VIEW_WAIT_TIME = 500;
var MIN_APP_STATE_WRITE_FREQ = 10 * 1000;

var globalPrefFields = [
  { id: "prefSampleInterval",
    prop: "sampleInterval",
    type: "number",
    constraints: [{fn:"required"}, {fn: "integer"}, {fn: "range", args: [10, null]}]
  },
  { id: "prefMaxSamples",
    prop: "maxSamples",
    type: "number",
    constraints: [{fn:"required"}, {fn: "integer"}, {fn: "range", args: [1, 10000]}]
  },
  { id: "prefDefaultTimeRange",
    prop: "defaultTimeRange",
    type: "number",
    constraints: [{fn:"required"}, {fn: "integer"}, {fn: "range", args: [1, null]}]
  },
  { id: "prefZoomPercent",
    prop: "zoomPercent",
    constraints: [{fn:"required"}, {fn: "integer"}, {fn: "range", args: [1, 99]}]
  }, {
    id: "prefDefaultChartType",
    prop: "defaultChartType",
    constraints: [{fn:"required"}, {fn: "member", args: [["barChart", "linePlot", "scatterPlot", "linearVGauge", "linearHGauge", "radialGauge"]]}]
  }, {
    id: "prefShowOverview",
    prop: "showOverview",
    constraints: [{fn:"required"}, {fn: "member", args: [["always", "whenSelected", "never"]]}]
  }
];

var globalPrefs = {
  sampleInterval: 20,
  maxSamples: 100,
  zoomPercent: 20,
  defaultTimeRange: 5,
  defaultChartType: "linePlot",
  showOverview: "whenSelected"
};

var appState = {
  splitterWidth: 300,
  splitterClosed: false,
  lastViewId: null
};

var loggedIn = true;
var viewsRoot = null; // root of the view tree
var builtinViews;
var userViews;

var currentView = null;
var $currentViewNode = null;
var activeViews = {};
var activeViewCount = 0;

var DEFAULT_COLOR = "#000000";
var DEFAULT_BACKGROUND = "#FFFFFF";
var DEFAULT_HIGHLIGHT = "#FBC8CA";

function writePrefs(data, fn) {
  if (!loggedIn) {
    return;
  }
  fn = fn || null;
  mvc.ajax("PUT", "data/preferences/current/dashboard", fn, data);
}

function storeGlobalPrefs() {
  writePrefs(globalPrefs);
}

var appStateTimerId = null;
var lastAppStateWriteTime = null;

function storeAppState() {
  var delta;

  if (appStateTimerId) {
    return; // a write will happen
  }
  // check if a write happend recently and if so wait a bit
  if (lastAppStateWriteTime === null) {
    delta = 0;
  } else {
    delta = (lastAppStateWriteTime + MIN_APP_STATE_WRITE_FREQ) - (new Date()).getTime();
    delta = delta < 0 ? 0 : delta;
  }
  appStateTimerId = setTimeout(function() {
    appStateTimerId = null;
    lastAppStateWriteTime = (new Date()).getTime();
    writePrefs(appState);
  }, delta);
}

function updateGlobalPrefs(store) {
  wldfDM.setSamplePeriod(globalPrefs.sampleInterval);
  wldfDM.setMaxSamples(globalPrefs.maxSamples);
  $.ui.chart.setDefaultTimeRange(globalPrefs.defaultTimeRange);
  $.ui.chart.setZoomPercent(globalPrefs.zoomPercent);
  $.ui.chart.setShowOverview(globalPrefs.showOverview);
  if (store === true) {
    storeGlobalPrefs();
  }
}

function validateGlobalPrefs(prefs) {
  var hasErrors = false;

  $.each(globalPrefFields, function(i, field) {
    var value = prefs[field.prop];
    var reason;
    var valid;
    if (!value) {
      return true; // continue. Missing value OK - default applied
    }
    field.value = value;
    valid = mvc.validate(field, null, function(f, r) {
      reason = r;
    });
    if (!valid) {
      hasErrors = true;
      delete prefs[field.prop];
      mvc.logError("Ignoring bad preference value: [" + field.prop + "=" + value + "]. Reason: " + reason);
    }
  });
  return hasErrors;
}

// Validate view. Repair or ignore invalid properties. Return false if can't be fixed.
function validateView(view) {
  var i, j, row, panels, panel;

  if (!view.id || !view.name || !view.panels) {
    mvc.logError("Ignoring bad view: missing name, id or, panels");
    return false;
  }
  panels = view.panels;
  for (i = 0; i < panels.length; i++) {
    row = panels[i];
    for (j = 0; j < row.length; j++) {
      panel = row[j];
      if (panel.type != "chart") {
        mvc.logError("Ignoring bad view '" + view.name + "': bad chart type");
        return false;
      }
      if (!$.ui.chart.validateState(row[j].state)) {
        mvc.logError("Ignoring bad view '" + view.name + "': bad chart content");
        return false;
      }
    }
  }
  return true;
}

function getInitialPageData(callback) {

  mvc.ajax("GET", "data/diagnostics/toolbar", function(result) {
    if (result) {
      loggedIn = true;
      $("#user").text(mvc.escapeHTMLContent(result.body.loggedInUser));
      $("#domain").text(mvc.escapeHTMLContent(result.body.domain));
      mvc.setLocaleSettings(result.body.localeSettings);
      if (callback) {
        callback();
      }
    }
  });
}

function loadGlobalPrefsAndViews() {
  // init current view and other view list state
  currentView = null;
  $currentViewNode = null;
  $("#viewArea").dashboardView("destroy");
  $("#viewTree").treeView("destroy");
  builtinViews.removeAllChildren();
  userViews.removeAllChildren();

  updateViewListAndMainActions();

  if (!loggedIn) {
    getInitialPageData(loadGlobalPrefsAndViews);
    return;
  }

  mvc.ajax("GET", "data/preferences/current/dashboard", function(result) {
    var prefs, fixNeeded = false, p;
    if (result) {
      prefs = result.body;

      fixNeeded = validateGlobalPrefs(prefs); // removes invalid ones
      // write to global prefs
      $.each(globalPrefs, function(key, val) {
        if (prefs[key]) {
          globalPrefs[key] = prefs[key];
        }
      });
      // app state prefs
      if (prefs.splitterWidth && prefs.splitterWidth > 10 && prefs.splitterWidth < 2000) {
        appState.splitterWidth = prefs.splitterWidth;
      }
      if (prefs.splitterClosed == "true" || prefs.splitterClosed == "false") {
        appState.splitterClosed = prefs.splitterClosed == "true";
      }
      if (prefs.lastViewId) {
        appState.lastViewId = prefs.lastViewId;
      }
      // views
      $.each(prefs, function(key, val) {
        var view;
        if (key.match(/^view_[a-z,A-Z,\-,0-9]*$/)) {
          view = JSON.parse(val);
          if (validateView(view)) {
            insertView(view);
          }
        }
      });
    }
    updateGlobalPrefs(fixNeeded);
    p = appState.splitterClosed ? 0 : appState.splitterWidth;
    $("#mainContent > .splitterH").css("left", p + "px");
    if (appState.splitterClosed) {
      $("#mainContent > .splitterH button").click();
    }
    $("#explorer").width(appState.splitterWidth + "px");
    split(p, appState.splitterClosed);

    initViewTemplates(); // this also creates the view list tree

  });
}

function parseObjectName(onText) {
  var on = {}, keys;
  var sep, list, i, nvp;

  sep = onText.indexOf(":");
  if (sep < 0) {
    throw "invalid object name";
  }
  on.domain = onText.substring(0, sep);
  on.keys = {};
  keys = on.keys;
  onText = onText.substring(sep + 1);
  list = onText.split(",");
  for (i = 0; i < list.length; i++) {
    nvp = list[i].split("=");
    if (nvp.length != 2) {
      throw "invalid object name key";
    }
    keys[nvp[0]] = nvp[1];
  }
  return on;
}

//
// Built in view template definitions
//
var builtInViewTemplates = [
  {
    path: "JVMRuntime",
    template: [{
      category: "",
      name: {templateFormatKey: "bview.jvm.viewTitle", bundle: BUNDLE},
      panels: {templateForeach: "servers", merge: "flattenInstances", body: [
        {
          type: "chart",
          state: {
            name: {templateFormatKey: "bview.jvm.chartTitle", bundle: BUNDLE, args:["server"]},
            type: "linePlot",
            units: {templateFormatKey: "bview.units.bytes", bundle: BUNDLE},
            background: {templateValue: "background"},
            color: {templateValue: "color"},
            highlight: {templateValue: "highlight"},
            autoScale: true,
            metrics: [
              {
                name: {templateFormatKey: "bview.jvm.metricSize", bundle: BUNDLE},
                metric: {on: {templateValue: "instance"},a:"HeapSizeCurrent",type:"weblogic.management.runtime.JVMRuntimeMBean",server: {templateValue: "server"}},
                color: "#FFCC33", markerShape: "circle", markerFill: true
              },
              {
                name: {templateFormatKey: "bview.jvm.metricFree", bundle: BUNDLE}, 
                metric: {on: {templateValue: "instance"},a:"HeapFreeCurrent",type:"weblogic.management.runtime.JVMRuntimeMBean",server: {templateValue: "server"}},
                color: "#33CCFF",markerShape: "diamond", markerFill: true
              }
            ]
          }
        }
      ]}
    }]
  },
  {
    path: "WorkManagerRuntimes",
    template: {templateForeach: "servers", body: {
      category: {templateValue: "server"},
      name: {templateFormatKey: "bview.serverwm.viewTitle", bundle: BUNDLE, args:["server"]},
      panels: {templateForeach: "instances", merge:"instanceName", body: [
        {
          type: "chart",
          state: {
            name: {templateFormatKey: "bview.wm.chartTitle1", bundle: BUNDLE, args:["instanceName"]},
            type: "linePlot",
            units: "",
            background: {templateValue: "background"},
            color: {templateValue: "color"},
            highlight: {templateValue: "highlight"},
            autoScale: true,
            metrics: [
              {
                name: {templateFormatKey: "bview.wm.metricCompleted", bundle: BUNDLE},
                metric: {on: {templateValue: "instance"},a:"CompletedRequests",type:"weblogic.management.runtime.WorkManagerRuntimeMBean",server: {templateValue: "server"}},
                color: "#FFCC33", markerShape: "circle", markerFill:true
              }
            ]
          }
        },
        {
          type: "chart",
          state: {
            name: {templateFormatKey: "bview.wm.chartTitle2", bundle: BUNDLE, args:["instanceName"]},
            type: "linePlot",
            units: "",
            background: {templateValue: "background"},
            color: {templateValue: "color"},
            highlight: {templateValue: "highlight"},
            autoScale: true,
            metrics: [
              {
                name: {templateFormatKey: "bview.wm.metricPending", bundle: BUNDLE},
                metric: {on: {templateValue: "instance"},a:"PendingRequests",type:"weblogic.management.runtime.WorkManagerRuntimeMBean",server: {templateValue: "server"}},
                color: "#FFCC33", markerShape: "circle", markerFill: true
              },
              {
                name: {templateFormatKey: "bview.wm.metricStuck", bundle: BUNDLE},
                metric:{on: {templateValue: "instance"},a:"StuckThreadCount",type:"weblogic.management.runtime.WorkManagerRuntimeMBean",server: {templateValue: "server"}},
                color: "#33CCFF", markerShape: "diamond",markerFill: true
              }
            ]
          }
        }
      ]}
    }}
  },
  {
    path: "ApplicationRuntimes.WorkManagerRuntimes",
    template: {templateForeach: "servers", body: {
      category: {templateValue: "server"},
      name: {templateFormatKey: "bview.appwm.viewTitle", bundle: BUNDLE, args:["server"]},
      panels: {templateForeach: "instances", merge:"instanceName", keys: ["ApplicationRuntime", "Name"], body: [
        {
          type: "chart",
          state: {
            name: {templateFormatKey: "bview.wm.chartTitle1", bundle: BUNDLE, args:["instanceName"]},
            type: "linePlot",
            units: "",
            background: {templateValue: "background"},
            color: {templateValue: "color"},
            highlight: {templateValue: "highlight"},
            autoScale: true,
            metrics: [
              {
                name: {templateFormatKey: "bview.wm.metricCompleted", bundle: BUNDLE},
                metric: {on: {templateValue: "instance"},a:"CompletedRequests",type:"weblogic.management.runtime.WorkManagerRuntimeMBean",server: {templateValue: "server"}},
                color: "#FFCC33", markerShape: "circle", markerFill:true
              }
            ]
          }
        },
        {
          type: "chart",
          state: {
            name: {templateFormatKey: "bview.wm.chartTitle2", bundle: BUNDLE, args:["instanceName"]},
            type: "linePlot",
            units: "",
            background: {templateValue: "background"},
            color: {templateValue: "color"},
            highlight: {templateValue: "highlight"},
            autoScale: true,
            metrics: [
              {
                name: {templateFormatKey: "bview.wm.metricPending", bundle: BUNDLE},
                metric: {on: {templateValue: "instance"},a:"PendingRequests",type:"weblogic.management.runtime.WorkManagerRuntimeMBean",server: {templateValue: "server"}},
                color: "#FFCC33", markerShape: "circle", markerFill: true
              },
              {
                name: {templateFormatKey: "bview.wm.metricStuck", bundle: BUNDLE},
                metric:{on: {templateValue: "instance"},a:"StuckThreadCount",type:"weblogic.management.runtime.WorkManagerRuntimeMBean",server: {templateValue: "server"}},
                color: "#33CCFF", markerShape: "diamond",markerFill: true
              }
            ]
          }
        }
      ]}
    }}
  },
  {
    path: "JMSRuntime",
    template: [{
      category: "",
      name: {templateFormatKey: "bview.jms.viewTitle", bundle: BUNDLE},
      panels: {templateForeach: "servers", merge: "flattenInstances", body: [
        {
          type: "chart",
          state: {
            name: {templateFormatKey: "bview.jms.chartTitle", bundle: BUNDLE, args:["server"]},
            type: "linePlot",
            units: "",
            background: {templateValue: "background"},
            color: {templateValue: "color"},
            highlight: {templateValue: "highlight"},
            autoScale: true,
            metrics: [
              {
                name: {templateFormatKey: "bview.jms.metricConn", bundle: BUNDLE},
                metric: {on: {templateValue: "instance"},a:"ConnectionsCurrentCount",type:"weblogic.management.runtime.JMSRuntimeMBean",server: {templateValue: "server"}},
                color: "#FFCC33", markerShape: "circle", markerFill: true
              }
            ]
          }
        }
      ]}
    }]
  },
  {
    path: "JMSRuntime.JMSServers",
    template: {templateForeach: "servers", body: {
      category: {templateValue: "server"},
      name: {templateFormatKey: "bview.jmssvr.viewTitle", bundle: BUNDLE, args:["server"]},
      panels: {templateForeach: "instances", merge:"instanceName", body: [
        {
          type: "chart",
          state: {
            name: {templateFormatKey: "bview.jmssvr.chartTitle1", bundle: BUNDLE, args:["instanceName"]},
            type: "linePlot",
            units: {templateFormatKey: "bview.units.messages", bundle: BUNDLE},
            background: {templateValue: "background"},
            color: {templateValue: "color"},
            highlight: {templateValue: "highlight"},
            autoScale: true,
            metrics: [
              {
                name: {templateFormatKey: "bview.jmssvr.metricMessagesCurrent", bundle: BUNDLE},
                metric: {on: {templateValue: "instance"},a:"MessagesCurrentCount",type:"weblogic.management.runtime.JMSServerRuntimeMBean",server: {templateValue: "server"}},
                color: "#FFCC33", markerShape: "circle", markerFill:true
              },
              {
                name: {templateFormatKey: "bview.jmssvr.metricMessagesPending", bundle: BUNDLE},
                metric: {on: {templateValue: "instance"},a:"MessagesPendingCount",type:"weblogic.management.runtime.JMSServerRuntimeMBean",server: {templateValue: "server"}},
                color: "#33CCFF", markerShape: "diamond", markerFill:true
              }
            ]
          }
        },
        {
          type: "chart",
          state: {
            name: {templateFormatKey: "bview.jmssvr.chartTitle2", bundle: BUNDLE, args:["instanceName"]},
            type: "linePlot",
            units: {templateFormatKey: "bview.units.bytes", bundle: BUNDLE},
            background: {templateValue: "background"},
            color: {templateValue: "color"},
            highlight: {templateValue: "highlight"},
            autoScale: true,
            metrics: [
              {
                name: {templateFormatKey: "bview.jmssvr.metricBytesCurrent", bundle: BUNDLE},
                metric: {on: {templateValue: "instance"},a:"BytesCurrentCount",type:"weblogic.management.runtime.JMSServerRuntimeMBean",server: {templateValue: "server"}},
                color: "#FFCC33", markerShape: "circle", markerFill: true
              },
              {
                name: {templateFormatKey: "bview.jmssvr.metricBytesPending", bundle: BUNDLE},
                metric:{on: {templateValue: "instance"},a:"BytesPendingCount",type:"weblogic.management.runtime.JMSServerRuntimeMBean",server: {templateValue: "server"}},
                color: "#33CCFF", markerShape: "diamond",markerFill: true
              }
            ]
          }
        }
      ]}
    }}
  },
  {
    path: "ThreadPoolRuntime",
    template: [{
      category: "",
      name: {templateFormatKey: "bview.threadpool.viewTitle", bundle: BUNDLE},
      panels: {templateForeach: "servers", merge: "flattenInstances", body: [
        {
          type: "chart",
          state: {
            name: {templateFormatKey: "bview.threadpool.chartTitle", bundle: BUNDLE, args:["server"]},
            type: "linePlot",
            units: "",
            background: {templateValue: "background"},
            color: {templateValue: "color"},
            highlight: {templateValue: "highlight"},
            autoScale: true,
            metrics: [
              {
                name: {templateFormatKey: "bview.threadpool.metricHogging", bundle: BUNDLE},
                metric: {on: {templateValue: "instance"},a:"HoggingThreadCount",type:"weblogic.management.runtime.ThreadPoolRuntimeMBean",server: {templateValue: "server"}},
                color: "#FFCC33", markerShape: "circle", markerFill: true
              },
              {
                name: {templateFormatKey: "bview.threadpool.metricStandby", bundle: BUNDLE},
                metric: {on: {templateValue: "instance"},a:"StandbyThreadCount",type:"weblogic.management.runtime.ThreadPoolRuntimeMBean",server: {templateValue: "server"}},
                color: "#33CCFF", markerShape: "diamond", markerFill: true
              },
              {
                name: {templateFormatKey: "bview.threadpool.metricUser", bundle: BUNDLE},
                metric: {on: {templateValue: "instance"},a:"PendingUserRequestCount",type:"weblogic.management.runtime.ThreadPoolRuntimeMBean",server: {templateValue: "server"}},
                color: "#CC3333", markerShape: "triangleUp", markerFill: true
              }
            ]
          }
        }
      ]}
    }]
  },
  {
    path: "JDBCServiceRuntime.JDBCDataSourceRuntimeMBeans",
    template: {templateForeach: "servers", body: {
      category: {templateValue: "server"},
      name: {templateFormatKey: "bview.jdbc.viewTitle", bundle: BUNDLE, args:["server"]},
      panels: {templateForeach: "instances", merge:"instanceName", bodyItems: [ [
        {
          type: "chart",
          state: {
            name: {templateFormatKey: "bview.jdbc.chartTitle1", bundle: BUNDLE, args:["instanceName"]},
            type: "linePlot",
            units: "",
            background: {templateValue: "background"},
            color: {templateValue: "color"},
            highlight: {templateValue: "highlight"},
            autoScale: true,
            metrics: [
              {
                name: {templateFormatKey: "bview.jdbc.metricCap", bundle: BUNDLE},
                metric: {on: {templateValue: "instance"},a:"CurrCapacity",type:"weblogic.management.runtime.JDBCDataSourceRuntimeMBean",server: {templateValue: "server"}},
                color: "#FFCC33", markerShape: "circle", markerFill:true
              },
              {
                name: {templateFormatKey: "bview.jdbc.metricAvail", bundle: BUNDLE},
                metric: {on: {templateValue: "instance"},a:"NumAvailable",type:"weblogic.management.runtime.JDBCDataSourceRuntimeMBean",server: {templateValue: "server"}},
                color: "#33CCFF", markerShape: "diamond", markerFill:true
              }
            ]
          }
        },
        {
          type: "chart",
          state: {
            name: {templateFormatKey: "bview.jdbc.chartTitle2", bundle: BUNDLE, args:["instanceName"]},
            type: "linePlot",
            units: "",
            background: {templateValue: "background"},
            color: {templateValue: "color"},
            highlight: {templateValue: "highlight"},
            autoScale: true,
            metrics: [
              {
                name: {templateFormatKey: "bview.jdbc.metricActive", bundle: BUNDLE},
                metric: {on: {templateValue: "instance"},a:"ActiveConnectionsCurrentCount",type:"weblogic.management.runtime.JDBCDataSourceRuntimeMBean",server: {templateValue: "server"}},
                color: "#FFCC33", markerShape: "circle", markerFill: true
              },
              {
                name: {templateFormatKey: "bview.jdbc.metricWaiting", bundle: BUNDLE},
                metric:{on: {templateValue: "instance"},a:"WaitingForConnectionCurrentCount",type:"weblogic.management.runtime.JDBCDataSourceRuntimeMBean",server: {templateValue: "server"}},
                color: "#33CCFF", markerShape: "diamond",markerFill: true
              }
            ]
          }
        } ],[
        {
          type: "chart",
          state: {
            name: {templateFormatKey: "bview.jdbc.chartTitle3", bundle: BUNDLE, args:["instanceName"]},
            type: "linePlot",
            units: "",
            background: {templateValue: "background"},
            color: {templateValue: "color"},
            highlight: {templateValue: "highlight"},
            autoScale: true,
            metrics: [
              {
                name: {templateFormatKey: "bview.jdbc.metricLeaked", bundle: BUNDLE},
                metric: {on: {templateValue: "instance"},a:"LeakedConnectionCount",type:"weblogic.management.runtime.JDBCDataSourceRuntimeMBean",server: {templateValue: "server"}},
                color: "#FFCC33", markerShape: "circle", markerFill:true
              },
              {
                name: {templateFormatKey: "bview.jdbc.metricFailed", bundle: BUNDLE},
                metric: {on: {templateValue: "instance"},a:"FailedReserveRequestCount",type:"weblogic.management.runtime.JDBCDataSourceRuntimeMBean",server: {templateValue: "server"}},
                color: "#33CCFF", markerShape: "diamond", markerFill:true
              }
            ]
          }
        },
        {
          type: "chart",
          state: {
            name: {templateFormatKey: "bview.jdbc.chartTitle4", bundle: BUNDLE, args:["instanceName"]},
            type: "linePlot",
            units: "",
            background: {templateValue: "background"},
            color: {templateValue: "color"},
            highlight: {templateValue: "highlight"},
            autoScale: true,
            metrics: [
              {
                name: {templateFormatKey: "bview.jdbc.metricSize", bundle: BUNDLE},
                metric: {on: {templateValue: "instance"},a:"PrepStmtCacheCurrentSize",type:"weblogic.management.runtime.JDBCDataSourceRuntimeMBean",server: {templateValue: "server"}},
                color: "#FFCC33", markerShape: "circle", markerFill: true
              },
              {
                name: {templateFormatKey: "bview.jdbc.metricHit", bundle: BUNDLE},
                metric:{on: {templateValue: "instance"},a:"PrepStmtCacheHitCount",type:"weblogic.management.runtime.JDBCDataSourceRuntimeMBean",server: {templateValue: "server"}},
                color: "#33CCFF", markerShape: "diamond",markerFill: true
              },
              {
                name: {templateFormatKey: "bview.jdbc.metricMiss", bundle: BUNDLE},
                metric:{on: {templateValue: "instance"},a:"PrepStmtCacheMissCount",type:"weblogic.management.runtime.JDBCDataSourceRuntimeMBean",server: {templateValue: "server"}},
                color: "#CC3333", markerShape: "triangleUp", markerFill: true
              }
            ]
          }
        } ]
      ]}
    }}
  },
  {
    path: "ServerChannelRuntimes",
    template: {templateForeach: "servers", filter: "byInstanceKey", key: "Name", body: {
      category: {templateValue: "server"},
      name: {templateFormatKey: "bview.chan.viewTitle", bundle: BUNDLE, args:["server", "name"]},
      panels: [ [
        {
          type: "chart",
          state: {
            name: {templateFormatKey: "bview.chan.chartTitle1", bundle: BUNDLE},
            type: "linePlot",
            units: "",
            background: {templateValue: "background"},
            color: {templateValue: "color"},
            highlight: {templateValue: "highlight"},
            autoScale: true,
            metrics: [
              {
                name: {templateFormatKey: "bview.chan.metricAccept", bundle: BUNDLE},
                metric: {on: {templateValue: "instance"},a:"AcceptCount",type:"weblogic.management.runtime.ServerChannelRuntimeMBean",server: {templateValue: "server"}},
                color: "#FFCC33", markerShape: "circle", markerFill:true
              }
            ]
          }
        },
        {
          type: "chart",
          state: {
            name: {templateFormatKey: "bview.chan.chartTitle2", bundle: BUNDLE},
            type: "linePlot",
            units: "",
            background: {templateValue: "background"},
            color: {templateValue: "color"},
            highlight: {templateValue: "highlight"},
            autoScale: true,
            metrics: [
              {
                name: {templateFormatKey: "bview.chan.metricConnect", bundle: BUNDLE},
                metric: {on: {templateValue: "instance"},a:"ConnectionsCount",type:"weblogic.management.runtime.ServerChannelRuntimeMBean",server: {templateValue: "server"}},
                color: "#FFCC33", markerShape: "circle", markerFill: true
              }
            ]
          }
        } ],[
        {
          type: "chart",
          state: {
            name: {templateFormatKey: "bview.chan.chartTitle3", bundle: BUNDLE},
            type: "linePlot",
            units: {templateFormatKey: "bview.units.bytes", bundle: BUNDLE},
            background: {templateValue: "background"},
            color: {templateValue: "color"},
            highlight: {templateValue: "highlight"},
            autoScale: true,
            metrics: [
              {
                name: {templateFormatKey: "bview.chan.metricBRecv", bundle: BUNDLE},
                metric: {on: {templateValue: "instance"},a:"BytesReceivedCount",type:"weblogic.management.runtime.ServerChannelRuntimeMBean",server: {templateValue: "server"}},
                color: "#FFCC33", markerShape: "circle", markerFill:true
              },
              {
                name: {templateFormatKey: "bview.chan.metricBSent", bundle: BUNDLE},
                metric: {on: {templateValue: "instance"},a:"BytesSentCount",type:"weblogic.management.runtime.ServerChannelRuntimeMBean",server: {templateValue: "server"}},
                color: "#33CCFF", markerShape: "diamond", markerFill:true
              }
            ]
          }
        },
        {
          type: "chart",
          state: {
            name: {templateFormatKey: "bview.chan.chartTitle4", bundle: BUNDLE},
            type: "linePlot",
            units: {templateFormatKey: "bview.units.messages", bundle: BUNDLE},
            background: {templateValue: "background"},
            color: {templateValue: "color"},
            highlight: {templateValue: "highlight"},
            autoScale: true,
            metrics: [
              {
                name: {templateFormatKey: "bview.chan.metricMRecv", bundle: BUNDLE},
                metric: {on: {templateValue: "instance"},a:"MessagesReceivedCount",type:"weblogic.management.runtime.ServerChannelRuntimeMBean",server: {templateValue: "server"}},
                color: "#FFCC33", markerShape: "circle", markerFill: true
              },
              {
                name: {templateFormatKey: "bview.chan.metricMSent", bundle: BUNDLE},
                metric:{on: {templateValue: "instance"},a:"MessagesSentCount",type:"weblogic.management.runtime.ServerChannelRuntimeMBean",server: {templateValue: "server"}},
                color: "#33CCFF", markerShape: "diamond",markerFill: true
              }
            ]
          }
        } ]
      ]
    }}
  }

];

var templateFunctions = {
  "default": function(val) {
    var merge = val;
    if (typeof val != "object") {
      merge = {};
      merge[this.item] = val;
    }
    return merge;
  },
  "flattenInstances": function(val) {
    return {server: val.server, instance: val.instances[0]};
  },
  "instanceName": function(val) {
    var on = parseObjectName(val);
    var keys = this.keys || ["Name"];
    var name = [];
    $.each(keys, function(i, key) {
      name.push(on.keys[key]);
    });
    return {instanceName: name.join(", "), instance: val};
  },
  "byInstanceKey": function(vars) {
    var c = [];

    function foreachInstance(instances, key, server) {
      $.each(instances, function(i, instance) {
        var on = parseObjectName(instance);
        c.push({server: server, name: on.keys[key], instance: instance});
      });
    }

    $.each(vars[this.templateForeach], function(i, item) {
      foreachInstance(item.instances, this.key || "Name", item.server);
    });
    return c;
  }
};

function processTemplate(t, vars) {
  var result, n, args, collection;
  if (t === null) {
    result = t;
  } else if ($.isArray(t)) {
    result = [];
    $.each(t,function(i, val) {
      result.push(processTemplate(val, vars));
    });
  } else if (typeof t === "object") {
    if (t.templateValue) {
      result = vars[t.templateValue];
    } else if (t.templateFormatKey) {
      args = [ t.bundle, t.templateFormatKey ];
      if (t.args) {
        $.each(t.args, function(i, val) {
          args.push(vars[val]);
        });
        result = mvc.format.apply(this, args);
      } else {
        result = mvc.getMessage.apply(this, args);
      }
    } else if (t.templateForeach) {
      result = [];
      collection = t.filter ? templateFunctions[t.filter].call(t, vars) : vars[t.templateForeach];
      $.each(collection, function(i, val) {
        var merge = templateFunctions[t.merge ? t.merge : "default"].call(t, val);
        merge.index = i;
        var newvars = $.extend({}, vars, merge);
        if (t.body) {
          result.push(processTemplate(t.body, newvars));
        } else if (t.bodyItems) {
          result = result.concat(processTemplate(t.bodyItems, newvars));
        }
      });
    } else {
      result = {};
      $.each(t,function(k, val) {
        result[k] = processTemplate(val, vars);
      });
    }
  } else {
    result = t;
  }
  return result;
}

//
// Metric browser
//
var currentMetrics = null;

function getCategoryAndDisplayTextForType(type) {
  var displayType = type;
  var category = "custom";
  if (type.indexOf("weblogic.") === 0 && type.indexOf("weblogic.management.mbeanservers.") !== 0) {
    displayType = type.substring(type.lastIndexOf(".") + 1).replace(/(.)RuntimeMBean$/, "$1");
    category = "wls";
  } else if (type.indexOf("sun.") === 0 || type.indexOf("com.sun.") === 0) {
    displayType = type.substring(type.lastIndexOf(".") + 1);
    category = "platform";
  } else if (type.indexOf("javax.management") === 0) {
    displayType = type.substring(type.lastIndexOf(".") + 1);
    category = "jmx";
  }
  return {display: displayType, category: category};
}

function getInstanceDisplayText(category, instance) {
  var displayText = instance;
  var on, keys, names;

  if (category == "wls") {
    names = [];
    on = parseObjectName(instance);
    keys = on.keys;
    $.each(["Name", "ApplicationRuntime", "WseePortRuntime"], function(i, v) {
      if (keys[v]) {
        names.push(keys[v]);
      }
    });
    if (names.length > 0) {
      displayText = names.join(", ");
    }
  }
  return displayText;
}

function getMetricDisplayText(metric) {
  return metric.name + " (" + metric.type + ")";
}

function getMetricName(metric, instance) {
  var name = metric.metric.name;
  if (instance) {
    name += "@" + getInstanceDisplayText(getCategoryAndDisplayTextForType(instance.type).category, instance.instance.on);
  }
  return name;
}

var mbCategoryToIcon = {
  "custom": "metricTypeCust",
  "wls": "metricTypeWLS",
  "platform": "metricTypePlatform",
  "jmx": "metricTypeJMX"
};
var mbCategoryToLabelKey = {
  "custom": "mb.note.category.custom",
  "wls": "mb.note.category.wls",
  "platform": "mb.note.category.platform",
  "jmx": "mb.note.category.jmx"
};

function mbCurrentType() {
  if (!currentMetrics) {
    return null;
  }
  var types = currentMetrics.harvestableTypes;
  var v = $("#mbTypesFL").filteredList("selection");
  if (v === null || v === "") {
    return v;
  }
  return types[v].type;
}

function mbCurrentInstance() {
  if (!currentMetrics) {
    return null;
  }
  var types = currentMetrics.harvestableTypes;
  var v = $("#mbInstancesFL").filteredList("selection");
  if (v === null) {
    return null;
  }
  var slashPos = v.indexOf("/");
  var typeIndex = v.substring(0, slashPos) * 1;
  var type = types[typeIndex];
  var instance = type.harvestableInstances[v.substring(slashPos + 1)];
  return {instance: instance, type: type.type, typeIndex: typeIndex};
}

function mbCurrentMetric() {
  if (!currentMetrics) {
    return null;
  }
  var types = currentMetrics.harvestableTypes;
  var v = $("#mbMetricsFL").filteredList("selection");
  if (v === null) {
    return null;
  }
  var slashPos = v.indexOf("/");
  var typeIndex = v.substring(0, slashPos) * 1;
  var type = types[typeIndex];
  var metric = type.harvestableAttributes[v.substring(slashPos + 1)];
  return {metric: metric, type: type.type, typeIndex: typeIndex};
}

function displayTypeNote($target, type) {
  if (type === "") {
    return;
  }
  var typeInfo = getCategoryAndDisplayTextForType(type);
  var note = mvc.formatNoteContent(BUNDLE, [
    { label: mbCategoryToLabelKey[typeInfo.category], value: null },
    { label: "mb.note.type", value: type }
  ]);
  mvc.showNote(note, $target);
}

function displayInstanceNote($target, instance) {
  var i = instance.instance;
  var note = mvc.formatNoteContent(BUNDLE, [
    { label: "mb.note.objectname", value: i.on.replace(/,/g, ", ") },
    { label: "mb.note.harvested", value: mvc.getMessage("mvccore", i.harvested ? "field.true" : "field.false") }
  ]);
  mvc.showNote(note, $target);
}

var $metricNote = null;

function displayMetricNote($target, metric) {
  var m = metric.metric;
  var instance = mbCurrentInstance();
  var description = m.description || "";

  if (m.description === undefined) {
    getDescriptionForMetrics(currentMetrics.serverName, metric.typeIndex);
  }
  var nc = [
    { label: "mb.note.metric", value: m.name },
    { label: "mb.note.mbeanType", value: metric.type },
    { label: "mb.note.type", value: m.type },
    { label: "mb.note.harvested", value: mvc.getMessage("mvccore", m.harvested ? "field.true" : "field.false") },
    { label: "mb.note.description", value: description, markup: true }
  ];
  if (instance === null) {
    nc.push({separator: true});
    nc.push({label: "mb.selectInstance", value: null });
  }
  var note = mvc.formatNoteContent(BUNDLE, nc);
  $metricNote = mvc.showNote(note, $target);
}

function updateMetricBrowserLists(selectedType) {
  var types, i, type, typeInfo;
  var instances, j, instance, label, count;
  var metrics, metric;
  var flTypes = [];
  var flInstances = [];
  var flMetrics = [];
  var harvestedOnly = $("#mbHarvestedOnly").get(0).checked;
  var allTypes = harvestedOnly ? false : $("#mbAllTypes").get(0).checked;

  types = currentMetrics.harvestableTypes;
  for (i = 0; i < types.length; i++) {
    type = types[i];
    if (!selectedType || selectedType == type.type) {
      
      typeInfo = getCategoryAndDisplayTextForType(type.type);
      instances = type.harvestableInstances;
      metrics = type.harvestableAttributes;
      if (allTypes || (metrics.length > 0 && instances.length > 0)) {
        count = 0;
        for (j = 0; j < instances.length; j++) {
          instance = instances[j];
          if (harvestedOnly && !instance.harvested) {
            continue;
          }
          count += 1;
          flInstances.push({l: getInstanceDisplayText(typeInfo.category, instance.on), v: i + "/" + j});
        }
        for (j = 0; j < metrics.length; j++) {
          metric = metrics[j];
          if (harvestedOnly && !metric.harvested) {
            continue;
          }
          count += 1;
          label = getMetricDisplayText(metric);
          flMetrics.push({l: label, v: i + "/" + j});
        }
        if (count > 0) {
          flTypes.push({l: typeInfo.display, v: i,
                        icon: mbCategoryToIcon[typeInfo.category], numInstances: instances.length });
        }
      }
    }
  }
  function lvcompare(a, b) {
    var sa = a.l, sb = b.l;
    return sa > sb ? 1 : sa < sb ? -1 : 0;
  }
  if (!selectedType && selectedType !== "") {
    flTypes.sort(lvcompare);
    flTypes.unshift({l: mvc.getMessage(BUNDLE, "mb.typeNone"), v: "" });
  }
  flInstances.sort(lvcompare);
  flMetrics.sort(lvcompare);

  mvc.hideNote(); // in case any lists about to be updated are showing a note
  if (!selectedType && selectedType !== "") {
    $("#mbTypesFL").filteredList("update", flTypes);
  }
  $("#mbInstancesFL").filteredList("update", flInstances);
  $("#mbMetricsFL").filteredList("option", "enableDrag", false).filteredList("update", flMetrics);
}

function getDescriptionForMetrics(server, typeIndex) {
  var type = currentMetrics.harvestableTypes[typeIndex];
  var inst = mbCurrentInstance();
  var instance = inst ? inst.instance.on : type.harvestableInstances.length > 0 ? type.harvestableInstances[0].on : null;

  var url = mvc.uriTemplate("data/diagnostics/MBeanMetaData?serverName={server}&mbeanType={type}{-prefix|&instanceName=|on}");

  if (!loggedIn) {
    return;
  }

  mvc.ajax("GET", url.expand({server: server, type: type.type, on: instance}), function(result) {
    var i, md, metrics, metric;
    var $target;

    if (currentMetrics && currentMetrics.serverName == server) {
      md = result ? result.body : {};
      metrics = type.harvestableAttributes;
      for (i = 0; i < metrics.length; i++) {
        metric = metrics[i];
        metric.description = md[metric.name] || mvc.getMessage(BUNDLE, "mb.note.description.unavailable");
      }
      if ($metricNote == mvc.getCurrentNote()) {
        $target = $("#mbMetricsFL").find(".select");
        metric = mbCurrentMetric();
        if (metric) {
          displayMetricNote($target, metric);
        }
      }
    }
  });
}

function getMetricsForServer(server) {
  var url = mvc.uriTemplate("data/servers/{server}/diagnostics/metriclist");

  $("#mbTypesFL").filteredList("update", []);
  $("#mbInstancesFL").filteredList("update", []);
  $("#mbMetricsFL").filteredList("update", []);
  currentMetrics = null;

  if (!loggedIn) {
    getInitialPageData(function() {
      getMetricsForServer(server);
    });
    return;
  }

  mvc.ajax("GET", url.expand({server: server}), function(result) {
    if (result) {
      currentMetrics = result.body;
      updateMetricBrowserLists();
    }
  });
}

function getRunningServers() {
  if (!loggedIn) {
    getInitialPageData(getRunningServers);
    return;
  }
  mvc.ajax("GET", "data/diagnostics/runningServers", function(result) {
    var servers = [], server, i, current;
    var $selServers = $("#mbServers");
    var out = mvc.htmlBuilder();

    if (result) {
      servers = result.body.servers;
    }
    current = $selServers.val();
    for (i = 0; i < servers.length; i++) {
      server = servers[i];
      out.markup("<option value='").attr(server).markup("'").optionalBoolAttr("selected", server === current).markup(">");
      out.content(server).markup("</option>");
    }
    $selServers.html(out.toString());

    if ($selServers.val() !== current) {
      getMetricsForServer($selServers.val());
    }
  });
}

var metricListAdapterPrototype = {
  icon: "", // customize this
  value: function(item) {
    return item.v;
  },
  renderLabel: function(out, item) {
    var icon = this.icon || item.icon;
    var lclass = "label";

    mvc.renderIcon(out, icon);
    if (item.numInstances === 0) {
      lclass += " dim";
    }
    out.markup("<span class='").attr(lclass).markup("'>").
      content(item.l).markup("</span>");
  }
};
function metricListAdapter(icon) {
  var that = Object.create(metricListAdapterPrototype);
  that.icon = icon;
  return that;
}

function mbTypesChanged() {
  var type = mbCurrentType();
  updateMetricBrowserLists(type === null ? "" : type);
}

function mbInstancesChanged() {
  var value = $("#mbInstancesFL").filteredList("selection");
  var instance = mbCurrentInstance();
  var type = mbCurrentType();
  if (instance && instance.type != type) {
    $("#mbTypesFL").filteredList("select", instance.typeIndex);
    $("#mbInstancesFL").filteredList("select", value);
  }
  $("#mbMetricsFL").filteredList("option", "enableDrag", instance !== null);
}

function mbMetricsChanged() {
  var value = $("#mbMetricsFL").filteredList("selection");
  var metric = mbCurrentMetric();
  var instance = mbCurrentInstance();
  var type = mbCurrentType();
  if (metric && metric.type != type) {
    $("#mbTypesFL").filteredList("select", metric.typeIndex);
    $("#mbMetricsFL").filteredList("select", value);
  }
}

function mbGetCurrentMetricValue() {
  var instance = mbCurrentInstance();
  var metric = mbCurrentMetric();
  var server = currentMetrics.serverName;
  var name;
  if (metric !== null && instance !== null) {
    name = getMetricName(metric, instance);
    return {name: name, metric: {on: instance.instance.on, a: metric.metric.name, type: instance.type, server: server} };
  }
  return null;
}

function mbHasCurrentMetricValue() {
  var instance = mbCurrentInstance();
  var metric = mbCurrentMetric();
  return metric !== null && instance !== null;
}

//
// View list
//
function binsearch(a, item, cmp) {
  var lb = 0;
  var ub = a.length;
  var m, c;

  while (ub - lb > 0) {
    m = lb + Math.floor((ub - lb) / 2);
    c = cmp(item, a[m]);
    if (c === 0) {
      return m;
    } else if (c < 0) {
      ub = m;
    } else {
      lb = m + 1;
    }
  }
  return -(lb + 1);
}

function nodeComp(a, b) {
  var a1 = a.getName();
  var b1 = b.getName();
  if (a1 === b1) {
    return 0;
  } else if (a1 < b1) {
    return -1;
  } // else >
  return 1;
}

function nodeNameComp(a1, b) {
  var b1 = b.getName();
  if (a1 === b1) {
    return 0;
  } else if (a1 < b1) {
    return -1;
  } // else >
  return 1;
}

/*
  All nodes require:
  getName() method that returns the unique name of the node. This is used to determine the sort order
  parent this is a reference to the nodes parent and is maintined by the node implementation

  Non leaf nodes require:
  children[] and ordered array of nodes
*/
var nodePrototype = {
  insertChild: function(node) {
    var c = this.children;
    var i = binsearch(c, node, nodeComp);
    if (i >= 0) {
      return -1;  // duplicate name
    }
    i = -i - 1;
    node.parent = this;
    if (i >= c.length) {
      c.push(node);
    } else {
      c.splice(i, 0, node);
    }
    return i;
  },
  getChildCount: function() {
    return this.children.length;
  },
  getChild: function(i) {
    return this.children[i];
  },
  removeAllChildren: function() {
    this.children = [];
  },
  removeChild: function(i) {
    var c = this.children;
    var node = c[i];
    node.parent = null;
    c.splice(i, 1);
    return node;
  },
  removeChildByName: function(name) {
    var c = this.children;
    var i = binsearch(c, name, nodeNameComp);
    if (i >= 0) {
      return this.removeChild(i);
    }
    return null;
  },
  findChildByName: function(name) {
    var c = this.children;
    var i = binsearch(c, name, nodeNameComp);
    if (i >= 0) {
      return c[i];
    }
    return null;
  }
};

function folderNode(name) {
  var that = Object.create(nodePrototype);
  that.children = [];
  that.name = name;
  that.parent = null;
  that.getName = function() { return this.name; };
  return that;
}

function viewNode(view) {
  var that = { view: view };
  that.parent = null;
  that.getName = function() { return this.view.name; };
  return that;
}

function insertView(view) {
  var i;
  view.activeState = !!activeViews[view.id];
  i = userViews.insertChild(viewNode(view));
  if (i < 0) {
    mvc.logError("Duplicate view name: " + view.name);
  }
  return;
}

function getCurrentViewFromTree() {
  var nodes = $("#viewTree").treeView("selectionNodes");
  if (nodes && nodes.length > 0 && nodes[0].view) {
    return nodes[0].view;
  }
  return null;
}

function getCurrentViewAndNodeFromTree() {
  var $nodes = $("#viewTree").treeView("selection");
  var nodes = $("#viewTree").treeView("getNodes", $nodes);
  if (nodes.length > 0 && nodes[0].view) {
    return {node: nodes[0], view: nodes[0].view, uiNode: $nodes.eq(0)};
  }
  return null;
}

function storeView(view, fn) {
  var key;
  var data = {};

  if (!view) {
    view = currentView;
  }
  if (!view) {
    return;
  }
  if (view.fromTemplate) {
    return; // don't store a built-in view
  }
  key = "view_" + view.id;
  data[key] = JSON.stringify(view);

  writePrefs(data, fn);
}

var viewStoreTimerId = null;
var delayedView = null;

function delayedStoreView() {
  var view = currentView;
  if (viewStoreTimerId === null) {
    delayedView = view;
    viewStoreTimerId = setTimeout(function() {
      delayedView = null;
      viewStoreTimerId = null;
      storeView(view);
    }, DELAY_VIEW_WAIT_TIME);
  } else {
    if (view != delayedView) {
      // view has changed write previous 
      clearTimeout(viewStoreTimerId);
      viewStoreTimerId = null;
      storeView(delayedView);
      delayedView = null;
      delayedStoreView();
      return;
    }
    // else just wait
  }
}

// not quite a GUID but good enough since views are stored per user
function genViewId() {
  return "" + Math.random().toFixed(10).substring(2) + "-" + (new Date()).getTime();
}

function genViewName(name) {
  var newName = name;
  var count = 0;
  var match, i, len, node;

  do {
    count += 1;
    match = false;
    len = userViews.getChildCount();
    for (i = 0; i < len; i++) {
      node = userViews.getChild(i);
      if (node.view.name == newName) {
        match = true;
        newName = name + " " + count;
        break;
      }
    }
  } while (match);
  return newName;
}

var viewTreeAdapter = {
  root: function() {
    return viewsRoot;
  },
  getName: function(n) {
    return n.getName();
  },
  child: function(n, i) {
    return n.getChild(i);
  },
  childCount: function(n) {
    return n.getChildCount();
  },
  hasChildren: function(n) {
    return n.getChildCount && n.getChildCount() > 0;
  },
  renderIcon: function(n, out) {
    var icon;
    if (n.view) {
      icon = activeViews[n.view.id] ? "viewListActive" : "viewListInactive";
    } else {
      icon = n.categoryType && n.categoryType == "server" ? "viewListServer" : "viewListFolder";
    }
    if (icon) {
      mvc.renderIcon(out, icon);
    }
  },
  renderLabel: function(n, out) {
    out.markup("<span title='").attr(n.getName()).markup("'>").content(n.getName()).markup("</span>");
  },
  addNode: function(parent, name, context, fn) {
    // parent is userViews
    var view = context ? context : { name: name, id: genViewId(), panels: [ ] };
    view.name = name; // must set if context used
    var node = viewNode(view);
    var index = parent.insertChild(node);
    var out;

    if (index < 0) {
      out = mvc.htmlBuilder();
      out.markup("<p>").content(mvc.getMessage(BUNDLE, "viewlist.msg.dupView")).markup("</p>");
      mvc.doMessageDialog(mvc.getMessage(BUNDLE, "viewlist.msg.dupView.title"), out.toString(), function() {
        fn(false, -1);
      });
      return;
    }
    storeView(view, function(result) {
      if (!result) {
        parent.removeChild(index);
        fn(null, -1);
      } else {
        fn(node, index);
      }
    });
  },
  allowRename: function(n) {
    return (n.view && !n.view.fromTemplate);
  },
  allowDelete: function(n) {
    return (n.view && !n.view.fromTemplate);
  },
  renameNode: function(n, newName, fn) {
    var view, oldName, out, index;
    var p = n.parent;
    if (!p) {
      fn(null, -1);
      return;
    }
    if (p.findChildByName(newName) !== null) {
      out = mvc.htmlBuilder();
      out.markup("<p>").content(mvc.getMessage(BUNDLE, "viewlist.msg.dupViewRename")).markup("</p>");
      mvc.doMessageDialog(mvc.getMessage(BUNDLE, "viewlist.msg.dupView.title"), out.toString(), function() {
        fn(false, -1);
      });
      return;
    }
    n = p.removeChildByName(n.getName());
    view = n.view;
    if (!n || !view) {
      fn(null, -1);
      return;
    }
    oldName = view.name;
    view.name = newName;
    index = p.insertChild(n);

    storeView(view, function(result) {
      if (!result) {
        view.name = oldName;
        p.insertChild(n);
        fn(null, -1);
      } else {
        currentView = null; // forget current view so dashboard will get refreshed with new name on next selection change
        fn(n, index);
      }
    });
  },
  deleteNode: function(n, fn) {
    deleteView(n, fn);
  }
  
  // fetchChildNodes: not needed
};

function initViewTree() {
  builtinViews = folderNode(mvc.getMessage(BUNDLE, "viewlist.node.builtin"));
  userViews = folderNode(mvc.getMessage(BUNDLE, "viewlist.node.custom"));
  viewsRoot = folderNode("<root>"); // root - name isn't displayed
  viewsRoot.insertChild(builtinViews);
  viewsRoot.insertChild(userViews);
}

function insertTemplateView(view) {
  var node = builtinViews;
  var i;
  if (view.category) {
    node = builtinViews.findChildByName(view.category);
    if (!node) {
      i = builtinViews.insertChild(folderNode(view.category));
      node = builtinViews.getChild(i);
      node.categoryType = "server";
    }
  }
  view.activeState = !!activeViews[view.id];
  i = node.insertChild(viewNode(view));
  if (i < 0) {
    mvc.logError("Duplicate built-in view name: " + view.name);
  }
}

function initViewTemplates() {
  var paths = [];
  $.each(builtInViewTemplates, function(i, template) {
    paths.push(template.path);
  });

  mvc.ajax("GET", "data/diagnostics/instances", function(result) {
    var paths, path, i, j, templates, template, views, view, id;
    var $initialNode = null;
    var vars = {
      background: DEFAULT_BACKGROUND,
      color: DEFAULT_COLOR,
      highlight: DEFAULT_HIGHLIGHT,
      servers: null};

    if (result) {
      paths = {};
      $.each(result.body.paths, function(i, path) {
        paths[path.path] = path;
      });
      templates = builtInViewTemplates;
      id = 0;
      for (i = 0; i < templates.length; i++) {
        template = templates[i];
        path = paths[template.path];
        if (!path) {
          continue; // no views for that template
        }
        vars.servers = path.servers;
        views = processTemplate(template.template, vars);
        for (j = 0; j < views.length; j++) {
          view = views[j];
          id += 1;
          view.id = "builtin-" + id;
          view.fromTemplate = true;
          insertTemplateView(view);
        }
      }

      $("#viewTree").treeView({
        showRoot: false,
        expandRoot: false,
        nodeAdapter: viewTreeAdapter
      });

      // Make an initial selection in the tree. First try saved last view then settle for first view
      if (appState.lastViewId) {
        $initialNode = $("#viewTree").treeView("find", null, function(n) {
          return !!(n.view && n.view.id == appState.lastViewId);
        }, 10);
      }
      if (!$initialNode || $initialNode.length === 0) {
        $initialNode = $("#viewTree").treeView("find", null, function(n) {
          return !!n.view;
        }, 2);
      }
      if ($initialNode && $initialNode.length > 0) {
        $("#viewTree").treeView("selectNode", $initialNode);
      }
    }
  }, {paths: paths});
}

function newView() {
  var $parent = $("#viewTree").treeView("find", null, function(n) {
    return n === userViews;
  });
  var name = genViewName(mvc.getMessage(BUNDLE,"viewlist.newViewName"));
  $("#viewTree").treeView("addNode", $parent, name);
}

function cloneView() {
  var $parent = $("#viewTree").treeView("find", null, function(n) {
    return n === userViews;
  });
  var view = getCurrentViewFromTree();
  if (!view) {
    return;
  }
  var newView = $.extend(true, {}, view); // deep copy
  newView.name = genViewName(view.name);
  delete newView.fromTemplate; // no longer associated with a template
  newView.activeState = false;
  newView.id = genViewId();
  $("#viewTree").treeView("addNode", $parent, newView.name, newView);
}

function deleteView(n, fn) {
  var data, view, vn, node;
  if (n) {
    view = n.view;
    if (!view) {
      return;
    }
    node = n;
  } else {
    vn = getCurrentViewAndNodeFromTree();
    if (!vn) {
      return;
    }
    view = vn.view;
    node = vn.node;
  }
  stopView();
  data = { preferences: [ "view_" + view.id ] };

  mvc.ajax("MULTI_DELETE", "data/preferences/current/dashboard", function(result) {
    var p;
    if (result) {
      p = node.parent;
      p.removeChildByName(node.getName());
      currentView = null;
      $currentViewNode = null;
      appState.lastViewId = null;
      updateViewListAndMainActions();
      $("#viewArea").dashboardView("destroy");
      if (fn) {
        fn(true);
      } else {
        $("#viewTree").treeView("deleteNode", vn.uiNode);
      }
    } else {
      if (fn) {
        fn(false);
      }
    }
  }, data);
}

function renameView() {
  var view;
  var vn = getCurrentViewAndNodeFromTree();

  if (!vn) {
    return;
  }
  view = vn.view;
  $("#viewTree").treeView("renameNode", vn.uiNode);
}

function selectView() {
  var vn = getCurrentViewAndNodeFromTree();
  var view;
  if (!vn) {
    return;
  }
  view = vn.view;
  if (!view || view == currentView) {
    return;
  }
  currentView = view;
  $currentViewNode = vn.uiNode;
  $("#viewArea").dashboardView("destroy");
  $("#viewArea").dashboardView({
    state: view,
    dropAccept: ".browserMetric, .chartMetric",
    getViews: function() {
      var result = [];

      function viewWalk(node, fn) {
        var i, len;
        fn(node);
        if (node.getChildCount) {
          len = node.getChildCount();
          for (i = 0; i < len; i++) {
            viewWalk(node.getChild(i), fn);
          }
        }
      }
      viewWalk(userViews, function(node) {
        var view = node.view;
        if (view) {
          result.push({name: view.name, value: view.id});
        }
      });
      result.sort(function(a, b) {
        var sa = a.name, sb = b.name;
        return sa > sb ? 1 : sa < sb ? -1 : 0;
      });
      return result;
    },
    addPanelToView: function(panel, viewId, fn) {
      var view, panels, i, len, node;

      len = userViews.getChildCount();
      for (i = 0; i < len; i++) {
        node = userViews.getChild(i);
        if (node.view.id == viewId) {
          view = node.view;
          break;
        }
      }
      if (view) {
        panels = view.panels;
        // TODO make name unique
        panels.push([panel]);
        storeView(view, function(result) {
          fn(result !== null);
        });
      } else {
        mvc.logError("Move/copy panel to view could not find view");
        fn(false);
      }
    },
    adapter: {
      newPanelState: function(type, name) {
        if (type == "chart") {
          return {
            name: name,
            type: globalPrefs.defaultChartType,
            background: DEFAULT_BACKGROUND,
            color: DEFAULT_COLOR,
            highlight: DEFAULT_HIGHLIGHT,
            units: "",
            timeRange: "current",
            metrics: []
          };
        }
        return null;
      },
      getName: function(state) {
        return state.name;
      },
      setName: function(state, name) {
        state.name = name;
      },
      initPanel: function($e, type, state, container) {
        if (type == "chart") {
          $e.chart({
            dropAccept: ".browserMetric",
            getNewMetric: mbGetCurrentMetricValue,
            hasNewMetric: mbHasCurrentMetricValue,
            getPanels: container.getPanels,
            getPanel: container.getPanel,
            getActiveView: container.getActiveView,
            restrictChanges: view.fromTemplate === true,
            state: state
          });
        }
      },
      updatePanel: function($e, type, state) {
        if (type == "chart") {
          $e.chart("update");
        }
      }
    }
  });
  updateViewListAndMainActions();
  appState.lastViewId = view.id;
  storeAppState();
}

function startView() {
  var view = currentView;

  if (!activeViews[view.id]) {
    $("#viewArea").dashboardView("start");
    $currentViewNode.children(".icon").removeClass("viewListInactive").addClass("viewListActive");
    activeViews[view.id] = view;
    activeViewCount += 1;
  }
  updateViewListAndMainActions();
}

function stopView() {
  var view = currentView;

  if (activeViews[view.id]) {
    $("#viewArea").dashboardView("stop");
    $currentViewNode.children(".icon").removeClass("viewListActive").addClass("viewListInactive");
    delete activeViews[view.id];
    activeViewCount -= 1;
  }
  updateViewListAndMainActions();
}

function stopAllViews() {
  stopView(); // stop the current view normally
  var $nodes = $("#viewTree .viewListActive");

  $nodes.removeClass("viewListActive").addClass("viewListInactive");

  $.each(activeViews, function(k, view) {
    var rows, charts, metrics;
    var r, c, m;
    
    // for each panel row for each chart for each metric
    rows = view.panels; 
    for (r = 0; r < rows.length; r++) {
      charts = rows[r];
      for (c = 0; c < charts.length; c++) {
        metrics = charts[c].state.metrics;
        for (m = 0; m < metrics.length; m++) {
          wldfDM.unregister(metrics[m].metric, view.id);
        }
      }
    }
    view.activeState = false;
  });
  activeViews = {};
  activeViewCount = 0;
  updateViewListAndMainActions();
}

function updateViewListAndMainActions() {
  var $menu = $("#viewListMenu");
  var selView = getCurrentViewFromTree();
  var curView = currentView;
  var viewSelected = (selView !== null);
  var userViewSelected = !!(viewSelected && !selView.fromTemplate);
  var viewStopped = !!(curView !== null && activeViews[curView.id] === undefined);
  var viewStarted = !!(curView !== null && activeViews[curView.id] !== undefined);

  $.each([
    {id: "cloneView", sel: "#tbViewClone", d: !viewSelected},
    {id: "deleteView", sel: "#tbViewDelete", d: !userViewSelected},
    {id: "renameView", sel: null, d: !userViewSelected},
    {id: "start", sel: "#tbMainStart", d: !viewStopped},
    {id: "stop", sel: "#tbMainStop", d: !viewStarted},
    {id: "stopAll", sel: "#tbMainStopAll", d: activeViewCount === 0}
  ], function(i, item) {
    var menuItem = $menu.menu("find", item.id);
    if (menuItem) {
      menuItem.disabled = item.d;
    }
    if (item.sel) {
      $(item.sel).toggleClass("disabled", item.d).each(function() { this.disabled = item.d; });
    }
  });
}

function dataPollerStarted() {
  $("#stamps .stampPoller span").removeClass("pollerInactive").addClass("pollerActive").
    attr("title", mvc.getMessage(BUNDLE, "main.poller.active"));
}

function dataPollerStopped() {
  $("#stamps .stampPoller span").removeClass("pollerActive").addClass("pollerInactive").
    attr("title", mvc.getMessage(BUNDLE, "main.poller.inactive"));
}

// console help context
wls.console.pageHelpURL = null;
wls.console.pageHelpKey = 'clientdashboardpagetitle';

function metricBrowserResize() {
  var $explorer = $("#explorerTabs");
  var $mb = $("#metricBrowser");
  var h = $explorer.innerHeight() - parseInt($mb.css("paddingTop"),10) - parseInt($mb.css("paddingBottom"),10) - $explorer.find("ul:first").outerHeight();
  var w = $explorer.innerWidth() - parseInt($mb.css("paddingRight"),10) - parseInt($mb.css("paddingLeft"),10);

  h -= $mb.find(".filteredList").offset().top - $mb.offset().top;
  $mb.find("div:not(.filteredList) select").each(function() {
    $(this).width(w - 16 - $mb.find("#mbGo").outerWidth());
  });
  $mb.find(".filteredList").css({width: w - 6, height: (h / 3) - 6}).filteredList("resize"); 
}

function viewListResize() {
  var $explorer = $("#explorerTabs");
  var $mb = $("#viewList");
  var h = $explorer.innerHeight() - parseInt($mb.css("paddingTop"),10) - parseInt($mb.css("paddingBottom"),10) - $explorer.find("ul:first").outerHeight();

  h -= $mb.find(".ui-toolbar").outerHeight();

  $mb.find("#viewTree").height(h - 4);
}

function split(p, closed) {
  var w = document.documentElement.clientWidth - 7; // 7 is width of splitter
  if (!closed) {
    w = w - p;
    $("#explorer").show().width(p);
    if ($("#metricBrowser").css("display") != "none") {
      metricBrowserResize();
    } else if ($("#viewList").css("display") != "none") {
      viewListResize();
    }
  } else {
    $("#explorer").hide();
  }
  $("#content").width(w);
  $("#viewArea").dashboardView("resize");
}

function resizeMain() {
  var $e;
  var h = document.documentElement.clientHeight;
  h -= $("#hd").outerHeight();
  h -= $("#mainMenuBar").outerHeight();
  $("#mainContent").height(h);
  $("#explorer, #content, #mainContent > .splitterH").height(h);
  $("#viewArea").height(h - 1); // extra 1px for skip links
  $e = $("#explorerTabs");
  $.each(["borderTopWidth", "borderBottomWidth", "paddingTop", "paddingBottom"], function(i, p) {
    var v = parseInt($e.css(p),10);
    if (!isNaN(v)) {
      h -= v;
    }
  });
  $e.height(h - 1); // extra 1px for rounding of parseInt
  split($("#explorer").width(), !$("#explorer").is(":visible"));
}

$(document).ready(function() {
  var out = mvc.htmlBuilder();

  mvc.ajaxOptions({
    processing: function(busy) {
      $("#pageStatus").toggleClass("busy", busy);
      $("body").css("cursor", busy ? "wait" : "auto");
    },
    sessionExpired: function() {
      loggedIn = false;
      $("#user").text("");
      $("#domain").text("");
      var out = mvc.htmlBuilder();
      out.markup("<p>").content(mvc.getMessage(BUNDLE, "message.status.expiredSession")).markup("</p>");
      out.markup("<p><a href='#' onclick='window.location.reload();'>").content(mvc.getMessage(BUNDLE, "main.login.link")).markup("</a></p>");
      mvc.doMessageDialog(mvc.getMessage(BUNDLE, "main.sessionExpired.title"), out.toString());
    }
  });

  //
  // localize static content
  //
  var bundles = [BUNDLE, "mvccore"];
  document.title = mvc.getMessage(bundles, "dashboard.title");
  $(".label.dol10n, label.dol10n").each(function() {
    var key = $(this).html();
    var target = $(this).attr("for");
    var type = "";
    var punctuation;

    if (target) {
      type = $("#" + target).attr("type");
    }
    punctuation = (type == "checkbox" || type == "radio") ? "" : ":";
    $(this).html(mvc.getMessage(bundles, key) + punctuation).removeClass("dol10n");
    key = $(this).attr("title");
    if (key) {
      $(this).attr("title", mvc.getMessage(bundles, key));
    }
  });
  $("a.dol10n, span.dol10n, button.dol10n").each(function() {
    var key = $(this).html();
    if (key && key.indexOf("<") == -1) {
      $(this).html(mvc.getMessage(bundles, key)).removeClass("dol10n");
    }
    key = $(this).attr("title");
    if (key) {
      $(this).attr("title", mvc.getMessage(bundles, key)).removeClass("dol10n");
    }
  });
  $("input.dol10n").each(function() {
    var key = $(this).attr("title");
    $(this).attr("title", mvc.getMessage(bundles, key)).removeClass("dol10n");
  });
  $("p.dol10n, option.dol10n, legend.dol10n").each(function() {
    var key = $(this).html();
    $(this).html(mvc.getMessage(bundles, key)).removeClass("dol10n");
  });
  $("img.dol10n").each(function() {
    var key = $(this).attr("alt");
    $(this).attr("alt", mvc.getMessage(bundles, key) + " ").removeClass("dol10n");
    key = $(this).attr("title");
    if (key) {
      $(this).attr("title", mvc.getMessage(bundles, key) + " ").removeClass("dol10n");
    }
  });

  $("#pageStatus").removeClass("busy");
  $("#logoutLink").attr("href", mvc.getContextRoot() + "/jsp/common/warnuserlockheld.jsp");

  //
  // Init toolbar/menubar
  //
  out.clear();
  mvc.renderButton(out, {id: "tbMainStart", icon: "start", nb: true, tooltip: mvc.getMessage(BUNDLE, "main.start")});
  mvc.renderButton(out, {id: "tbMainStop", icon: "stop", nb: true, tooltip: mvc.getMessage(BUNDLE, "main.stop")});
  mvc.renderButton(out, {id: "tbMainStopAll", icon: "stopAll", nb: true, tooltip: mvc.getMessage(BUNDLE, "main.stopAll")});
  mvc.renderButton(out, {id: "tbMainMenu", icon: "menu", nb: true, tooltip: mvc.getMessage(BUNDLE, "main.menu")});
  $("#mainMenuBar .ui-toolbar").html(out.toString());

  getInitialPageData();

  $("#tbMainStart").click(function() {
    startView();
  });
  $("#tbMainStop").click(function() {
    stopView();
  });
  $("#tbMainStopAll").click(function() {
    stopAllViews();
  });
  $("#tbMainMenu").click(function() {
    var $menu = $("#mainMenu");
    $menu.menu("toggle", $(this));
  });

  var mainMenu = {
    menubar: false,
    items: [
      { id: "clear", type: "action", label: mvc.getMessage(BUNDLE, "main.menu.clearData"), action: function() { wldfDM.clear(); } },
      { id: "reset", type: "action", label: mvc.getMessage(BUNDLE, "main.menu.resetStats"), action: function() { wldfDM.resetStats(); } }
    ]
  };
  $("#mainMenu").menu(mainMenu);

  //
  // Init tabs
  //
  $("#explorerTabs").tabs({
    show: function(e, ui) {
      if (ui.panel.id == "metricBrowser") {
        metricBrowserResize();
        getRunningServers();
      } else if (ui.panel.id == "viewList") {
        viewListResize();
      }
    }
  }).show();

  //
  // View List Tab
  //

  // view toolbar
  out.clear();
  mvc.renderButton(out, {id: "tbViewNew", icon: "new", nb: true, tooltip: mvc.getMessage(BUNDLE, "viewlist.menu.new")});
  mvc.renderButton(out, {id: "tbViewClone", icon: "clone", nb: true, tooltip: mvc.getMessage(BUNDLE, "viewlist.menu.clone")});
  mvc.renderButton(out, {id: "tbViewDelete", icon: "delete", nb: true, tooltip: mvc.getMessage(BUNDLE, "viewlist.menu.delete")});
  mvc.renderButton(out, {id: "tbViewMenu", icon: "menu", nb: true, tooltip: mvc.getMessage(BUNDLE, "viewlist.menu")});
  $("#viewList .ui-toolbar").html(out.toString());

  $("#tbViewNew").click(function() {
    newView();
  });
  $("#tbViewClone").click(function() {
    cloneView();
  });
  $("#tbViewDelete").click(function() {
    deleteView();
  });

  $("#tbViewMenu").click(function() {
    var $menu = $("#viewListMenu");
    updateViewListAndMainActions();
    $menu.menu("toggle", $(this));
  });

  var viewListMenu = {
    menubar: false,
    items: [
      { id: "newView", type: "action", label: mvc.getMessage(BUNDLE, "viewlist.menu.new"), icon: "new", action: function() { newView(); return true;} },
      { id: "cloneView", type: "action", label: mvc.getMessage(BUNDLE, "viewlist.menu.clone"), icon: "clone", action: function() { cloneView(); return true;} },
      { id: "deleteView", type: "action", label: mvc.getMessage(BUNDLE, "viewlist.menu.delete"), icon: "delete", action: function() { deleteView(); } },
      { id: "renameView", type: "action", label: mvc.getMessage(BUNDLE, "viewlist.menu.rename"), action: function() { renameView(); return true; } },
      { type: "separator" },
      { type: "action", label: mvc.getMessage(BUNDLE, "viewlist.menu.refresh"), action: function() { loadGlobalPrefsAndViews(); } }
    ]
  };
  $("#viewListMenu").menu(viewListMenu);

  initViewTree();
  // tree view doesn't get created untill after we have all the nodes which is after loading templates

  $("#viewTree").bind("treeViewchange", function() {
    updateViewListAndMainActions();
    selectView();
  });

  //
  // Metric Browser Tab
  //
  $("#mbTypesFL").filteredList({
    itemAdapter: metricListAdapter(),
    label: mvc.getMessage(BUNDLE, "mb.types"),
    filterOnTitle: false
  }).bind("filteredListchange", mbTypesChanged);
  $("#mbInstancesFL").filteredList({
    itemAdapter: metricListAdapter("metricInstance"),
    label: mvc.getMessage(BUNDLE, "mb.instances"),
    filterOnTitle: false
  }).bind("filteredListchange", mbInstancesChanged);
  $("#mbMetricsFL").filteredList({
    itemAdapter: metricListAdapter("metricAttr"),
    itemLabel: "browserMetric",
    label: mvc.getMessage(BUNDLE, "mb.metrics"),
    filterOnTitle: false,
    enableDrag: false,
    draggable: {
      addClasses: false,
      cursor: "default",
      opacity: 0.8,
      distance: 5,
      revert: "invalid",
      appendTo: "#mainContent",
      containment: "#mainContent",
      zIndex: 950,
      helper: function() {
        var metric = mbCurrentMetric();
        var instance = mbCurrentInstance();
        var out = mvc.htmlBuilder();
        var name = getMetricName(metric, instance);

        out.markup("<div class='dragMetric'>").content(name).markup("</div>");
        return $(out.toString());
      }
    }
  }).bind("filteredListchange", mbMetricsChanged);

  mvc.bindFocus($("#mbTypesFL").get(0), function(e) {
    var $target = $(e.target).closest(".select");
    var type = mbCurrentType();
    if (type !== null) {
      displayTypeNote($target, type);
    }
  });
  mvc.bindBlur($("#mbTypesFL").get(0), function(e) {
    mvc.hideNote();
  });

  mvc.bindFocus($("#mbInstancesFL").get(0), function(e) {
    var $target = $(e.target).closest(".select");
    var instance = mbCurrentInstance();
    if (instance) {
      displayInstanceNote($target, instance);
    }
  });
  mvc.bindBlur($("#mbInstancesFL").get(0), function(e) {
    mvc.hideNote();
  });

  mvc.bindFocus($("#mbMetricsFL").get(0), function(e) {
    var $target = $(e.target).closest(".select");
    var metric = mbCurrentMetric();
    if (metric) {
      displayMetricNote($target, metric);
    }
  });
  mvc.bindBlur($("#mbMetricsFL").get(0), function(e) {
    mvc.hideNote();
  });

  $("#mbHarvestedOnly").click(function() {
    var checked = $(this).get(0).checked;
    $("#mbAllTypes").get(0).disabled = checked;
    updateMetricBrowserLists();
  });

  $("#mbAllTypes").click(function() {
    updateMetricBrowserLists();
  });

  $("#mbServers").keypress(function(e) {
    if (e.keyCode == VK_ENTER) {
      $("#mbGo").click();
    }
  });
  $("#mbGo").click(function() {
    var server = $("#mbServers").val();
    if (server) {
      getMetricsForServer(server);
    }
  });

  //
  // Global init
  //
  mvc.initIcons();

  mvc.bindResize(resizeMain);
  resizeMain();

  $("#global-links a:eq(0)").click(function() {
    var $dialog = $("#preferencesDialog");
    var fh = mvc.formHelper(globalPrefFields, { formContext: $dialog });

    var curOverview = globalPrefs.showOverview;
    mvc.doModalDialog($dialog, { 
      title: mvc.getMessage(BUNDLE, "main.preferences.title"),
      formHelper: fh,
      formData: globalPrefs,
      ok: function() {
        updateGlobalPrefs(true);
        if (curOverview != globalPrefs.showOverview) {
          $("#viewArea .chart").chart("updateOverviewState");
        }
      },
      focusElemAfter: this
    });
    return false;
  });

  // help
  $("#global-links a:eq(1)").click(function() {
    return wls.console.launchHelp();
  });

  mvc.splitter({
    orientation: "horizontal",
    edge: "left",
    after: "#explorer",
    closed: false,
    collapseText: mvc.getMessage("mvccore", "button.collapsePane.label"),
    restoreText: mvc.getMessage("mvccore", "button.restorePane.label"),
    change: function(pos, closed) {
      split(pos, closed);
      if (!closed) {
        appState.splitterWidth = pos;
      }
      appState.splitterClosed = closed;
      storeAppState();
    }
  });

  // preferences
  loadGlobalPrefsAndViews();

  $("#viewArea").bind("chartstatechange", function(e) {
    delayedStoreView();
  }).bind("dashboardViewstatechange", function(e) {
    delayedStoreView();
  });

  dataPollerStopped();

  wldfDM.bind(function(type) {
    if (type == "start") {
      dataPollerStarted();
    } else if (type == "stop") {
      dataPollerStopped();
    } else if (type == "newData") {
      if ($("#viewArea").dashboardView("isActive")) {
        $("#viewArea").dashboardView("update");
      }
    } else if (type == "harvestedData") {
      $("#viewArea").dashboardView("update");
    }
  });

  window.onbeforeunload = function() {
    if (wldfDM.hasData()) {
      return mvc.getMessage(BUNDLE, "main.unsavedchanges");
    }
  };

});

})(jQuery); 
