/*!
  UI widget for dashboard chart
  Copyright (c) 2009, 2010, Oracle and/or its affiliates. All rights reserved.
*/
/*
  Depends on external markup for chart dialog, #chartDialog, and metric dialog, #chartMetricDialog.
*/
/*global window,jQuery,mvc,wldfDM */
(function($) {

var VK_ENTER = 13,
    VK_ESC = 27,
    VK_LEFT = 37,
    VK_UP = 38,
    VK_RIGHT = 39,
    VK_DOWN = 40,
    VK_NP_PLUS = 107, // + on numpad
    VK_PLUS = 187,  // shift = 
    VK_NP_MINUS = 109, // - on numpad
    VK_MINUS = 189,
    VK_ZERO = 48,
    VK_DEL = 46,
    VK_F2 = 113;

var ONEDAY = 24*60*60; // seconds
var ONEHOUR = 60*60; // seconds
var SELECTION_COLOR = "#888888"; // should get style from CSS
var BUNDLE = "dashboard";
var MIN_LEGEND_HEIGHT = 60;
var MIN_CHART_HEIGHT = 100;
var MIN_CHART_WIDTH = 300;

var DATA_PLOT = "plot";
var DATA_OVERVIEW_PLOT = "overviewPlot";
var DATA_GAUGE = "gauge";
var DATA_TIME_AXIS = "timeAxis";
var DATA_CURPOINT = "curPoint";
var DATA_SHOW_CURPOINT = "showCurPoint";
var DATA_EDIT_MODE = "editMode";

var TYPE_HGAUGE = "linearHGauge";
var TYPE_VGAUGE = "linearVGauge";
var TYPE_RGAUGE = "radialGauge";
var TYPE_BAR = "barChart";
var TYPE_LINE = "linePlot";
var TYPE_SCATTER = "scatterPlot";

var EVENT_STATE_CHANGE = "statechange";

var CLASS_CHART = "chart";
var CLASS_SELECTED = "selected";

var SEL_FOCUSABLE = "h3, button, .chartMain, .chartOverview, .legendLabel";

// Global defaults apply to all charts
var defaultTimeRange = 5 * 60; // 5 minutes
var zoomPercent = 20;
var showOverview = "S"; // one of A = always, N = never, S = when chart selected

// metric marker colors and shapes
var markers = [
  { color: "#FFCC33", shape: "circle", fill: true },
  { color: "#33CCFF", shape: "diamond", fill: true },
  { color: "#CC3333", shape: "triangleUp", fill: true },
  { color: "#669966", shape: "triangleDown", fill: true },
  { color: "#666699", shape: "square", fill: true },
  { color: "#009900", shape: "bar", fill: true },
  { color: "#990000", shape: "halfCircleLeft", fill: true },
  { color: "#000099", shape: "halfCircleRight", fill: true },
  { color: "#999900", shape: "circle", fill: false },
  { color: "#009999", shape: "diamond", fill: false },
  { color: "#990099", shape: "triangleUp", fill: false },
  { color: "#FF0099", shape: "triangleDown", fill: false },
  { color: "#00FF99", shape: "square", fill: false },
  { color: "#9900FF", shape: "bar", fill: false },
  { color: "#0099FF", shape: "halfCircleLeft", fill: false },
  { color: "#99FF00", shape: "halfCircleRight", fill: false }
];

function initMetricMarker(metric, metrics, mi) {
  var ci;

  function shapeInUse(shape, fill) {
    var i, m;
    for (i = 0; i < metrics.length; i++) {
      m = metrics[i];
      if (m !== metric && shape == m.markerShape && fill === m.markerFill ) {
        return true;
      }
    }
    return false;
  }

  if (!metric.color || !metric.markerShape ||
    (metric.markerShape && shapeInUse(metric.markerShape, metric.markerFill))) {
    // Choose a unique marker
    if (metrics.length >= markers.length) {
      // all shapes must be in use so change the colors
      ci = mi + Math.floor(metrics.length / markers.length);
      ci = ci % markers.length;
    } else {
      while (shapeInUse(markers[mi].shape, markers[mi].fill)) {
        mi += 1;
        if (mi >= markers.length) {
          mi = 0;
        }
      }
      ci = mi;
    }
    metric.color = markers[ci].color;
    metric.markerShape = markers[mi].shape;
    metric.markerFill = markers[mi].fill;
  }

  mi += 1;
  if (mi >= markers.length) {
    mi = 0;
  }

  return mi;
}

var chartFields = [
  { id: "chartTitle",
    prop: "name",
    constraints: ["required"]
  },
  {
    prop: "type",
    constraints: [{fn:"required"}, {fn: "member", args: [["linearHGauge", "linearVGauge", "radialGauge", "barChart", "linePlot", "scatterPlot"]]}]
  },
  { id: "chartUnits",
    prop: "units"
  },
  { id: "chartColor",
    prop: "color",
    constraints: ["required", "color"]
  },
  { id: "chartBackgroundColor",
    prop: "background",
    constraints: ["required", "color"]
  },
  { id: "chartHighlightColor",
    prop: "highlight",
    constraints: ["required", "color"]
  },
  { id: "chartAutoRange",
    prop: "autoRange",
    type: "boolean"
  },
  { id: "chartRangeMax",
    prop: "rangeMax",
    type: "number",
    constraints: ["required", "number"]
  },
  { id: "chartRangeMin",
    prop: "rangeMin",
    type: "number",
    constraints: ["required", "number"]
  },
  { id: "chartThresholdMax",
    prop: "thresholdMax",
    type: "number",
    constraints: ["number"]
  },
  { id: "chartThresholdMin",
    prop: "thresholdMin",
    type: "number",
    constraints: ["number"]
  },
  { name: "chartRangeType",
    prop: "timeRange",
    constraints: [{fn: "member", args: [["current", "custom"]]}]
  },
  { id: "chartTimeRangeStart",
    prop: "timeStart",
    type: "date",
    formatter: {fn: "datetime", args: []},
    constraints: ["required", {fn: "datetime", args: []}]
  },
  { id: "chartTimeRangeDuration",
    prop: "timeDuration",
    formatter: {fn: "deltatime", args: ["m"]},
    constraints: ["required", "deltatime"]
  }
];

var metricFields = [ // just for validation
  { prop: "markerShape",
    constraints: [{fn: "member", args: [["circle", "diamond", "triangleUp", "triangleDown", "square", "bar", "halfCircleLeft", "halfCircleRight"]]}]
  },
  { prop: "markerFill",
    type: "boolean",
    constraints: [{fn: function(value) {
      value = value.toString();
      if (value == "true" || value == "false") {
        return null;
      }
      return "Must be a boolean value";
    } }]
  },
  { prop: "color",
    constraints: ["color"]
  }
];

function typeIsGauge(type) {
  return type == TYPE_HGAUGE || type == TYPE_VGAUGE || type == TYPE_RGAUGE;
}

function formatDate(d) {
  return mvc.formatDate(d, { fullYear: true });
}

function formatTime(d) {
  return mvc.formatTime(d, { });
}

function formatDateTime(d) {
  return mvc.formatDateTime(d, { fullYear: true });
}

function tickFormatTime(ms) {
  var d = new Date(ms);
  return mvc.formatTime(d, {excludeAmPm: true });
}

function tickFormatDateTime(ms) {
  var d = new Date(ms);
  return mvc.formatDate(d, { fullYear: true }) + "<br>" + mvc.formatTime(d, { excludeAmPm: true });
}

function leftPad(n) {
  n = "" + n;
  return n.length == 1 ? "0" + n : n;
}

function formatDeltaTime(sec) {
  var days = Math.floor(sec / ONEDAY);
  sec -= days * ONEDAY;
  var hours = Math.floor(sec / ONEHOUR);
  sec -= hours * ONEHOUR;
  var minutes = Math.floor(sec / 60);
  sec -= minutes * 60;
  sec = Math.floor(sec);
  return days + " " + leftPad(hours) + ":" + leftPad(minutes) + ":" + leftPad(sec);
}

function formatValue(v) {
  return mvc.formatNumber(v);
}

var selecting = false;
var updateAfterSelect = false;

function showMetricPointDetails(item, $for) {
  var sampleTime = new Date(item.series.data[item.dataIndex][0]); // use the original data. datapoints may be adjusted
  var note = mvc.formatNoteContent(BUNDLE, [
    { label: "chart.note.name", value: item.series.label },
    { label: "chart.note.date", value: formatDate(sampleTime) },
    { label: "chart.note.time", value: formatTime(sampleTime) },
    { label: "chart.note.value", value: formatValue(item.datapoint[1]) }
  ]);
  mvc.showNote(note, $for, item.pageX, item.pageY);
}

function showMetricPoint(plot, curPoint, $for) {
  var series, item, s, p, ps, offset, pt;
  series = plot.getData();
  s = series[curPoint.series];
  p = curPoint.point;
  ps = s.datapoints.pointsize;
  pt = s.datapoints.points.slice(p * ps, (p + 1) * ps);
  offset = plot.offset();
  item = {
    datapoint: pt,
    dataIndex: p,
    series: s,
    seriesIndex: curPoint.series,
    pageX: parseInt(s.xaxis.p2c(pt[0]),10) + offset.left,
    pageY: parseInt(s.yaxis.p2c(pt[1]),10) + offset.top
  };
  plot.unhighlight();
  plot.highlight(s, pt);
  showMetricPointDetails(item, $for);
}

function showGaugeDataDetails(item, $for) {
  var note = mvc.formatNoteContent(BUNDLE, [
    { label: "chart.note.name", value: item.data.label },
    { label: "chart.note.from", value: formatDateTime(new Date(item.data.since)) },
    { label: "chart.note.to", value: formatDateTime(new Date(item.data.time)) },
    { label: "chart.note.value", value: formatValue(item.data.value) },
    { label: "chart.note.min", value: formatValue(item.data.min) },
    { label: "chart.note.max", value: formatValue(item.data.max) },
    { label: "chart.note.avg", value: formatValue(item.data.avg) },
    { label: "chart.note.stdev", value: formatValue(item.data.stdev) }
  ]);
  mvc.showNote(note, $for, item.pageX, item.pageY);
}

function showGaugeData(gauge, curPoint, $for) {
  var data, item, i, offset, pos;

  data = gauge.data;
  i = curPoint.series;
  item = data[i];
  offset = gauge.offset();
  pos = gauge.getValuePos(i, item.value);
  item = {
    data: item,
    index: i,
    pageX: pos.x + offset.left,
    pageY: pos.y + offset.top
  };
  gauge.unhighlight();
  gauge.highlight(i);
  showGaugeDataDetails(item, $for);
}

function showOverviewNote(plot, timeAxis, $for) {
  var i, data, count, maxCount = 0, last, note;
  var series = plot.getData();
  var totalRange = 0;
  var minTime = Number.MAX_VALUE;
  var maxTime = 0;

  for (i = 0; i < series.length; i++) {
    data = series[0].datapoints;
    count = data.points.length / data.pointsize;
    if (count > maxCount) {
      maxCount = count;
    }
    if (data.points[0] < minTime) {
      minTime = data.points[0];
    }
    last = data.points.length - data.pointsize;
    if (data.points[last] > maxTime) {
      maxTime = data.points[last];
    }
  }
  if (maxCount > 0) {
    totalRange = formatDeltaTime((maxTime - minTime) / 1000);
    note = mvc.formatNoteContent(BUNDLE, [
      { label: "chart.note.samples", value: maxCount },
      { label: "chart.note.range", value: formatDeltaTime(timeAxis.range) },
      { label: "chart.note.totalRange", value: totalRange },
      { label: "chart.note.from", value: formatDateTime(new Date(minTime)) },
      { label: "chart.note.to", value: formatDateTime(new Date(maxTime)) }
    ]);
    mvc.showNote(note, $for);
  } else {
    note = mvc.formatNoteContent(BUNDLE, [
      { label: "chart.note.overviewEmpty", value: null },
      { label: "chart.note.range", value: formatDeltaTime(timeAxis.range) }
    ]);
    mvc.showNote(note, $for);
  }
}

function makeCurPointVisible(curPoint, xaxis, data) {
  var x, y, i;
  var visible = true;

  x = data[curPoint.point][0];
  y = data[curPoint.point][1];
  if (y === null) {
    if (curPoint.point < data.length - 1) {
      curPoint.point += 1;
    } else {
      curPoint.point -= 1;
    }
  }
  if (x < xaxis.min) {
    visible = false;
    for (i = curPoint.point; i < data.length; i++) {
      x = data[i][0];
      y = data[i][1];
      if (x >= xaxis.min && y !== null) {
        visible = true;
        curPoint.point = i;
        break;
      }
    }
  } else if (x > xaxis.max) {
    visible = false;
    for (i = curPoint.point; i >=0; i--) {
      x = data[i][0];
      y = data[i][1];
      if (x <= xaxis.max && y !== null) {
        visible = true;
        curPoint.point = i;
        break;
      }
    }
  }
  return visible;
}

function getChartMenu(id) {
  return $("#" + id + "ChartMenu");
}

function getMetricMenu(id) {
  return $("#" + id + "MetricMenu");
}

function canvasCleanup($ph) {
  // can't use .html("") because IE needs the canvas removed with innerHTML
  // but just in case make sure jQuery stuff is cleaned up 
  $ph.find("canvas").each(function(){
    $.event.remove(this);
    $.removeData(this);
  });
  $ph.each(function() {
    this.innerHTML = "";
  });
}

function paintMarker($placeholder, bg, shape, color, fill) {
  var canvas, ctx;

  if (window.G_vmlCanvasManager) {
    window.G_vmlCanvasManager.init_(document); // init excanvas if present
  }

  canvas = document.createElement("canvas");
  canvas.width = 12;
  canvas.height = 12;
  if (window.G_vmlCanvasManager) {
    canvas = window.G_vmlCanvasManager.initElement(canvas);
  }

  $placeholder.css({"position": "relative"});
  $placeholder.html("").append(canvas);
  ctx = canvas.getContext("2d");
  if (bg) {
    ctx.fillStyle = bg;
    ctx.fillRect(0, 0, 12, 12);
  }
  $.plot.markers.draw(ctx, shape, 6, 6, 4, 2, 0, color, fill);
}

$.widget("ui.chart", {

  _init: function() {
    var self = this;
    var o = this.options;
    var state;
    var $ctrl = this.element;
    var id = $ctrl.get(0).id;
    var prevHoverIndex = null;
    var prevHoverSIndex = null;
    var out = mvc.htmlBuilder();
    var now, range, i, metrics;

    function rangeSelected(e, ranges) {
      var plot;
      if (!selecting) {
        return;
      }
      selecting = false;
      var a = ranges.xaxis;
      if (a) {
        self.setRange(a.from, Math.round((a.to - a.from) / 1000));
        updateAfterSelect = false; // setRange does an update
      } else {
        plot = self._getData(DATA_OVERVIEW_PLOT);
        plot.clearSelection(true);
        plot = self._getData(DATA_PLOT);
        plot.clearSelection(true);
      }
    }

    function rangeSelecting(e, ranges) {
      if (ranges) {
        selecting = true;
      }
    }

    id = id ? id : "";
    // apply state defaults. Don't use $.extend because state must be a reference to underlying view state - not a copy
    $.each($.ui.chart.defaults.state, function(prop, value) {
      if (o.state[prop] === undefined) {
        o.state[prop] = value;
      }
    });
    state = o.state;
    // init metric markers
    metrics = state.metrics;
    for (i = 0; i < metrics.length; i++) {
      state.markerIndex = initMetricMarker(metrics[i], metrics, state.markerIndex);
    }

    o.isCustomRange = false;
    o.viewportRange = defaultTimeRange;
    now = new Date();
    now = now.getTime() - now.getTimezoneOffset() * 60 * 1000;
    this._setData(DATA_TIME_AXIS, {offset: 0, range: o.viewportRange, maxRange: o.viewportRange, max: now, min: now - (o.viewportRange * 1000) });

    range = self._getRange();
    if (range) {
      self._registerRange(range); // updates DATA_TIME_AXIS, viewportRange, isCustomRange, rangeUpdated
    }

    out.markup("<div class='dashPanelHeader'><h3 tabindex='-1'>").content(state.name).markup("</h3><div class='headerBtns'>");
    mvc.renderButton(out, {id: id + "Edit", icon: "edit", nb: true, tooltip: mvc.getMessage(BUNDLE, "chart.edit")});
    mvc.renderButton(out, {id: id + "Menu", icon: "menu", nb: true, tooltip: mvc.getMessage(BUNDLE, "chart.menu")});
    out.markup("</div></div><div class='chartArea'><p class='chartYaxis'><span class='yAxisUnits'>").content(state.units).
      markup("</span>&nbsp;<span class='yAxisMsg'></span></p>");
    out.markup("<div class='chartMain' tabindex='-1'></div></div>");
    out.markup("<div class='chartCtrls' style='overflow:hidden;width:100%;'><div class='chartLegend' style='float:left;'></div>");
    out.markup("<div class='overview' style='float:right;'><div class='chartOverview' tabindex='-1'></div>");
    out.markup("<div class='chartActions ui-toolbar'>");
    mvc.renderButton(out, {id: id + "PanLeft", icon: "panLeft", tooltip: mvc.getMessage(BUNDLE, "chart.panLeft")});
    mvc.renderButton(out, {id: id + "ZoomIn", icon: "zoomIn", tooltip: mvc.getMessage(BUNDLE, "chart.zoomIn")});
    mvc.renderButton(out, {id: id + "ZoomOut", icon: "zoomOut", tooltip: mvc.getMessage(BUNDLE, "chart.zoomOut")});
    mvc.renderButton(out, {id: id + "PanRight", icon: "panRight", tooltip: mvc.getMessage(BUNDLE, "chart.panRight")});
    out.markup("</div></div></div>");

    $ctrl.html(out.toString()).addClass(CLASS_CHART).find(".chartArea").css({
      backgroundColor: state.background,
      color: state.color
    });
    mvc.initIcons($ctrl);
    o.selectedMetricIndex = -1;

    out.clear();
    out.markup("<div id='").attr(id + "ChartMenu").markup("'></div>");
    $(out.toString()).appendTo("body").hide();
    out.clear();
    out.markup("<div id='").attr(id + "MetricMenu").markup("'></div>"); // one for all metrics
    $(out.toString()).appendTo("body").hide();
    this._initMenus();

    $("#" + id + "ZoomIn").click(function() {
      self.zoomIn();
    });
    $("#" + id + "ZoomOut").click(function() {
      self.zoomOut();
    });
    $("#" + id + "PanLeft").click(function() {
      self.panLeft();
    });
    $("#" + id + "PanRight").click(function() {
      self.panRight();
    });
    $("#" + id + "Edit").click(function() {
      self._editMode($(this));
    });
    $("#" + id + "Menu").click(function() {
      var $menu = getChartMenu(id);

      $menu.menu("find", "addMetric").disabled = o.restrictChanges ? o.restrictChanges : !o.hasNewMetric();
      $menu.menu("toggle", $(this));
    });

    $ctrl.find(".chartMain").bind("plothover", function (event, pos, item) {
      var plot = self._getData(DATA_PLOT);
      if (item) {
        if (prevHoverIndex !== item.dataIndex || prevHoverSIndex !== item.seriesIndex) {
          prevHoverIndex = item.dataIndex;
          prevHoverSIndex = item.seriesIndex;
          plot.unhighlight();
          plot.highlight(item.series, item.datapoint);
          showMetricPointDetails(item, $(this));
        }
      }
      else {
        plot.unhighlight();
        mvc.hideNote();
        prevHoverIndex = null;
        prevHoverSIndex = null;
      }
    }).bind("plotclick", function(event, pos, item) {
      var curPoint;
      if (item) {
        curPoint = {series: item.seriesIndex, point: item.dataIndex};
        self._setData(DATA_CURPOINT, curPoint);
      }
    }).bind("gaugehover", function (event, pos, item) {
      var gauge = self._getData(DATA_GAUGE);
      if (item) {
        if (prevHoverIndex !== item.index) {
          prevHoverIndex = item.index;
          gauge.unhighlight();
          gauge.highlight(item.index);
          showGaugeDataDetails(item, $(this));
        }
      }
      else {
        gauge.unhighlight();
        mvc.hideNote();
        prevHoverIndex = null;
      }
    }).bind("gaugeclick", function(event, pos, item) {
      // TODO handle click on gauge
    }).mouseleave(function() {
      mvc.hideNote();
      prevHoverIndex = null;
      prevHoverSIndex = null;
    }).bind("focus", function(e) {
      var curPoint, data, i, m, ms;
      var range;
      curPoint = self._getData(DATA_CURPOINT);
      if (!curPoint) {
        range = self._getRange();
        // make a good guess at a starting point
        for (i = 0; i < state.metrics.length; i++) {
          m = state.metrics[i];
          ms = wldfDM.getMetricSeries(m.metric);
          if (ms) {
            if (range) {
              data = ms.harvested[range].data;
            } else {
              data = ms.data;
            }
            if (data.length > 0) {
              curPoint = {series: i, point: data.length - 1};
              break;
            }
          }
        }
        if (curPoint) {
          self._setData(DATA_CURPOINT, curPoint);
        }
      }
      if (curPoint) {
        self._setData(DATA_SHOW_CURPOINT, true);
        self._showCurrentMetricPoint();
      }
    }).bind("blur", function(e) {
      var plot = self._getData(DATA_PLOT); // TODO handle gauge also
      if (plot) {
        plot.unhighlight();
        mvc.hideNote();
      }
      self._setData(DATA_SHOW_CURPOINT, false);
    }).bind("plotselecting", rangeSelecting).bind("plotselected", rangeSelected);

    $ctrl.find(".chartOverview").hover(function(event) {
      var plot = self._getData(DATA_PLOT);
      var timeAxis = self._getData(DATA_TIME_AXIS);
      showOverviewNote(plot, timeAxis, $(this));
    }, function() {
      mvc.hideNote();
    }).bind("focus", function(e) {
      var plot = self._getData(DATA_PLOT);
      var timeAxis = self._getData(DATA_TIME_AXIS);
      showOverviewNote(plot, timeAxis, $(this));
    }).bind("blur", function(e) {
      mvc.hideNote();
    }).bind("plotunselected", function(event) {
      // put the selection back the way it was
      var plot = self._getData(DATA_PLOT);
      var xaxis = plot.getAxes().xaxis;
      var overviewPlot = self._getData(DATA_OVERVIEW_PLOT);
      var range = {xaxis: {from: xaxis.min, to: xaxis.max}};
      overviewPlot.setSelection(range);
    }).bind("plotselecting", rangeSelecting).bind("plotselected", rangeSelected);

    this._internalUpdateOverviewState(false);
    this._renderLegend();
    this.resize();
    // initially unselected
    this.unSelect(); // should be no change in overview state so there will be no extra resize

    $ctrl.bind("keydown.chart", function(e) {
      var kc, index, $li;
      var isMain = false;
      var isGauge = false;
      var $target = $(e.target);
      var $focus;

      if (e.altKey || $target.is("input, select, textarea")) {
        return;
      }
      kc = e.keyCode;
      isGauge = typeIsGauge(state.type);
      if (e.ctrlKey) {
        isMain = $(e.target).hasClass("chartMain");
        if (kc == VK_LEFT) {
          if (!isGauge) {
            if (isMain) {
              self._moveCurrentPoint(e.shiftKey ? -10 : -1);
            } else {
              self.panLeft(e.shiftKey);
            }
          }
          return false;
        } else if (kc == VK_RIGHT) {
          if (!isGauge) {
            if (isMain) {
              self._moveCurrentPoint(e.shiftKey ? 10 : 1);
            } else {
              self.panRight(e.shiftKey);
            }
          }
          return false;
        } else if (kc == VK_UP) {
          if (isMain) {
            self._moveCurrentSeries(-1);
          }
          return false;
        } else if (kc == VK_DOWN) {
          if (isMain) {
            self._moveCurrentSeries(1);
          }
          return false;
        }
      } else if (!isGauge) {
        if (kc == VK_PLUS || kc == VK_NP_PLUS) {
          self.zoomIn();
          return false;
        } else if (kc == VK_MINUS || kc == VK_NP_MINUS) {
          self.zoomOut();
          return false;
        } else if (kc == VK_ZERO) {
          self.viewportRestore();
          return false;
        }
      }
      if (kc == VK_F2) {
        $focus = e.target.tabIndex === 0 ? $target : null;
        if (!self._getData(DATA_EDIT_MODE)) {
          self._editMode($("#" + id + "Edit"), $focus);
        }
      }
      if (!o.restrictChanges) {
        if ($target.closest(".legendLabel").length > 0) {
          if (kc == VK_DEL) {
            $li = $target.closest("li");
            index = $li.parent().children("li").index($li.get(0));
            self._removeMetric(index);
          }
        }
      }
    });

    // d&d
    if (o.dropAccept && !o.restrictChanges) {
      $ctrl.droppable({
        accept: function(d) {
          var $chart = d.parents(".chart:first");
          return d.is(o.dropAccept + ", .chartMetric") && 
            ($chart.length === 0 || $chart.get(0) !== $ctrl.get(0));
        },
        greedy: true,
        addClasses: false,
        hoverClass: "dropActive",
        drop: function(e, ui) {
          self.addMetricFromDraggable(ui.draggable);
        }
      });
    }
  },

  getMinSize: function() {
    var $ctrl = this.element;
    var $ch = $ctrl.find(".chartMain");
    var h = MIN_CHART_HEIGHT;
    var w = MIN_CHART_WIDTH;

    w += parseInt($ch.css("margin-left"), 10) + parseInt($ch.css("margin-right"), 10);
    h += parseInt($ch.css("margin-top"), 10) + parseInt($ch.css("margin-bottom"), 10);
    h += $ctrl.find(".dashPanelHeader").outerHeight(true);
    h += $ctrl.find(".chartYaxis").outerHeight(true);
    h += $ctrl.find(".chartCtrls").outerHeight(true);
    return {h: h, w: w};
  },

  resize: function() {
    var $ctrl = this.element;
    var $ch = $ctrl.find(".chartMain");
    var h = $ctrl.innerHeight() - parseInt($ch.css("margin-top"), 10) - parseInt($ch.css("margin-bottom"), 10);
    var w = $ctrl.innerWidth() - parseInt($ch.css("margin-left"), 10) - parseInt($ch.css("margin-right"), 10);
    h -= $ctrl.find(".dashPanelHeader").outerHeight() + $ctrl.find(".chartYaxis").outerHeight();
    h -= Math.max(MIN_LEGEND_HEIGHT, $ctrl.find(".chartLegend").outerHeight());
    if (h < MIN_CHART_HEIGHT) {
      //TODO tell view to resize
      h = MIN_CHART_HEIGHT;
    }
    $ch.css({height: h, width: w});
    if (this._overviewShowing()) {
      $ctrl.find(".chartLegend").css({width: w/2});
      $ctrl.find(".overview").css({width: w/2});
      $ctrl.find(".chartOverview").css({height: 30});
    } else {
      $ctrl.find(".chartLegend").css({width: w});
    }

    this.update(true);
  },

  update: function(force) {
    if (selecting) {
      updateAfterSelect = true;
      return;
    }
    var o = this.options;
    var state = o.state;

    if (force === true || !o.isCustomRange || !o.rangeUpdated) {
      if (typeIsGauge(state.type)) {
        this._updateGauge();
      } else {
        this._updateSeries();
      }
      this._updateActionStates();
  
      if (this._getData(DATA_SHOW_CURPOINT)) {
        this._showCurrentMetricPoint();
      }
    }
  },

  getSelectedMetric: function() {
    var o = this.options;
    var metrics = o.state.metrics;

    if (o.selectedMetricIndex == -1) {
      return null;
    }
    return metrics[o.selectedMetricIndex];
  },

  addMetricFromDraggable: function(d) {
    var o = this.options;

    if (!o.restrictChanges) {
      if (d.hasClass("chartMetric")) {
        this._addMetricFromOtherChart(d.parents(".chart:first"), true);
      } else {
        this._addMetric();
      }
    }
  },

  addMetric: function(metric) {
    var $ctrl = this.element;
    var $legend = $ctrl.find(".chartLegend");
    var o = this.options;
    var state = o.state;
    var activeView;
    var metrics = state.metrics;

    function hasMetric(m) {
      var i, cm;
      for (i = 0; i < metrics.length; i++) {
        cm = metrics[i].metric;
        if (cm.on == m.on && cm.a == m.a && cm.type == m.type && cm.server == m.server) {
          return true;
        }
      }
      return false;
    }
  
    if (hasMetric(metric.metric)) {
      setTimeout(function() {
        var out = mvc.htmlBuilder();
        out.markup("<p>").content(mvc.getMessage(BUNDLE, "chart.msg.dupMetric")).markup("</p>");
        mvc.doMessageDialog(mvc.getMessage(BUNDLE, "chart.msg.dupMetric.title"), out.toString(), $ctrl.find("h3").get(0));
      }, 10);
      return false;
    }

    state.markerIndex = initMetricMarker(metric, metrics, state.markerIndex);
    state.metrics.push(metric);

    activeView = o.getActiveView();
    if (activeView !== null) {
      wldfDM.register(metric.metric, activeView);
    }
    if (o.isCustomRange) {
      o.rangeUpdated = false;
      wldfDM.registerRange(metric.metric, this._getRange());
    }

    this._renderLegend();

    if ($legend.height() > MIN_LEGEND_HEIGHT) {
      this.resize(); // also does update
    } else {
      this.update(true);
    }
    this._trigger(EVENT_STATE_CHANGE, 0);
    return true;
  },

  removeMetric: function(metric) {
    var $legend = this.element.find(".chartLegend");
    var o = this.options;
    var state = o.state;
    var activeView = o.getActiveView();
    var metrics = state.metrics;
    var remove = metric.metric;
    var index = null;

    $.each(metrics, function(i, m) {
      var cur = m.metric;
      if (cur.on == remove.on && cur.a == remove.a && cur.type == remove.type && cur.server == remove.server) {
        index = i;
        return false;
      }
    });
    if (index === null) {
      mvc.logError("No metric to remove");
      return;
    }
    metric = metrics[index];

    if (activeView !== null) {
      wldfDM.unregister(metric.metric, activeView);
    }
    if (o.isCustomRange) {
      wldfDM.unregisterRange(metric.metric, this._getRange());
    }

    metrics.splice(index, 1);
    if (metrics.length === 0) {
      state.markerIndex = 0;
    }

    this._renderLegend();

    if ($legend.height() > MIN_LEGEND_HEIGHT) {
      this.resize(); // also does update
    } else {
      this.update(true);
    }
    this._trigger(EVENT_STATE_CHANGE, 0);
  },

  setRange: function(offset, range) {
    var timeAxis = this._getData(DATA_TIME_AXIS);
    if (range < 1) {
      range = 1;
    }
    timeAxis.offset = offset;
    timeAxis.range = range;
    this.update(true);
  },

  zoomIn: function() {
    var timeAxis = this._getData(DATA_TIME_AXIS);
    timeAxis.range -= this._getZoomAmount();
    if (timeAxis.range < 1) {
      timeAxis.range = 1;
    }
    if (timeAxis.offset !== 0) {
      timeAxis.offset += this._getZoomAmount() * 500; // (_getZoomAmount() * 1000) / 2
    }
    this.update(true);
  },

  zoomOut: function() {
    var o = this.options;
    var timeAxis = this._getData(DATA_TIME_AXIS);
    var maxRange = timeAxis.maxRange / 1000;
    if (maxRange < o.viewportRange) {
      maxRange = o.viewportRange;
    }
    timeAxis.range += this._getZoomAmount();
    if (timeAxis.range > maxRange) {
      timeAxis.range = maxRange;
    }
    if (timeAxis.offset !== 0) {
      timeAxis.offset -= this._getZoomAmount() * 500;
    }
    this.update(true);
  },

  viewportRestore: function() {
    var o = this.options;
    var timeAxis = this._getData(DATA_TIME_AXIS);
    timeAxis.range = o.viewportRange;
    timeAxis.offset = 0;
    this.update(true);
  },

  panLeft: function(large) {
    var shift = large ? 500 : 250;
    var timeAxis = this._getData(DATA_TIME_AXIS);
    var min = timeAxis.min;
    if (timeAxis.offset === 0) {
      timeAxis.offset = timeAxis.max - (timeAxis.range * 1000);
    }
    timeAxis.offset -= timeAxis.range * shift;
    if (timeAxis.offset < min) {
      timeAxis.offset = min;
    }
    this.update(true);
  },

  panRight: function(large) {
    var shift = large ? 500 : 250;
    var timeAxis = this._getData(DATA_TIME_AXIS);
    var max = timeAxis.max - (timeAxis.range * 1000);
    if (timeAxis.offset !== 0) {
      timeAxis.offset += timeAxis.range * shift;
    }
    if (timeAxis.offset > max) {
      timeAxis.offset = max;
    }
    this.update(true);
  },

  select: function(setfocus) {
    var $ctrl = this.element;
    $ctrl.addClass(CLASS_SELECTED);
    this._setTabIndex(SEL_FOCUSABLE, 0);
    if (this._getData(DATA_EDIT_MODE)) {
      setfocus = false;
      this._setTabIndex("h3, .legendLabel", -1);
    }
    if (setfocus) {
      mvc.setFocus($ctrl.find("h3").get(0));
    }
    this.updateOverviewState();
  },

  unSelect: function() {
    var $ctrl = this.element;
    var id = $ctrl.get(0).id;
    $ctrl.removeClass(CLASS_SELECTED);
    if (this._getData(DATA_EDIT_MODE)) {
      this._editMode($("#" + id + "Edit")); // exit edit mode
    }
    this._setTabIndex(SEL_FOCUSABLE, -1);
    this.updateOverviewState();
  },

  // call when something happens that could affect the state of the overview area
  // returns true if resize was done
  updateOverviewState: function() {
    return this._internalUpdateOverviewState(true);
  },

  destroy: function() {
    var $ctrl = this.element;
    var id = $ctrl.get(0).id;
    id = id ? id : "";

    $ctrl.unbind(".chart");
    $ctrl.droppable("destroy");
    $(this.element).html("").removeClass(CLASS_CHART);
    getChartMenu(id).remove();
    getMetricMenu(id).remove();
    $.widget.prototype.destroy.call(this);
  },

  _updateGauge: function() {
    var $ctrl = this.element;
    var o = this.options;
    var state = o.state;
    var i, d, m, ms, data, stats, di, hr;
    var gaugeData = [];
    var range = null;

    if (o.isCustomRange) {
      range = this._getRange();
      o.rangeUpdated = true; // assume it is
    }

    for (i = 0; i < state.metrics.length; i++) {
      m = state.metrics[i];
      ms = wldfDM.getMetricSeries(m.metric);

      if (ms) {
        if (range) {
          hr = ms.harvested[range];
          data = hr.data;
          stats = hr.stats;
          if (hr.checkForHarvested) {
            o.rangeUpdated = false;
          }
        } else {
          data = ms.data;
          stats = ms.stats;
        }
      } else {
        data = [];
        stats = null;
      }

      if (data && data.length > 0) {
        di = data[data.length - 1];
        d = { label: m.name,
          time: di[0],
          color: m.color,
          shape: m.markerShape,
          fill: m.markerFill, 
          value: di[1],
          since: stats.since,
          avg: stats.avg,
          stdev: stats.stdev,
          min: stats.min,
          max: stats.max
        };
      } else {
        d = { value: null };
      }
      gaugeData[i] = d;
    }

    var gaugeOptions = {
      type: null,
      min: state.autoRange ? null : state.rangeMin,
      max: state.autoRange ? null : state.rangeMax,
      thresholdMin: state.thresholdMin,
      thresholdMax: state.thresholdMax,
      tickFormatter: formatValue,
      color: state.color,
      fillColor: state.background,
      highlightColor: state.highlight,
      clickable: true,
      hoverable: true
    };
    if (state.type == TYPE_HGAUGE) {
      gaugeOptions.type = "linearHorizontal";
    } else if (state.type == TYPE_VGAUGE) {
      gaugeOptions.type = "linearVertical";
    } else if (state.type == TYPE_RGAUGE) {
      gaugeOptions.type = "radial";
    }

    var gauge = $.gauge($ctrl.find(".chartMain"), gaugeData, gaugeOptions);
    this._setData(DATA_GAUGE, gauge);
  },

  _updateSeries: function() {
    var self = this;
    var o = this.options;
    var state = o.state;
    var $ctrl = this.element;
    var timeAxis = this._getData(DATA_TIME_AXIS);
    var curPoint = this._getData(DATA_CURPOINT);
    var chartData = [];
    var i, j, m, s, ms, data, outOfRange, hr, daySpan, si, minSampleInterval;
    var minTime = Number.MAX_VALUE;
    var maxTime = 0;
    var maxSamples = 0;
    var range = null;

    if (o.isCustomRange) {
      range = this._getRange();
      o.rangeUpdated = true; // assume it is
    }

    for (i = 0; i < state.metrics.length; i++) {
      m = state.metrics[i];
      ms = wldfDM.getMetricSeries(m.metric);

      if (ms) {
        if (range) {
          hr = ms.harvested[range];
          data = hr.data;
          if (hr.checkForHarvested) {
            o.rangeUpdated = false;
          }
        } else {
          if (curPoint && i == curPoint.series) {
            if (ms.lastShift) {
              curPoint.point -= ms.lastShift;
              if (curPoint.point < 0) {
                curPoint.point = 0;
              }
            }
          }
          data = ms.data;
        }
      } else {
        data = [];
      }

      if (data.length > 0) {
        if (data.length > maxSamples) {
          maxSamples = data.length;
        }
        if (data[0][0] < minTime) {
          minTime = data[0][0];
        }
        if (data[data.length - 1][0] > maxTime) {
          maxTime = data[data.length - 1][0];
        }
      }
      s = { label: m.name, data: data, markers: {shape: m.markerShape, fill: m.markerFill}, color: m.color};
      if (state.type == TYPE_BAR) {
        s.order = i;
      }
      chartData[i] = s;
    }
    if (maxSamples === 0) {
      // if there are no samples then we have no idea what the server time is
      // guess that it is client local time. No big deal if it isn't since this is just for an empty graph.
      maxTime = new Date();
      maxTime = maxTime.getTime() - maxTime.getTimezoneOffset() * 60 * 1000;
      minTime = maxTime - (timeAxis.range * 1000);
    }
    var chartMin, chartMax;
    if (timeAxis.offset === 0) {
      chartMin = maxTime - (timeAxis.range * 1000);
      chartMax = maxTime;
    } else {
      chartMin = timeAxis.offset;
      chartMax = timeAxis.offset + (timeAxis.range * 1000);
      // snap to latest if within 10 seconds
      if (Math.abs(maxTime - chartMax) < 10000) {
        timeAxis.offset = 0;
      }
    }
    daySpan = (new Date(chartMax)).getUTCDate() - (new Date(chartMin)).getUTCDate();

    outOfRange = false;
    if (!state.autoRange || state.type == TYPE_BAR) {
      minSampleInterval = chartMax - chartMin;
      for (i = 0; i < chartData.length && (!outOfRange || state.type == TYPE_BAR); i++) {
        data = chartData[i].data;
        for (j = 0; j < data.length; j++) {
          m = data[j];
          if (m[0] >= chartMin && m[0] <= chartMax) {
            if (j > 1) {
              si = m[0] - data[j - 1][0];
              if (si < minSampleInterval) {
                minSampleInterval = si;
              }
            }
            if (m[1] > state.rangeMax || m[1] < state.rangeMin) {
              outOfRange = true;
              if (state.type != TYPE_BAR) {
                break; // can exit early if not interested in minSampleInterval
              }
            }
          }
        }
      }
    }

    var chartOptions = {
      legend: { show: false },
      xaxis: { mode: "time",
        tickFormatter: daySpan > 0 ? tickFormatDateTime : tickFormatTime,
        min: chartMin - 2000,
        max: chartMax + 2000
      },
      yaxis: { tickFormatter: function(val, axis) { return formatValue(val); },
        min: state.autoRange ? null : state.rangeMin,
        max: state.autoRange ? null : state.rangeMax
      },
      series: {
        lines: { show: (state.type == TYPE_LINE) },
        bars: { show: (state.type == TYPE_BAR), adjustX: true, lineWidth: 0, fill: 1, 
          barWidth: Math.ceil(minSampleInterval / (chartData.length + 1)) },
        markers: { show: (state.type == TYPE_LINE || state.type == TYPE_SCATTER), radius: 5, fillColor: state.background},
        shadowSize: 0 // no shadow because not supporded for all chart/gauge types
      },
      selection: { mode: "x", color: SELECTION_COLOR },
      grid: { clickable: true, hoverable: true, color: state.color },
      autoHighlight: false
    };
    if (state.thresholdMax !== null || state.thresholdMin !== null) {
      chartOptions.grid.markings = [];
      if (state.thresholdMax !== null) {
        chartOptions.grid.markings.push({yaxis:{from: state.thresholdMax, to: Number.MAX_VALUE }, color: state.highlight});
      }
      if (state.thresholdMin !== null) {
        chartOptions.grid.markings.push({yaxis:{from: -Number.MAX_VALUE, to: state.thresholdMin }, color: state.highlight});
      }
    }
    // check if any value of of range
    outOfRange = false;
    if (!state.autoRange) {
      for (i = 0; i < chartData.length && !outOfRange; i++) {
        data = chartData[i].data;
        for (j = 0; j < data.length; j++) {
          m = data[j];
          if (m[0] >= chartMin && m[0] <= chartMax) {
            if (m[1] > state.rangeMax || m[1] < state.rangeMin) {
              outOfRange = true;
              break;
            }
          }
        }
      }
    }
    $ctrl.find(".yAxisMsg").html(outOfRange ? mvc.getMessage(BUNDLE, "chart.msg.outOfRange") : "");

    var plot = $.plot($ctrl.find(".chartMain"), chartData, chartOptions);
    this._setData(DATA_PLOT, plot);

    minTime = minTime < chartMin ? minTime : chartMin;
    maxTime = maxTime > chartMax ? maxTime : chartMax;
    timeAxis.min = minTime;
    timeAxis.max = maxTime;
    timeAxis.maxRange = maxTime - minTime;

    if (this._overviewShowing()) {
      var overviewOptions = {
        legend: {show: false },
        xaxis: {mode: "time", ticks: 0,
          min: minTime,
          max: maxTime
        },
        yaxis: { ticks: 0 },
        series: {
          lines: {show: true},
          points: {show: false},
          bars: {show: false},
          shadowSize: 0
        },
        selection: { mode: "x", color: SELECTION_COLOR },
        grid: { clickable: false, hoverable: false, color: state.color },
        autoHighlight: false
      };
      plot = $.plot($ctrl.find(".chartOverview"), chartData, overviewOptions);
      plot.setSelection({xaxis: {from: chartMin, to: chartMax}}, true);
      this._setData(DATA_OVERVIEW_PLOT, plot);
    }
  },

  _renderLegend: function() {
    var self = this;
    var o = this.options;
    var state = o.state;
    var $ctrl = this.element;
    var $legend = $ctrl.find(".chartLegend");
    var id = $ctrl.get(0).id;
    var out = mvc.htmlBuilder();
    var chartSelected = $ctrl.hasClass(CLASS_SELECTED);
    var metrics = state.metrics;
    var m, i, btnId;

    id = id ? id : "";
    btnId += "_m_";

    function metricNote(i, $item) {
      var metric = metrics[i].metric;
      var name = metrics[i].name;
      var note = mvc.formatNoteContent(BUNDLE, [
        { label: "chart.note.name", value: name },
        { label: "chart.note.metric", value: metric.a },
        { label: "chart.note.type", value: metric.type },
        { label: "chart.note.server", value: metric.server },
        { label: "chart.note.instance", value: metric.on.replace(/,/g, ", ") }
      ]);
      mvc.showNote(note, $item);
    }

    out.markup("<ul>");
    for (i = 0; i < metrics.length; i++) {
      m = metrics[i];
      out.markup("<li class='chartMetric'><span class='legendLabel'><span class='canvasPlaceholder'></span><span class='label'>").content(m.name).markup("</span></span>");
      mvc.renderButton(out, {id: btnId + i, icon: "menu", nb: true, tooltip: mvc.getMessage(BUNDLE, "chart.menu")});
      out.markup("</li>");
    }
    out.markup("</ul>");
    canvasCleanup($legend.find(".canvasPlaceholder"));
    $legend.html(out.toString()).find("li").each(function(i) {
      m = metrics[i];
      paintMarker($(this).find(".canvasPlaceholder"), state.background, m.markerShape, m.color, m.markerFill ? state.background : m.color);

      $(this).find(".legendLabel").hover(function(event) {
        metricNote(i, $(this));
      }, function() {
        mvc.hideNote();
      }).focus(function(event) {
        $(this).parent().addClass(CLASS_SELECTED);
        o.selectedMetricIndex = i;
        metricNote(i, $(this));
      }).blur(function(event) {
        $(this).parent().removeClass(CLASS_SELECTED);
        o.selectedMetricIndex = -1;
        mvc.hideNote();
      }).click(function(e) {
        mvc.setFocus(this, 0);  // FF needs this when li is a draggable
      }).get(0).tabIndex = chartSelected ? 0 : -1;

      if (!o.restrictChanges) {
        $(this).draggable({
          addClasses: false,
          cursor: "default",
          opacity: 0.8,
          distance: 5,
          revert: "invalid",
          scroll: false,
          appendTo: "#main",
          containment: "#main",
          zIndex: 950,
          helper: function() {
            var metric = metrics[i];
            var out = mvc.htmlBuilder();
            out.markup("<div class='dragMetric'>").content(metric.name).markup("</div>");
            return $(out.toString());
          },
          start: function() {
            o.selectedMetricIndex = i;
            mvc.hideNote();
          },
          stop: function() {
            o.selectedMetricIndex = -1;
          }
        });
      }

      $(this).find("button").click(function(event) {
        var $menu = getMetricMenu(id);
        var item;
        var panels = o.getPanels();

        function addPanelMenuItems(item, move, excludeSelf) {
          var i, p, $panel;
          var count = 0;

          function action(m) {
            self._moveOrCopyMetricTo(m.metricIndex, this.value, move);
          }

          item.menu.items = [];
          for (i = 0; i < panels.length; i++) {
            p = panels[i];
            if (excludeSelf) {
              $panel = o.getPanel(p.r, p.c);
              if ($panel.get(0) === $ctrl.get(0)) {
                continue;
              }
            }
            count += 1;
            item.menu.items.push({ type: "action", label: p.name, value: p, action: action });
          }
          item.disabled = (count === 0);
        }

        $menu.menu("option", "metricIndex", i);
        $menu.menu("find", "moveUp").disabled = (i === 0);
        $menu.menu("find", "moveDown").disabled = (i == metrics.length - 1);
        // update copy to/move to menus based on view container state
        item = $menu.menu("find", "moveTo");
        if (o.restrictChanges) {
          item.disabled = true;
        } else {
          addPanelMenuItems(item, true, true);
        }
        item = $menu.menu("find", "copyTo");
        if (o.restrictChanges) {
          item.disabled = true;
        } else {
          addPanelMenuItems(item, false, true); // TODO copy to self useful if can select different instance for metric
        }
        $menu.menu("find", "moveUp").disabled = o.restrictChanges;
        $menu.menu("find", "moveDown").disabled = o.restrictChanges;
        $menu.menu("find", "delete").disabled = o.restrictChanges;

        $menu.menu("toggle", $(this));
      }).focus(function(event) {
        $(this).parent().addClass(CLASS_SELECTED);
      }).blur(function(event) {
        $(this).parent().removeClass(CLASS_SELECTED);
      });
    });
    mvc.initIcons($legend);
  },

  _internalUpdateOverviewState: function(resize) {
    var $ctrl = this.element;
    var state = this.options.state;
    var curState = this._overviewShowing();
    var show;

    if (!typeIsGauge(state.type)) {
      show = true;
      if (showOverview == "N" || (showOverview == "S" && !$ctrl.hasClass(CLASS_SELECTED))) {
        show = false;
      }
    } else {
      show = false;
    }
    if (show != curState) {
      $ctrl.find(".overview").css("display", show ? "block" : "none");
      if (resize) {
        this.resize(false);
        return true;
      }
    }
    return false;
  },

  _overviewShowing: function() {
    var $ctrl = this.element;
    return $ctrl.find(".overview").css("display") != "none";
  },

  _updateActionStates: function() {
    var o = this.options;
    var state = o.state;
    var id = this.element.get(0).id;
    id = id ? id : "";
    var $chartMenu = getChartMenu(id);
    var timeAxis = this._getData(DATA_TIME_AXIS);
    // disabled flags
    var zoomIn = true;
    var zoomOut = true;
    var panLeft = true;
    var panRight = true;
    var restore = true;

    if (!typeIsGauge(state.type)) {
      zoomIn = timeAxis.range <= 1;
      zoomOut = timeAxis.range >= Math.max(timeAxis.maxRange / 1000, o.viewportRange);
      panLeft = timeAxis.offset === 0 ? timeAxis.range >= timeAxis.maxRange / 1000 : timeAxis.offset <= timeAxis.min;
      panRight = timeAxis.offset === 0 || timeAxis.offset >= timeAxis.max - (timeAxis.range * 1000);
      restore = timeAxis.range === o.viewportRange && timeAxis.offset === 0;
    }

    $.each([
      {id: "zoomIn", sel: "#" + id + "ZoomIn", d: zoomIn},
      {id: "zoomOut", sel: "#" + id + "ZoomOut", d: zoomOut},
      {id: "panLeft", sel: "#" + id + "PanLeft", d: panLeft},
      {id: "panRight", sel: "#" + id + "PanRight", d: panRight},
      {id: "restore", d: restore}
    ], function(i, item) {
      var menuItem = $chartMenu.menu("find", item.id);
      if (menuItem) {
        menuItem.disabled = item.d;
      }
      if (item.sel) {
        $(item.sel).toggleClass("disabled", item.d).each(function() { this.disabled = item.d; });
      }
    });
  },

  _moveCurrentSeries: function(offset) {
    var state = this.options.state;
    var curPoint, series, plot, gauge, prevSeries, prevData, data, d, xaxis;

    curPoint = this._getData(DATA_CURPOINT);
    if (typeIsGauge(state.type)) {
      gauge = this._getData(DATA_GAUGE);
      if (gauge) {
        series = gauge.data;
        curPoint.series += offset;
        if (curPoint.series < 0) {
          curPoint.series = series.length - 1;
        } else if (curPoint.series >= series.length) {
          curPoint.series = 0;
        }
        showGaugeData(gauge, curPoint, this.element.find(".chartMain"));
      }
    } else {
      plot = this._getData(DATA_PLOT);
      if (plot) {
        series = plot.getData();
        // Adjust for different series lengths
        prevSeries = curPoint.series;
        curPoint.series += offset;
        if (curPoint.series < 0) {
          curPoint.series = series.length - 1;
        } else if (curPoint.series >= series.length) {
          curPoint.series = 0;
        }
        prevData = plot.getData()[prevSeries].data;
        data = plot.getData()[curPoint.series].data;
        if (prevData.length === 0 || data.length === 0) {
          curPoint.point = data.length > 0 ? data.length - 1 : 0;
        } else if (prevData[prevData.length - 1][0] == data[data.length - 1][0]) {
          d = prevData.length - curPoint.point;
          curPoint.point = data.length - d;
          if (curPoint.point < 0) {
            curPoint.point = 0;
          }
        } else {
          if (curPoint.point >= data.length) {
            curPoint.point = data.length - 1;
          }
        }
        // Make sure there is a point in this series
        if (data.length > 0) {
          xaxis = plot.getAxes().xaxis;
          if (makeCurPointVisible(curPoint, xaxis, data)) {
            showMetricPoint(plot, curPoint, this.element.find(".chartMain"));
          }
        } else {
          mvc.hideNote();
        }
      }
    }
  },

  _moveCurrentPoint: function(offset) {
    var curPoint, series, pt, x, y, plot, xaxis;

    function movePoint(dist) {
        curPoint.point += dist;
        if (curPoint.point < 0) {
          curPoint.point = 0;
        } else if (curPoint.point >= series.data.length) {
          curPoint.point = series.data.length - 1;
        }
    }

    // only applies to series
    plot = this._getData(DATA_PLOT);
    if (plot) {
      curPoint = this._getData(DATA_CURPOINT);
      series = plot.getData()[curPoint.series];
      if (series.data.length === 0) {
        return; // this can happen when a new metric is added
      }
      movePoint(offset);
      // skip gaps
      pt = series.data[curPoint.point];
      x = pt[0];
      y = pt[1];
      if (y === null) {
        movePoint(offset < 0 ? -1 : 1); // shouldn't be more than one null in a row
        pt = series.data[curPoint.point];
        x = pt[0];
      }
      // make sure its visible by moving the viewport
      xaxis = plot.getAxes().xaxis; 
      if (x < xaxis.min) {
        this.panLeft(true);
        return; // the update after pan will show the point again
      } else if (x > xaxis.max) {
        this.panRight(true);
        return; // the update after pan will show the point again
      }
      showMetricPoint(plot, curPoint, this.element.find(".chartMain"));
    }
  },

  _showCurrentMetricPoint: function() {
    var state = this.options.state;
    var curPoint, series, xaxis, data;
    var plot, gauge;

    curPoint = this._getData(DATA_CURPOINT);
    if (typeIsGauge(state.type)) {
      gauge = this._getData(DATA_GAUGE);
      showGaugeData(gauge, curPoint, this.element.find(".chartMain"));
    } else {
      plot = this._getData(DATA_PLOT);
      series = plot.getData();
      // make sure its visible by moving the point if needed
      data = series[curPoint.series].data;
      xaxis = plot.getAxes().xaxis;
      if (makeCurPointVisible(curPoint, xaxis, data)) {
        showMetricPoint(plot, curPoint, this.element.find(".chartMain"));
      }
    }
  },

  _addMetric: function() {
    var o = this.options;

    if (!o.getNewMetric) {
      return;
    }
    var metric = o.getNewMetric();
    if (metric) {
      this.addMetric(metric);
    } else {
      mvc.logError("No metric to add");
    }
  },

  _addMetricFromOtherChart: function($chart, move) {
    var $ctrl = this.element;
    var metric, added;

    if ($ctrl.get(0) === $chart.get(0)) {
      return; // can't add metric from self
    }
    metric = $chart.chart("getSelectedMetric");
    if (metric) {
      added = this.addMetric($.extend(true, {}, metric));
      if (added && move) {
        // wait till after drag completes to delete the source metric
        setTimeout(function() {
          $chart.chart("removeMetric", metric);
        }, 10);
      }
    }
  },

  _removeMetric: function(index) {
    var $ctrl = this.element;
    var state = this.options.state;
    var metrics = state.metrics;

    mvc.hideNote();
    this.removeMetric(metrics[index]);
    if (metrics.length > 0) {
      if (index > metrics.length - 1) {
        index = metrics.length - 1;
      }
      mvc.setFocus($ctrl.find(".legendLabel:eq(" + index + ")").get(0), 0);
    } else {
      mvc.setFocus($ctrl.find("h3").get(0), 0);
    }
  },

  _moveOrCopyMetricTo: function(index, other, move) {
    var o = this.options;
    var metrics = o.state.metrics;
    var metric = metrics[index];
    var $chart = o.getPanel(other.r, other.c);
    var added = false;

    added = $chart.chart("addMetric", $.extend(true, {}, metric)); // add copy of metric to other chart
    if (added && move) {
      this.removeMetric(metric);
    }
  },

  _metricUp: function(index) {
    var self = this;
    var $ctrl = this.element;
    var state = this.options.state;
    var metrics = state.metrics;
    var metric, $li;

    if (index >= 1 && index < metrics.length) {
      metric = metrics[index];
      metrics.splice(index, 1);
      index -= 1;
      metrics.splice(index, 0, metric);
      // just moving the DOM li nodes is no good because of handlers that have captured the metric
      this._renderLegend();
      // put focus back
      mvc.setFocus($ctrl.find(".chartLegend li:eq(" + index + ") button").get(0), 0);
    }
  },

  _metricDown: function(index) {
    var self = this;
    var $ctrl = this.element;
    var state = this.options.state;
    var metrics = state.metrics;
    var metric, $li;

    if (index >= 0 && index < metrics.length - 1) {
      metric = metrics[index];
      metrics.splice(index, 1);
      index += 1;
      metrics.splice(index, 0, metric);
      // just moving the DOM li nodes is no good because of handlers that have captured the metric
      this._renderLegend();
      // put focus back
      mvc.setFocus($ctrl.find(".chartLegend li:eq(" + index + ") button").get(0), 0);
    }
  },

  _doMetricPropertiesDialog: function(index, f) {
    var self = this;
    var $ctrl = this.element;
    var metric = this.options.state.metrics[index];
    var $dialog = $("#chartMetricDialog");

    var fh = mvc.formHelper([
      { id: "metricName",
        prop: "name",
        constraints: ["required"]
      }
    ], {
      formContext: $dialog,
      layout: function ($dialog) {
        $dialog.find(".filteredList").filteredList("resize");
      }
    });

    // init stuff that form helper can't
    $("#metricMarkers").filteredList("select", metric.markerShape + "," + metric.markerFill);
    $("#metricColors").filteredList("select", metric.color);

    mvc.doModalDialog($dialog, {
      title: mvc.getMessage(BUNDLE, "chart.metric.properties.title"),
      formHelper: fh,
      formData: metric,
      ok: function() {
        // save stuff that formhelper can't
        var marker = $("#metricMarkers").filteredList("selection").split(",");
        metric.markerShape = marker[0];
        metric.markerFill = (marker[1] == "true");
        metric.color = $("#metricColors").filteredList("selection");

        self._trigger(EVENT_STATE_CHANGE, 0);

        setTimeout(function() {
          self._renderLegend();
          self.update(true);
          mvc.setFocus($ctrl.find(".chartLegend li:eq(" + index + ") button").get(0), 0);
        }, 10);
      },
      focusElemAfter: f
    });

    // In theory could be done just once (by replacing the placeholder) but IE7 needs them drawn each time the dialog is shown.
    canvasCleanup($("#metricMarkers").find(".canvasPlaceholder"));
    $("#metricMarkers").find(".canvasPlaceholder").each(function(i) {
      var m = markers[i];
      paintMarker($(this), null, m.shape, "#000000", m.fill ? "#ffffff" : "#000000");
    });

  },

  _getZoomAmount: function() {
    return zoomPercent * this.options.viewportRange / 100;
  },

  _getRange: function() {
    var state = this.options.state;
    var d, s, range;
    var duration = 0;

    if (state.timeRange == "custom") {
      d = new Date(state.timeStart);
      s = state.timeDuration.split(" ");
      if (s.length == 2) {
        duration = s[0] * 24 * 60;
        s = s[1];
      } else {
        s = s[0];
      }
      s = s.split(":");
      duration += (s[0] * 60) + (s[1] * 1);

      range = leftPad(d.getUTCMonth() + 1) + "/" +
        leftPad(d.getUTCDate()) + "/" +
        d.getUTCFullYear() + " " +
        leftPad(d.getUTCHours()) + ":" +
        leftPad(d.getUTCMinutes()) + ";" + duration;
      return range;
    }
    return null;
  },

  _registerRange: function(range) {
    var o = this.options;
    var state = o.state;
    var i, m;
    var metrics = state.metrics;
    var timeAxis = this._getData(DATA_TIME_AXIS);

    o.isCustomRange = true;
    o.rangeUpdated = false;
    o.viewportRange = range.split(";")[1] * 30; // 1/2 of the total duration
    timeAxis.offset = 0;
    timeAxis.range = o.viewportRange;
    timeAxis.maxRange = o.viewportRange;
    for (i = 0; i < metrics.length; i++) {
      m = metrics[i];
      wldfDM.registerRange(m.metric, range);
    }
  },

  _unregisterRange: function(range) {
    var o = this.options;
    var state = o.state;
    var i, m;
    var metrics = state.metrics;
    var timeAxis = this._getData(DATA_TIME_AXIS);

    o.isCustomRange = false;
    o.viewportRange = defaultTimeRange;
    timeAxis.offset = 0;
    timeAxis.range = o.viewportRange;
    timeAxis.maxRange = o.viewportRange;
    for (i = 0; i < metrics.length; i++) {
      m = metrics[i];
      wldfDM.unregisterRange(m.metric, range);
    }
  },

  _doPropertiesDialog: function(f) {
    var self = this;
    var $ctrl = this.element;
    var id = $ctrl.get(0).id;
    var state = this.options.state;
    var $dialog = $("#chartDialog");
    var oldRange = this._getRange();

    if (state.timeStart !== null) {
      state.timeStart = new Date(Date.parse(state.timeStart)); // convert to date
      if (isNaN(state.timeStart)) {
        state.timeStart = null;
      }
    }

    var fh = mvc.formHelper(chartFields, {
      formContext: $dialog,
      globalConstraints: function(fields) {
        var autoRange = fields.find("autoRange").value;
        var min, max;
        if (!autoRange) {
          min = fields.find("rangeMin").value * 1;
          max = fields.find("rangeMax").value * 1;
          if (min >= max) {
            return mvc.getMessage(BUNDLE, "chart.props.validation.range");
          }
        }
        min = fields.find("thresholdMin").value;
        max = fields.find("thresholdMax").value;
        if (min !== "" && max !== "") {
          min = min * 1;
          max = max * 1;
          if (min >= max) {
            return mvc.getMessage(BUNDLE, "chart.props.validation.threshold");
          }
        }
        return null;
      }
    });

    if ($("#chartTimeRangeStartFormat").text() === "") {
      $("#chartTimeRangeStartFormat").text(mvc.getDateTimeDisplayFormat());
    }

    function updateRangeMinMax(disabled) {
      $("#chartRangeMax, #chartRangeMin").each(function() {
        this.disabled = disabled;
        $(this).toggleClass("disabled", disabled);
      });
    }
    $("#chartAutoRange").click(function(e) {
      updateRangeMinMax(this.checked);
    });
    updateRangeMinMax(state.autoRange);

    function updateTimeRange(disabled) {
      $("#chartTimeRangeStart, #chartTimeRangeDuration").each(function() {
        this.disabled = disabled;
        $(this).toggleClass("disabled", disabled);
      });
    }
    $("input:radio[name=chartRangeType]").click(function(e) {
      updateTimeRange($(this).filter(":checked").val() != "custom");
    });
    updateTimeRange(state.timeRange != "custom");

    mvc.doModalDialog($dialog, {
      title: mvc.getMessage(BUNDLE, "chart.properties.title"),
      formHelper: fh,
      formData: state,
      ok: function() {
        var out = mvc.htmlBuilder();
        var range = self._getRange();

        if (range != oldRange) {
          if (oldRange !== null) {
            self._unregisterRange(oldRange);
          }
          if (range !== null) {
            self._registerRange(range);
          }
        }

        if (state.timeStart !== null) {
          state.timeStart = state.timeStart.toUTCString(); // convert to string
        }

        $ctrl.find(".chartArea").css({
          backgroundColor: state.background,
          color: state.color
        });
        self._renderLegend();
        if (!self.updateOverviewState()) {
          self.update(true);
        }
        // display title and units
        out.clear();
        $ctrl.find("h3").html(out.content(state.name).toString());
        out.clear();
        $ctrl.find(".yAxisUnits").html(out.content(state.units).toString());
        
        self._trigger(EVENT_STATE_CHANGE, 0);
      },
      focusElemAfter: f
    });
  },

  _initMenus: function() {
    this._initChartMenus();
    this._initMetricMenus();
  },

  _initChartMenus: function() {
    var self = this;
    var $ctrl = this.element;
    var state = this.options.state;
    var id = $ctrl.get(0).id;
    id = id ? id : "";
    var $menu = getChartMenu(id);

    var chartMenu = {
      menubar: false,
      items: [
        { id: "addMetric", type: "action", label: mvc.getMessage(BUNDLE, "chart.menu.addMetric"), action: function() { self._addMetric(); } },
        { type: "subMenu", label: mvc.getMessage(BUNDLE, "chart.menu.type"), menu: {
          items: [
            { type: "radioGroup", get: function() { return state.type; }, set: function(v) {
                if (v != state.type) {
                  state.type = v;
                  if (!self.updateOverviewState()) {
                    self.update(true);
                  }
                  self._trigger(EVENT_STATE_CHANGE, 0);
                }
              }, choices:
              [
                { label: mvc.getMessage(BUNDLE, "chart.menu.type.bar"), value: TYPE_BAR },
                { label: mvc.getMessage(BUNDLE, "chart.menu.type.line"), value: TYPE_LINE },
                { label: mvc.getMessage(BUNDLE, "chart.menu.type.scatter"), value: TYPE_SCATTER },
                { label: mvc.getMessage(BUNDLE, "chart.menu.type.linearV"), value: TYPE_VGAUGE },
                { label: mvc.getMessage(BUNDLE, "chart.menu.type.linearH"), value: TYPE_HGAUGE },
                { label: mvc.getMessage(BUNDLE, "chart.menu.type.radial"), value: TYPE_RGAUGE }
              ] }
          ]
        } },
        { type: "separator" },
        { id: "zoomIn", type: "action", label: mvc.getMessage(BUNDLE, "chart.menu.zoomIn"), icon: "zoomIn", action: function() { self.zoomIn(); } },
        { id: "zoomOut", type: "action", label: mvc.getMessage(BUNDLE, "chart.menu.zoomOut"), icon: "zoomOut", action: function() { self.zoomOut(); } },
        { id: "panLeft", type: "action", label: mvc.getMessage(BUNDLE, "chart.menu.panLeft"), icon: "panLeft", action: function() { self.panLeft(); } },
        { id: "panRight", type: "action", label: mvc.getMessage(BUNDLE, "chart.menu.panRight"), icon: "panRight", action: function() { self.panRight(); } },
        { id: "restore", type: "action", label: mvc.getMessage(BUNDLE, "chart.menu.restore"), action: function() { self.viewportRestore(); } },
        { type: "separator" },
        { type: "action", label: mvc.getMessage(BUNDLE, "chart.menu.properties"), action: function(o, f) { self._doPropertiesDialog(f); return true;} }
      ]
    };
    $menu.menu(chartMenu);
  },

  _initMetricMenus: function() {
    var self = this;
    var $ctrl = this.element;
    var state = this.options.state;
    var id = $ctrl.get(0).id;
    id = id ? id : "";
    var $menu = getMetricMenu(id);

    var metricMenu = {
      menubar: false,
      metricIndex: null, // set this before opening the menu
      items: [
        { id: "delete", type: "action", label: mvc.getMessage(BUNDLE, "chart.menu.delete"), action: function(m) { self._removeMetric(m.metricIndex); } },
        { id: "moveTo", type: "subMenu", label: mvc.getMessage(BUNDLE, "chart.menu.moveTo"), menu: {
          items: [
            // dynamic list of charts in view
          ]
        } },
        { id: "copyTo", type: "subMenu", label: mvc.getMessage(BUNDLE, "chart.menu.copyTo"), menu: {
          items: [
            // dynamic list of charts in view
          ]
        } },
        { type: "separator" },
        { id: "moveUp", type: "action", label: mvc.getMessage(BUNDLE, "chart.menu.moveUp"), action: function(m) { self._metricUp(m.metricIndex); } },
        { id: "moveDown", type: "action", label: mvc.getMessage(BUNDLE, "chart.menu.moveDown"), action: function(m) { self._metricDown(m.metricIndex); } },
        { type: "separator" },
        { type: "action", label: mvc.getMessage(BUNDLE, "chart.menu.properties"), action: function(m, f) { self._doMetricPropertiesDialog(m.metricIndex, f); return true;} }
      ]
    };
    $menu.menu(metricMenu);
  },

  _editMode: function($btn, $focus) {
    var self = this;
    var state = this.options.state;
    var $ctrl = this.element;
    var editMode = this._getData(DATA_EDIT_MODE);
    var before, after;
    var startFocus;

    function editField(sel, value) {
      var out = mvc.htmlBuilder();
      out.markup("<input type='text' maxlength='100' value='").attr(value).markup("'>");
      $ctrl.find(sel).html(out.toString());
    }

    function endEditField(sel, orig) {
      var out = mvc.htmlBuilder();
      var value = $ctrl.find(sel + " input").val();
      if (!value && orig) {
        value = orig;
      }
      $ctrl.find(sel).html(out.content(value).toString());
      return value;
    }

    function abortEditField(sel, value) {
      var out = mvc.htmlBuilder();
      var encVal;
      if (!value) {
        encVal = "&nbsp";
      } else {
        encVal = out.content(value).toString();
      }
      $ctrl.find(sel).html(encVal);
    }
    
    function getText() {
      var text = state.name + state.units;
      $.each(state.metrics, function(i, metric) {
        text += metric.name;
      });
      return text;
    }

    function cleanup() {
      $ctrl.find(".chartLegend li, .yAxisUnits").removeClass("editMode");
      if ($ctrl.hasClass(CLASS_SELECTED)) {
        self._setTabIndex("h3, .legendLabel", 0);
      }
      $btn.removeClass("active").attr("title", mvc.getMessage(BUNDLE, "chart.edit"));
      $ctrl.unbind("keydown.editmode");
      if ($ctrl.hasClass(CLASS_SELECTED)) {
        mvc.setFocus($focus ? $focus.get(0) : $btn.get(0), 0);
      }
    }

    editMode = !editMode;
    this._setData(DATA_EDIT_MODE, editMode);

    if (editMode) {
      editField("h3", state.name);
      editField(".yAxisUnits", state.units);
      $.each(state.metrics, function(i, metric) {
        editField(".legendLabel:eq(" + i + ") .label", metric.name);
      });
      $ctrl.find(".chartLegend li, .yAxisUnits").addClass("editMode");
      startFocus = $ctrl.find("h3 input").get(0);
      if ($focus && $focus.is(".legendLabel")) {
        startFocus = $focus.find("input").get(0);
      }
      startFocus.select();
      mvc.setFocus(startFocus, 0);
      this._setTabIndex("h3, .legendLabel", -1);
      $btn.addClass("active").attr("title", mvc.getMessage(BUNDLE, "chart.edit.active"));
      $ctrl.bind("keydown.editmode", function(e) {
        var kc = e.keyCode;
        if (e.altKey || e.ctrlKey || !$(e.target).is(":text")) {
          return;
        }
        if (kc == VK_ESC) {
          self._setData(DATA_EDIT_MODE, false);
          abortEditField("h3", state.name);
          abortEditField(".yAxisUnits", state.units);
          $.each(state.metrics, function(i, metric) {
            abortEditField(".legendLabel:eq(" + i + ") .label", metric.name);
          });
          cleanup();
          return false;
        } else if (kc == VK_ENTER) {
          self._editMode($btn, $focus);
          return false;
        }
      });
    } else {
      before = getText();
      state.name = endEditField("h3", state.name);
      state.units = endEditField(".yAxisUnits");
      $.each(state.metrics, function(i, metric) {
        metric.name = endEditField(".legendLabel:eq(" + i + ") .label", metric.name);
      });
      cleanup();
      after = getText();
      if (after != before) {
        this._trigger(EVENT_STATE_CHANGE, 0);
      }
    }
  },

  _setTabIndex: function(sel, val) {
    var $ctrl = this.element;
    $ctrl.find(sel).each(function() {
      this.tabIndex = val;
    });
  }
  

});

$.extend($.ui.chart, {
  version: "1.7",
  getter: ["getSelectedMetric", "addMetric", "getMinSize"],
  defaults: {
    // here current refers to something outside of the chart. These call backs are used for adding new metrcs
    getNewMetric: null, // fn() returns  {name:, metric:, color:, markerShape: , markerFill: } 
                                      // or null if none. Only name and metric are required
    hasNewMetric: null, // fn() returns true if getNewMetric would return an object
                                      // and false otherwise
    dropAccept: null,
    getPanels: null, // fn() returns [{name:, r:, c:},...]
    getPanel: null, // fn(r, c) returns $panel
    getActiveView: null, // fn() returns name of active view if active or null otherwise
    restrictChanges: false, // if true only allow presentational changes
    // Internal state:
    // isCustomRange: // <bool> true if this chart is showing historical (harvested data) 
    // rangeUpdated:  // <bool> the data for a historical range doesn't change so only needs to be updated 
                      // (drawn) once (update is still done for presentation changes)
    // viewportRange: // from defaultTimeRange for current timeRange and based on amount of data for custom timeRange 
    state: {
      name: null, // chart name or title
      units: "",  // label for Y axis
      type: TYPE_LINE, // one of: "linearHGauge" | "linearVGauge" | "radialGauge" | "barChart" | "linePlot" | "scatterPlot"
      color: "#000000",
      background: "#ffffff",
      highlight: "#FBC8CA",
      autoRange: true, // if true rangeMin and rangeMax are ignored
      rangeMin: 0,
      rangeMax: 100,
      thresholdMin: null,
      thresholdMax: null,
      markerIndex: 0, // internal book keeping but it is persisted
      timeRange: "current", // one of: current, custom 
      timeStart: "", 
      timeDuration: "",
      metrics: [] // array of: {name:, metric:, color:, markerShape: , markerFill: } see wldfDataManager for metric type
    }
  },
  setDefaultTimeRange: function(range) {
    defaultTimeRange = range * 60; // convert to seconds
  },
  setZoomPercent: function(zoom) {
    zoomPercent = zoom;
  },
  setShowOverview: function(show) {
    if (show == "always") {
      showOverview = "A";
    } else if (show == "never") {
      showOverview = "N";
    } else {
      showOverview = "S"; // when selected
    }
  },
  start: function(state, view) {
    var i;
    var metrics = state.metrics;

    for (i = 0; i < metrics.length; i++) {
      wldfDM.register(metrics[i].metric, view);
    }
  },
  stop: function(state, view) {
    var i;
    var metrics = state.metrics;

    for (i = 0; i < metrics.length; i++) {
      wldfDM.unregister(metrics[i].metric, view);
    }
  },
  validateState: function (state) {
    var i, metrics, metric, m;

    function checkFields(fields, obj, message) {
      $.each(fields, function(i, field) {
        var value = obj[field.prop];
        var reason;
        var valid;
        if (value === null || value === "") {
          return true; // continue. Missing value OK - default applied
        }
        field.value = value;
        valid = mvc.validate(field, null, function(f, r) {
          reason = r;
        });
        if (!valid) {
          delete obj[field.prop];
          mvc.logError(message + ": [" + field.prop + "=" + value + "]. Reason: " + reason);
        }
      });
    }

    if (!state.name) {
      mvc.logError("Missing chart name");
      return false;
    }

    if (state.timeStart) {
      state.timeStart = new Date(Date.parse(state.timeStart)); // convert to date
      if (isNaN(state.timeStart)) {
        state.timeStart = null;
      } else {
        state.timeStart = mvc.formatDateTime(state.timeStart, {fullYear: true});
      }
    }

    checkFields(chartFields, state, "Ignoring bad chart value");

    if (state.timeStart) {
      state.timeStart = mvc.parseDateTime(state.timeStart, {fullYear: true});
      state.timeStart = state.timeStart.toUTCString(); // convert to string
    } else {
      state.timeRange = "current";
    }

    metrics = state.metrics;
    for (i = 0; i < metrics.length; i++) {
      metric = metrics[i];
      if (!metric.name) {
        mvc.logError("Missing metric name");
        return false;
      }
      m = metric.metric;
      if (!m || !m.on || !m.a || !m.type || !m.server) {
        mvc.logError("Missing or invalid metric");
        return false;
      }
      checkFields(metricFields, metric, "Ignoring bad metric value");
      if (!metric.markerShape || !metric.color || typeof metric.markerFill != "boolean") {
        delete metric.markerShape;
        delete metric.color;
        delete metric.markerFill;
      }
    }
    return true; // errors removed and will be defaulted
  }

});

$(document).ready(function() {

  mvc.formHelper.addValidator("color", function(value) {
    if (value === "") {
      return null;
    }
    // Look for rgb(num,num,num), rgb(num%,num%,num%), #a0b1c2, or #fff
    if (/^rgb\(\s*[0-9]{1,3}\s*,\s*[0-9]{1,3}\s*,\s*[0-9]{1,3}\s*\)$/.test(value) ||
        /^rgb\(\s*[0-9]+(?:\.[0-9]+)?\%\s*,\s*[0-9]+(?:\.[0-9]+)?\%\s*,\s*[0-9]+(?:\.[0-9]+)?\%\s*\)$/.test(value) ||
        /^#[a-fA-F0-9]{6}$/.test(value) ||
        /^#[a-fA-F0-9]{3}$/.test(value) ) {
      return null;
    }
    return mvc.getMessage(BUNDLE, "chart.validation.color");
  });

  // initialize the metric properties dialog
  $("#metricMarkers").filteredList({
    itemAdapter: {
      value: function(item) {
        return item.shape + "," + item.fill;
      },
      renderLabel: function(out, item) {
        out.markup("<span class='label'><span class='canvasPlaceholder'></span></span>");
      }
    },
    label: null,
    list: markers
  }); // painting the markers here didn't work for IE so wait until dialog is shown.

  $("#metricColors").filteredList({
    itemAdapter: {
      value: function(item) {
        return item.color;
      },
      renderLabel: function(out, item) {
        out.markup("<span class='label'><span style='background-color:").attr(item.color).markup(";'/></span></span>");
      }
    },
    label: null,
    list: markers
  });

});

})(jQuery);
