/*******************************************************************************
 * Copyright (c) 2010, 2013 Tasktop Technologies
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Tasktop Technologies - initial API and implementation
 ******************************************************************************/
package com.tasktop.c2c.server.common.service.web;

import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.InetSocketAddress;
import java.net.Proxy;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Arrays;
import java.util.Iterator;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.httpclient.Header;
import org.apache.commons.httpclient.HostConfiguration;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.MultiThreadedHttpConnectionManager;
import org.apache.commons.httpclient.UsernamePasswordCredentials;
import org.apache.commons.httpclient.auth.AuthScope;
import org.apache.commons.httpclient.methods.DeleteMethod;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.httpclient.methods.StringRequestEntity;
import org.apache.commons.httpclient.protocol.DefaultProtocolSocketFactory;
import org.apache.commons.httpclient.protocol.Protocol;
import org.apache.commons.httpclient.protocol.ProtocolSocketFactory;
import org.apache.commons.httpclient.protocol.SSLProtocolSocketFactory;
import org.codehaus.jackson.map.DeserializationConfig;
import org.codehaus.jackson.map.ObjectMapper;

import com.tasktop.c2c.server.common.service.WrappedCheckedException;

/**
 * A RestClientDelegate implementation using Apache Commons HTTP Client 3.1
 * 
 * @author Danny Ju (danny.ju@oracle.com)
 * 
 */
public class ApacheHttpRestClientDelegate extends RestClientDelegate {

	/** Captures URI template variable names. */
	private static final Pattern NAMES_PATTERN = Pattern.compile("\\{([^/]+?)\\}");

	private HttpClient httpClient = null;
	private Proxy proxy = null;
	private String userName;
	private String passwd;
	private String proxyUserName = null;
	private String proxyPassword = null;

	private static final int CONNNECT_TIMEOUT = 60 * 1000;

	private static final int SOCKET_TIMEOUT = 3 * 60 * 1000;

	private static final int HTTP_PORT = 80;

	private static final int HTTPS_PORT = 443;

	private static ProtocolSocketFactory sslSocketFactory = new SSLProtocolSocketFactory();

	private static ProtocolSocketFactory socketFactory = new DefaultProtocolSocketFactory();

	ApacheResponseErrorHandler errorHandler = new ApacheResponseErrorHandler();

	/**
	 * Ctor
	 * 
	 * @param userName
	 * @param passwd
	 */
	public ApacheHttpRestClientDelegate(String userName, String passwd) {
		MultiThreadedHttpConnectionManager connectionManager = new MultiThreadedHttpConnectionManager();
		httpClient = new HttpClient(connectionManager);
		this.userName = userName;
		this.passwd = passwd;
	}

	/**
	 * Ctor to allow user pass in a host configuration, which may include proxy info, etc
	 * 
	 * @param cfg
	 * @param userName
	 * @param passwd
	 */
	public ApacheHttpRestClientDelegate(String userName, String passwd, Proxy proxy) {
		this(userName, passwd);
		this.proxy = proxy;
	}

	public ApacheHttpRestClientDelegate(String userName, String passwd, Proxy proxy, String proxyUser,
			String proxyPasswd) {
		this(userName, passwd, proxy);
		this.proxyUserName = proxyUser;
		this.proxyPassword = proxyPasswd;
	}

	private static void configureHttpClientConnectionManager(HttpClient client) {
		client.getHttpConnectionManager().getParams().setSoTimeout(SOCKET_TIMEOUT);
		client.getHttpConnectionManager().getParams().setConnectionTimeout(CONNNECT_TIMEOUT);
		client.getHttpConnectionManager().getParams()
				.setMaxConnectionsPerHost(HostConfiguration.ANY_HOST_CONFIGURATION, 100);
		client.getHttpConnectionManager().getParams().setMaxTotalConnections(1000);
	}

	public static String getHost(String repositoryUrl) {
		String result = repositoryUrl;
		int colonSlashSlash = repositoryUrl.indexOf("://"); //$NON-NLS-1$

		if (colonSlashSlash >= 0) {
			result = repositoryUrl.substring(colonSlashSlash + 3);
		}

		int colonPort = result.indexOf(':');
		int requestPath = result.indexOf('/');

		int substringEnd;

		// minimum positive, or string length
		if (colonPort > 0 && requestPath > 0) {
			substringEnd = Math.min(colonPort, requestPath);
		} else if (colonPort > 0) {
			substringEnd = colonPort;
		} else if (requestPath > 0) {
			substringEnd = requestPath;
		} else {
			substringEnd = result.length();
		}

		return result.substring(0, substringEnd);
	}

	private static boolean isRepositoryHttps(String repositoryUrl) {
		return repositoryUrl.matches("https.*"); //$NON-NLS-1$
	}

	public static int getPort(String repositoryUrl) {
		int colonSlashSlash = repositoryUrl.indexOf("://"); //$NON-NLS-1$
		int firstSlash = repositoryUrl.indexOf("/", colonSlashSlash + 3); //$NON-NLS-1$
		int colonPort = repositoryUrl.indexOf(':', colonSlashSlash + 1);
		if (firstSlash == -1) {
			firstSlash = repositoryUrl.length();
		}
		if (colonPort < 0 || colonPort > firstSlash) {
			return isRepositoryHttps(repositoryUrl) ? HTTPS_PORT : HTTP_PORT;
		}

		int requestPath = repositoryUrl.indexOf('/', colonPort + 1);
		int end = requestPath < 0 ? repositoryUrl.length() : requestPath;
		String port = repositoryUrl.substring(colonPort + 1, end);
		if (port.length() == 0) {
			return isRepositoryHttps(repositoryUrl) ? HTTPS_PORT : HTTP_PORT;
		}

		return Integer.parseInt(port);
	}

	public HostConfiguration createHostConfiguration(HttpClient client, String url) {

		String host = getHost(url);
		int port = getPort(url);
		configureHttpClientConnectionManager(client);

		// configure HTTP client authentication
		client.getParams().setAuthenticationPreemptive(true);
		if (userName != null) {
			AuthScope authScope = new AuthScope(host, port, AuthScope.ANY_REALM);
			UsernamePasswordCredentials credentials = new UsernamePasswordCredentials(userName, passwd);
			client.getState().setCredentials(authScope, credentials);
		}

		// Create a new HostConfiguration
		HostConfiguration hostConfig = new HostConfiguration();

		// Configure Proxy
		if (proxy != null && !Proxy.NO_PROXY.equals(proxy)) {
			InetSocketAddress address = (InetSocketAddress) proxy.address();
			hostConfig.setProxy(address.getHostName(), address.getPort());
			if (proxyUserName != null) {
				UsernamePasswordCredentials proxyCredentials = new UsernamePasswordCredentials(proxyUserName,
						proxyPassword);
				AuthScope proxyAuthScope = new AuthScope(address.getHostName(), address.getPort(), AuthScope.ANY_REALM);
				client.getState().setProxyCredentials(proxyAuthScope, proxyCredentials);
			}
		} else {
			hostConfig.setProxyHost(null);
		}

		if (isRepositoryHttps(url)) {
			Protocol protocol = new Protocol("https", sslSocketFactory, HTTPS_PORT); //$NON-NLS-1$
			hostConfig.setHost(host, port, protocol);
		} else {
			Protocol protocol = new Protocol("http", socketFactory, HTTP_PORT); //$NON-NLS-1$
			hostConfig.setHost(host, port, protocol);
		}

		return hostConfig;
	}

	public <T> T getForObject(String url, Class<T> responseType, Object... urlVariables) {
		try {
			url = expandUriComponent(url, urlVariables);
			return getForObject(new URI(url), responseType);
		} catch (URISyntaxException ue) {
			throw new WrappedCheckedException(ue);
		}
	}

	public <T> T postForObject(String url, Object request, Class<T> responseType, Object... uriVariables) {
		try {
			HostConfiguration hostConfig = createHostConfiguration(httpClient, url);

			url = expandUriComponent(url, uriVariables);

			PostMethod method = new PostMethod(url);
			method.setDoAuthentication(true);
			method.setRequestHeader("Content-Type", "application/json");
			try {

				String postData = (new ObjectMapper()).writeValueAsString(request);
				StringRequestEntity requestEntity = new StringRequestEntity(postData, "application/json", "UTF-8");
				method.setRequestEntity(requestEntity);
				int statusCode = httpClient.executeMethod(hostConfig, method);
				InputStream in = method.getResponseBodyAsStream();

				if (errorHandler.hasError(statusCode)) {
					System.err.println("Method failed: " + method.getStatusLine());
					errorHandler.handleError(statusCode, in);
				}
				try {
					ObjectMapper m = new ObjectMapper();
					m.configure(DeserializationConfig.Feature.FAIL_ON_UNKNOWN_PROPERTIES, false);
					T results = m.readValue(new InputStreamReader(in), responseType);
					return results;
				} finally {
					in.close();
				}
			} finally {
				method.releaseConnection();
			}
		} catch (RuntimeException re) {
			throw re;
		} catch (Exception e) {
			throw new WrappedCheckedException(e);
		}
	}

	@SuppressWarnings("rawtypes")
	public Object postForEntity(String url, Object request, Class responseType, Map<String, ?> uriVariables) {
		try {
			HostConfiguration hostConfig = createHostConfiguration(httpClient, url);

			url = expandUriComponent(url, uriVariables);

			PostMethod method = new PostMethod(url);
			method.setDoAuthentication(true);
			method.setRequestHeader("Content-Type", "application/json");
			try {
				String postData = (new ObjectMapper()).writeValueAsString(request);
				StringRequestEntity requestEntity = new StringRequestEntity(postData, "application/json", "UTF-8");
				method.setRequestEntity(requestEntity);
				// Execute the method.
				int statusCode = httpClient.executeMethod(hostConfig, method);
				InputStream in = method.getResponseBodyAsStream();
				if (errorHandler.hasError(statusCode)) {
					System.err.println("Method failed: " + method.getStatusLine());
					errorHandler.handleError(statusCode, in);
				}
				try {
					ObjectMapper m = new ObjectMapper();
					m.configure(DeserializationConfig.Feature.FAIL_ON_UNKNOWN_PROPERTIES, false);
					@SuppressWarnings("unchecked")
					Object results = m.readValue(new InputStreamReader(in), responseType);
					return results;
				} finally {
					in.close();
				}

			} finally {
				method.releaseConnection();
			}
		} catch (RuntimeException re) {
			throw re;
		} catch (Exception e) {
			throw new WrappedCheckedException(e);
		}
	}

	public URI postForLocation(String url, Object request, Object... urlVariables) {
		try {
			HostConfiguration hostConfig = createHostConfiguration(httpClient, url);

			url = expandUriComponent(url, urlVariables);

			PostMethod method = new PostMethod(url);
			method.setDoAuthentication(true);
			method.setRequestHeader("Content-Type", "application/json");
			try {
				String postData = (new ObjectMapper()).writeValueAsString(request);
				StringRequestEntity requestEntity = new StringRequestEntity(postData, "application/json", "UTF-8");
				method.setRequestEntity(requestEntity);

				int statusCode = httpClient.executeMethod(hostConfig, method);

				if (errorHandler.hasError(statusCode)) {
					System.err.println("Method failed: " + method.getStatusLine());
					InputStream in = method.getResponseBodyAsStream();
					errorHandler.handleError(statusCode, in);
				}
				for (Header h : method.getResponseHeaders()) {
					if (h.getName().equals("Location")) {
						return new URI(h.getValue());
					}
				}

			} finally {
				method.releaseConnection();
			}
		} catch (RuntimeException re) {
			throw re;
		} catch (Exception e) {
			throw new WrappedCheckedException(e);
		}
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.tasktop.c2c.server.common.service.web.RestClientDelegate#getForObject(java.lang.String, java.lang.Class,
	 * java.util.Map)
	 */
	@Override
	public <T> T getForObject(String url, Class<T> responseType, Map<String, ?> uriVariables) {
		try {

			url = expandUriComponent(url, uriVariables);
			return getForObject(new URI(url), responseType);
		} catch (Exception e) {
			e.printStackTrace();
		}
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.tasktop.c2c.server.common.service.web.RestClientDelegate#getForObject(java.net.URI, java.lang.Class)
	 */
	@Override
	public <T> T getForObject(URI url, Class<T> responseType) {
		try {

			HostConfiguration hostConfig = createHostConfiguration(httpClient, url.toString());

			GetMethod method = new GetMethod(url.toString());
			method.setDoAuthentication(true);
			method.setRequestHeader("Content-Type", "application/json");
			try {
				int statusCode = httpClient.executeMethod(hostConfig, method);
				InputStream in = method.getResponseBodyAsStream();

				if (errorHandler.hasError(statusCode)) {
					System.err.println("Method failed: " + method.getStatusLine());
					errorHandler.handleError(statusCode, in);
				}
				try {
					ObjectMapper m = new ObjectMapper();
					m.configure(DeserializationConfig.Feature.FAIL_ON_UNKNOWN_PROPERTIES, false);
					T results = m.readValue(new InputStreamReader(in), responseType);
					return results;
				} finally {
					in.close();
				}
			} finally {
				method.releaseConnection();
			}
		} catch (RuntimeException re) {
			throw re;
		} catch (Exception e) {
			throw new WrappedCheckedException(e);
		}
	}

	@Override
	public void deleteForObject(String url, Object... uriVariables) {
		try {
			url = expandUriComponent(url, uriVariables);
			deleteForObject(new URI(url));
		} catch (URISyntaxException ex) {
			Logger.getLogger(ApacheHttpRestClientDelegate.class.getName()).log(Level.SEVERE, null, ex);
		}
	}

	@Override
	public void deleteForObject(URI url) {
		try {
			HostConfiguration hostConfig = createHostConfiguration(httpClient, url.toString());

			DeleteMethod method = new DeleteMethod(url.toString());
			method.setDoAuthentication(true);
			method.setRequestHeader("Content-Type", "application/json");
			try {
				int statusCode = httpClient.executeMethod(hostConfig, method);
				InputStream in = method.getResponseBodyAsStream();
				if (errorHandler.hasError(statusCode)) {
					System.err.println("Method failed: " + method.getStatusLine());
					errorHandler.handleError(statusCode, in);
				}

				if (statusCode != HttpStatus.SC_OK) {
					System.err.println("Method failed: " + method.getStatusLine());
				}
			} finally {
				method.releaseConnection();
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	private static String expandUriComponent(String source, Object... uriVariableValues) {
		if (source == null) {
			return null;
		}
		if (source.indexOf('{') == -1) {
			return source;
		}

		Iterator<Object> it = Arrays.asList(uriVariableValues).iterator();
		if (!it.hasNext())
			return source;

		Matcher matcher = NAMES_PATTERN.matcher(source);
		StringBuffer sb = new StringBuffer();
		while (matcher.find()) {
			// String match = matcher.group(1);
			if (it.hasNext()) {
				String variableValueString = it.next().toString();
				String replacement = Matcher.quoteReplacement(variableValueString);
				matcher.appendReplacement(sb, replacement);
			}
		}
		matcher.appendTail(sb);
		return sb.toString();
	}

	private static String expandUriComponent(String source, Map<String, ?> uriVariables) {
		if (source == null) {
			return null;
		}
		if (source.indexOf('{') == -1) {
			return source;
		}

		Iterator<String> it = uriVariables.keySet().iterator();
		if (!it.hasNext())
			return source;

		Matcher matcher = NAMES_PATTERN.matcher(source);
		StringBuffer sb = new StringBuffer();
		while (matcher.find()) {
			String match = matcher.group(1);
			String variableValueString = uriVariables.get(match).toString();
			String replacement = Matcher.quoteReplacement(variableValueString);
			matcher.appendReplacement(sb, replacement);
		}
		matcher.appendTail(sb);
		return sb.toString();
	}

}
