/*
 * Decompiled with CFR 0.152.
 */
package oracle.kv.impl.monitor;

import java.rmi.NotBoundException;
import java.rmi.RemoteException;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Logger;
import oracle.kv.impl.measurement.Measurement;
import oracle.kv.impl.measurement.ProxiedServiceStatusChange;
import oracle.kv.impl.monitor.Monitor;
import oracle.kv.impl.monitor.MonitorAgentAPI;
import oracle.kv.impl.topo.ResourceId;
import oracle.kv.impl.util.ConfigurableService;
import oracle.kv.impl.util.KVThreadFactory;
import oracle.kv.impl.util.registry.RegistryUtils;
import oracle.kv.impl.util.server.LoggerUtils;

public class Collector {
    private ScheduledThreadPoolExecutor collectors;
    private final Map<ResourceId, AgentInfo> agents;
    private final Monitor monitor;
    private final Logger logger;
    private final AtomicBoolean isShutdown;
    private static final int MIN_THREADS = 2;

    Collector(Monitor monitor) {
        this.monitor = monitor;
        this.agents = new ConcurrentHashMap<ResourceId, AgentInfo>();
        this.logger = LoggerUtils.getLogger(this.getClass(), monitor.getParams());
        this.isShutdown = new AtomicBoolean(false);
        this.setupCollectors();
    }

    private void setupCollectors() {
        this.collectors = new ScheduledThreadPoolExecutor(2, new CollectorThreadFactory(this.logger));
    }

    synchronized void registerAgent(String snHostname, int snRegistryPort, ResourceId agentId) {
        long pollMillis = this.monitor.getParams().getAdminParams().getPollPeriodMillis();
        this.logger.finest("Monitor interval for " + agentId + " =" + pollMillis);
        this.unregisterAgent(agentId);
        this.logger.info("Monitor collector: adding " + agentId + " to monitoring");
        AgentInfo agentInfo = new AgentInfo(snHostname, snRegistryPort);
        this.setupFuture(agentInfo, agentId, pollMillis);
        this.agents.put(agentId, agentInfo);
        this.collectors.setCorePoolSize(Math.max(this.agents.size(), 2));
    }

    private void setupFuture(AgentInfo info, ResourceId id, long pollMillis) {
        PollTask pollTask = new PollTask(info, id);
        ScheduledFuture<?> future = this.collectors.scheduleAtFixedRate(pollTask, 0L, pollMillis, TimeUnit.MILLISECONDS);
        info.setFuture(future);
    }

    synchronized void unregisterAgent(ResourceId agentId) {
        AgentInfo info = this.agents.remove(agentId);
        if (info == null) {
            return;
        }
        if (info.future == null) {
            return;
        }
        this.logger.info("Removing " + agentId + " from monitoring");
        info.future.cancel(false);
        this.collectors.setCorePoolSize(Math.max(this.agents.size(), 2));
    }

    synchronized void resetAgents(long pollMillis) {
        this.logger.info("Monitor collector: resetting interval to: " + pollMillis + " milliseconds (" + this.agents.size() + " agents)");
        Set<Map.Entry<ResourceId, AgentInfo>> entries = this.agents.entrySet();
        for (Map.Entry<ResourceId, AgentInfo> entry : entries) {
            ResourceId key = entry.getKey();
            AgentInfo info = this.agents.remove(key);
            if (info.future != null) {
                info.future.cancel(false);
            }
            this.setupFuture(info, key, pollMillis);
            this.agents.put(key, info);
        }
    }

    void shutdown() {
        this.logger.info("Shutting down monitor collector");
        this.isShutdown.set(true);
        this.collectors.shutdown();
        try {
            this.collectors.awaitTermination(1000L, TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException e) {
            this.logger.info("Collector interrupted during shutdown: " + LoggerUtils.getStackTrace(e));
        }
    }

    void collectNow() {
        for (ResourceId resourceId : this.agents.keySet()) {
            this.collectNow(resourceId);
        }
    }

    void collectNow(ResourceId resourceId) {
        AgentInfo agentInfo = this.agents.get(resourceId);
        if (agentInfo == null) {
            return;
        }
        this.collectors.execute(new PollTask(agentInfo, resourceId));
    }

    int getNumAgents() {
        return this.agents.size();
    }

    private class AgentInfo {
        private final String snHostname;
        private final int snRegistryPort;
        private Future<?> future;

        AgentInfo(String snHostname, int snRegistryPort) {
            this.snHostname = snHostname;
            this.snRegistryPort = snRegistryPort;
        }

        void setFuture(Future<?> f) {
            this.future = f;
        }
    }

    private class CollectorThreadFactory
    extends KVThreadFactory {
        CollectorThreadFactory(Logger logger) {
            super(null, logger);
        }

        @Override
        public String getName() {
            return Collector.this.monitor.getParams().getAdminParams().getAdminId() + "_MonitorCollector";
        }
    }

    private class PollTask
    implements Runnable {
        private MonitorAgentAPI agent;
        private final AgentInfo agentInfo;
        private final ResourceId agentId;
        private boolean doingRetry;
        private int numFaults;
        private static final int MAX_FAULTS = 5;

        PollTask(AgentInfo agentInfo, ResourceId agentId) {
            this.agentInfo = agentInfo;
            this.agentId = agentId;
            this.doingRetry = false;
            this.numFaults = 0;
        }

        @Override
        public void run() {
            try {
                List<Measurement> measurements;
                if (Collector.this.isShutdown.get()) {
                    Collector.this.logger.fine("Collector is shutdown");
                    return;
                }
                Collector.this.logger.fine("Monitor collector polling " + this.agentId);
                if (this.agent == null) {
                    String storeName = Collector.this.monitor.getParams().getGlobalParams().getKVStoreName();
                    this.agent = RegistryUtils.getMonitor(storeName, this.agentInfo.snHostname, this.agentInfo.snRegistryPort, this.agentId, Collector.this.monitor.getLoginManager());
                    Collector.this.logger.finer("Monitor collector looking in registry for " + this.agentId.getFullName());
                }
                if ((measurements = this.agent.getMeasurements()).size() > 0) {
                    Collector.this.monitor.publish(this.agentId, measurements);
                }
                Collector.this.logger.finest("Collected " + measurements.size() + " measurements from " + this.agentId);
                this.doingRetry = false;
                this.numFaults = 0;
            }
            catch (RemoteException e) {
                this.setupForRetry(e);
            }
            catch (NotBoundException e) {
                this.setupForRetry(e);
            }
            catch (Exception e) {
                String msg = "Collector: exception when polling agentId: ";
                if (++this.numFaults > 5) {
                    Collector.this.logger.severe(msg + LoggerUtils.getStackTrace(e));
                }
                Collector.this.logger.info(msg + e);
            }
        }

        private void setupForRetry(Exception e) {
            Collector.this.logger.fine("Collector encountered problem when polling " + this.agentId + ", clear stub for retry. " + e);
            if (Collector.this.isShutdown.get()) {
                Collector.this.logger.fine("Collector is shutdown");
                return;
            }
            this.agent = null;
            if (this.doingRetry) {
                Collector.this.monitor.publish(new ProxiedServiceStatusChange(this.agentId, ConfigurableService.ServiceStatus.UNREACHABLE));
            }
            this.doingRetry = true;
        }
    }
}

