/*******************************************************************************
 * Copyright (c) 2010, 2012 Tasktop Technologies
 * Copyright (c) 2010, 2011 SpringSource, a division of VMware
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Tasktop Technologies - initial API and implementation
 ******************************************************************************/
package com.tasktop.c2c.server.profile.service;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.tasktop.c2c.server.cloud.domain.PoolStatus;
import com.tasktop.c2c.server.cloud.domain.ProjectServiceStatus;
import com.tasktop.c2c.server.common.service.EntityNotFoundException;
import com.tasktop.c2c.server.common.service.ValidationException;
import com.tasktop.c2c.server.common.service.WrappedCheckedException;
import com.tasktop.c2c.server.common.service.domain.QueryResult;
import com.tasktop.c2c.server.common.service.web.AbstractVersionedRestServiceClient;
import com.tasktop.c2c.server.profile.domain.project.Agreement;
import com.tasktop.c2c.server.profile.domain.project.AgreementProfile;
import com.tasktop.c2c.server.profile.domain.project.Organization;
import com.tasktop.c2c.server.profile.domain.project.PasswordResetToken;
import com.tasktop.c2c.server.profile.domain.project.Profile;
import com.tasktop.c2c.server.profile.domain.project.Project;
import com.tasktop.c2c.server.profile.domain.project.ProjectInvitationToken;
import com.tasktop.c2c.server.profile.domain.project.ProjectsQuery;
import com.tasktop.c2c.server.profile.domain.project.SignUpToken;
import com.tasktop.c2c.server.profile.domain.project.SshPublicKey;
import com.tasktop.c2c.server.profile.domain.project.SshPublicKeySpec;

/**
 * A client to communicate to the task REST webservice.
 * <p/>
 * This client object should be configured with a RestClientDelegate, e.g. ApacheHttpRestClientDelegate, which is
 * provided in SDK, to handle HTTP REST communication.
 * <p/>
 * Usage:
 * 
 * <pre>
 * ProfileWebServiceClient client = new ProfileWebServiceClient();
 * ApacheHttpRestClientDelegate delegate = new ApacheHttpRestClientDelegate(user, passwd);
 * client.setRestClientDelegate(delegate);
 * client.setBaseUrl(REST_API_URL);
 * </pre>
 * 
 * Note: delete operations are now not supported in this default client implementation.
 */
public class ProfileWebServiceClient extends AbstractVersionedRestServiceClient implements ProfileWebService {

	protected static class ServiceCallResult {
		private Long projectId;
		private Long profileId;
		private Profile profile;
		private Boolean isWatching;
		private String token;
		private List<Project> projectList;
		private List<Agreement> agreements;
		private List<AgreementProfile> agreementProfiles;
		private Project project;
		private String[] roles;
		private QueryResult<Project> queryResult;
		private SignUpToken signUpToken;
		private List<SignUpToken> signUpTokens;
		private ProjectInvitationToken projectInvitationToken;

		private SshPublicKey sshPublicKey;
		private List<SshPublicKey> sshPublicKeyList;
		private List<ProjectServiceStatus> projectServiceStatusList;
		private Organization organization;
		private Map<String, String> serviceVersion;

		public void setProfile(Profile profile) {
			this.profile = profile;
		}

		public Profile getProfile() {
			return profile;
		}

		public List<Project> getProjectList() {
			return projectList;
		}

		public void setProjectList(List<Project> projectList) {
			this.projectList = projectList;
		}

		public void setAgreementList(List<Agreement> agreements) {
			this.agreements = agreements;
		}

		public List<Agreement> getAgreementList() {
			return agreements;
		}

		public void setAgreementProfileList(List<AgreementProfile> agreementProfiles) {
			this.agreementProfiles = agreementProfiles;
		}

		public List<AgreementProfile> getAgreementProfileList() {
			return agreementProfiles;
		}

		public Project getProject() {
			return project;
		}

		public void setProject(Project project) {
			this.project = project;
		}

		public String[] getRoles() {
			return roles;
		}

		public void setRoles(String[] roles) {
			this.roles = roles;
		}

		public QueryResult<Project> getQueryResult() {
			return queryResult;
		}

		public void setQueryResult(QueryResult<Project> queryResult) {
			this.queryResult = queryResult;
		}

		public SignUpToken getSignUpToken() {
			return signUpToken;
		}

		public void setSignUpToken(SignUpToken signUpToken) {
			this.signUpToken = signUpToken;
		}

		public List<SignUpToken> getSignUpTokenList() {
			return signUpTokens;
		}

		public void setSignUpTokenList(List<SignUpToken> signUpTokens) {
			this.signUpTokens = signUpTokens;
		}

		public ProjectInvitationToken getProjectInvitationToken() {
			return projectInvitationToken;
		}

		public void setProjectInvitationToken(ProjectInvitationToken projectInvitationToken) {
			this.projectInvitationToken = projectInvitationToken;
		}

		public Long getProjectId() {
			return projectId;
		}

		public void setProjectId(Long projectId) {
			this.projectId = projectId;
		}

		public Boolean getIsWatching() {
			return isWatching;
		}

		public void setIsWatching(Boolean isWatching) {
			this.isWatching = isWatching;
		}

		public String getToken() {
			return token;
		}

		public void setToken(String token) {
			this.token = token;
		}

		public Long getProfileId() {
			return profileId;
		}

		public void setProfileId(Long profileId) {
			this.profileId = profileId;
		}

		public SshPublicKey getSshPublicKey() {
			return sshPublicKey;
		}

		public void setSshPublicKey(SshPublicKey sshPublicKey) {
			this.sshPublicKey = sshPublicKey;
		}

		public List<SshPublicKey> getSshPublicKeyList() {
			return sshPublicKeyList;
		}

		public void setSshPublicKeyList(List<SshPublicKey> sshPublicKeyList) {
			this.sshPublicKeyList = sshPublicKeyList;
		}

		public List<ProjectServiceStatus> getProjectServiceStatusList() {
			return projectServiceStatusList;
		}

		public void setProjectServiceStatusList(List<ProjectServiceStatus> projectServiceStatusList) {
			this.projectServiceStatusList = projectServiceStatusList;
		}

		public Organization getOrganization() {
			return organization;
		}

		public void setOrganization(Organization organization) {
			this.organization = organization;
		}

		public Map<String, String> getServiceVersion() {
			return serviceVersion;
		}

		public void setServiceVersion(Map<String, String> serviceVersion) {
			this.serviceVersion = serviceVersion;
		}
	}

	private abstract class GetCall<T> {

		public abstract T getValue(ServiceCallResult result);

		public T doCall(String urlStub, Object... variables) {
			ServiceCallResult callResult = delegate.getForObject(computeUrl(urlStub), ServiceCallResult.class,
					variables);

			T retVal = getValue(callResult);

			if (retVal == null) {
				throw new IllegalStateException("Illegal result from call to profileWebService");
			}

			return retVal;
		}
	}
        
	private abstract class PostCall<T> {

		public abstract T getValue(ServiceCallResult result);

		public T doCall(String urlStub, Object objToPost, Object... variables) {
			ServiceCallResult callResult = delegate.postForObject(computeUrl(urlStub), objToPost,
					ServiceCallResult.class, variables);

			T retVal = getValue(callResult);

			if (retVal == null) {
				throw new IllegalStateException("Illegal result from call to profileWebService");
			}

			return retVal;
		}
	}

	public static final String GET_PROFILE_URL = "profile";

	public Profile getCurrentProfile() {
		return new GetCall<Profile>() {
			public Profile getValue(ServiceCallResult result) {
				return result.getProfile();
			}
		}.doCall(GET_PROFILE_URL);
	}

	public static final String UPDATE_PROFILE_URL = "profile";

	public void updateProfile(Profile profile) throws ValidationException, EntityNotFoundException {
		// FIXME this cannot be templated as usual because it returns void rather than a ServiceCallResult
		try {
			delegate.postForObject(computeUrl(UPDATE_PROFILE_URL), profile, Void.class);
		} catch (WrappedCheckedException e) {
			convertEntityNotFoundException(e);
			convertValidationException(e);
			throw (e);
		}
	}

	public static final String CREATE_PROJECT_URL = "profile/project";

	public Project createProject(Project project) throws EntityNotFoundException, ValidationException {
		try {
			return new PostCall<Project>() {
				public Project getValue(ServiceCallResult result) {
					return result.getProject();
				}
			}.doCall(CREATE_PROJECT_URL, project);
		} catch (WrappedCheckedException e) {
			convertEntityNotFoundException(e);
			convertValidationException(e);
			throw e;
		}
	}

        public void deleteProject(String projectIdentifier) throws EntityNotFoundException {
            try {
                delegate.deleteForObject(computeUrl(GET_PROJECT_BY_IDENTIFIER_URL), projectIdentifier);
            } catch (WrappedCheckedException e) {
                convertEntityNotFoundException(e);
                throw e;
            }
        }

	public static final String GET_PENDING_AGREEMENTS_URL = "agreements/pending";

	public List<Agreement> getPendingAgreements() throws EntityNotFoundException {
		try {
			return new GetCall<List<Agreement>>() {
				public List<Agreement> getValue(ServiceCallResult result) {
					return result.getAgreementList();
				}
			}.doCall(GET_PENDING_AGREEMENTS_URL);
		} catch (WrappedCheckedException e) {
			convertEntityNotFoundException(e);
			throw e;
		}
	}

	public static final String AGREEMENT_ID_URLPARAM = "agreementId";

	public static final String APPROVE_AGREEMENT_URL = "agreements/{" + AGREEMENT_ID_URLPARAM + "}/approve";

	public void approveAgreement(Long agreementId) throws EntityNotFoundException {
		// FIXME this cannot be templated as usual because it returns a Void rather than a ServiceCallResult
		try {
			delegate.postForObject(computeUrl(APPROVE_AGREEMENT_URL), null, Void.class, String.valueOf(agreementId));
		} catch (WrappedCheckedException e) {
			convertEntityNotFoundException(e);
			throw e;
		}
	}

	public static final String GET_APPROVED_AGREEMENTS_URL = "agreements/approved";

	public List<AgreementProfile> getApprovedAgreementProfiles() throws EntityNotFoundException {
		try {
			return new GetCall<List<AgreementProfile>>() {
				public List<AgreementProfile> getValue(ServiceCallResult result) {
					return result.getAgreementProfileList();
				}
			}.doCall(GET_APPROVED_AGREEMENTS_URL);
		} catch (WrappedCheckedException e) {
			convertEntityNotFoundException(e);
			throw e;
		}
	}

	public static final String TOKEN_URLPARAM = "token";

	public static final String GET_PROJECT_FOR_INVITATION_URL = "invitation/{" + TOKEN_URLPARAM + "}/project";

	public Project getProjectForInvitationToken(String token) throws EntityNotFoundException {
		try {
			return new GetCall<Project>() {
				public Project getValue(ServiceCallResult result) {
					return result.getProject();
				}
			}.doCall(GET_PROJECT_FOR_INVITATION_URL, token);
		} catch (WrappedCheckedException e) {
			convertEntityNotFoundException(e);
			throw e;
		}
	}

	public static final String PROJECT_IDENTIFIER_URLPARAM = "projectIdentifier";

	public static final String GET_PROJECT_BY_IDENTIFIER_URL = "projects/{" + PROJECT_IDENTIFIER_URLPARAM + "}";

	public Project getProjectByIdentifier(String projectIdentifier) throws EntityNotFoundException {
		try {
			return new GetCall<Project>() {
				public Project getValue(ServiceCallResult result) {
					return result.getProject();
				}
			}.doCall(GET_PROJECT_BY_IDENTIFIER_URL, projectIdentifier);
		} catch (WrappedCheckedException e) {
			convertEntityNotFoundException(e);
			throw e;
		}
	}

	public static final String GET_ROLES_FOR_PROJECT_URL = "projects/{" + PROJECT_IDENTIFIER_URLPARAM + "}/roles";

	public String[] getRolesForProject(String projectIdentifier) throws EntityNotFoundException {
		try {
			return new GetCall<String[]>() {
				public String[] getValue(ServiceCallResult result) {
					return result.getRoles();
				}
			}.doCall(GET_ROLES_FOR_PROJECT_URL, projectIdentifier);
		} catch (WrappedCheckedException e) {
			convertEntityNotFoundException(e);
			throw e;
		}
	}

	public static final String UPDATE_PROJECT_URL = "projects/{" + PROJECT_IDENTIFIER_URLPARAM + "}";

	public Project updateProject(Project project) throws EntityNotFoundException, ValidationException {
		try {
			return new PostCall<Project>() {
				public Project getValue(ServiceCallResult result) {
					return result.getProject();
				}
			}.doCall(UPDATE_PROJECT_URL, project, project.getIdentifier());
		} catch (WrappedCheckedException e) {
			convertEntityNotFoundException(e);
			throw e;
		}
	}

	public static final String FIND_PROJECTS_URL = "projects/search";

	public QueryResult<Project> findProjects(ProjectsQuery query) {
		return new PostCall<QueryResult<Project>>() {
			public QueryResult<Project> getValue(ServiceCallResult result) {
				return result.getQueryResult();
			}
		}.doCall(FIND_PROJECTS_URL, query);
	}

	public static final String PROJECT_WATCH_URL = "projects/{" + PROJECT_IDENTIFIER_URLPARAM + "}/watch";

	public void watchProject(String projectIdentifier) throws EntityNotFoundException {
		Map<String, String> vars = new HashMap<String, String>();
		vars.put(PROJECT_IDENTIFIER_URLPARAM, projectIdentifier);

		try {
			delegate.postForEntity(computeUrl(PROJECT_WATCH_URL), null, Void.class, vars);
		} catch (WrappedCheckedException e) {
			convertEntityNotFoundException(e);
			throw (e);
		}
	}

	public Boolean isWatchingProject(String projectIdentifier) throws EntityNotFoundException {
		try {
			return new GetCall<Boolean>() {
				public Boolean getValue(ServiceCallResult result) {
					return result.getIsWatching();
				}
			}.doCall(PROJECT_WATCH_URL, projectIdentifier);

		} catch (WrappedCheckedException e) {
			convertEntityNotFoundException(e);
			throw (e);
		}
	}

	public static final String PROJECT_UNWATCH_URL = "projects/{" + PROJECT_IDENTIFIER_URLPARAM + "}/unwatch";

	public void unwatchProject(String projectIdentifier) throws EntityNotFoundException {
		Map<String, String> vars = new HashMap<String, String>();
		vars.put(PROJECT_IDENTIFIER_URLPARAM, projectIdentifier);

		try {
			delegate.postForEntity(computeUrl(PROJECT_UNWATCH_URL), null, Void.class, vars);
		} catch (WrappedCheckedException e) {
			convertEntityNotFoundException(e);
			throw (e);
		}
	}

	public static final String CREATE_SIGNUP_TOKEN_URL = "signup_token/";

	public SignUpToken createSignUpToken(SignUpToken token) throws ValidationException {
		try {
			return new PostCall<SignUpToken>() {
				public SignUpToken getValue(ServiceCallResult result) {
					return result.getSignUpToken();
				}
			}.doCall(CREATE_SIGNUP_TOKEN_URL, token);
		} catch (WrappedCheckedException e) {
			convertValidationException(e);
			throw (e);
		}
	}

	public static final String GET_UNUSED_SIGNUP_TOKENS_URL = "signup_tokens/";

	public List<SignUpToken> getUnusedSignUpTokens() {
		try {
			return new GetCall<List<SignUpToken>>() {
				public List<SignUpToken> getValue(ServiceCallResult result) {
					return result.getSignUpTokenList();
				}
			}.doCall(GET_UNUSED_SIGNUP_TOKENS_URL);
		} catch (WrappedCheckedException e) {
			throw (e);
		}
	}

	public static final String EMAIL_URLPARAM = "email";

	public static final String SEND_SIGNUP_INVITATION_URL = "signup_invitation/{" + EMAIL_URLPARAM + "}";

	public void sendSignUpInvitation(String email) throws EntityNotFoundException {
		// FIXME this cannot be templated as usual because it returns void rather than a ServiceCallResult
		try {
			delegate.postForLocation(computeUrl(SEND_SIGNUP_INVITATION_URL), null, email);
		} catch (WrappedCheckedException e) {
			convertEntityNotFoundException(e);
			throw (e);
		}
	}

	public static final String GET_SIGNUP_TOKEN_URL = "signup/{" + TOKEN_URLPARAM + "}";

	public SignUpToken getSignUpToken(String token) {
		try {
			return new GetCall<SignUpToken>() {
				public SignUpToken getValue(ServiceCallResult result) {
					return result.getSignUpToken();
				}
			}.doCall(GET_SIGNUP_TOKEN_URL, token);
		} catch (WrappedCheckedException e) {
			throw (e);
		}
	}

	public static final String CREATE_PROFILE_WITH_SIGNUP_TOKEN_URL = "signup/{" + TOKEN_URLPARAM + "}";

	public Long createProfileWithSignUpToken(Profile profile, String token) throws ValidationException {
		try {
			return new PostCall<Long>() {
				public Long getValue(ServiceCallResult result) {
					return result.getProfileId();
				}
			}.doCall(CREATE_PROFILE_WITH_SIGNUP_TOKEN_URL, profile, token);

		} catch (WrappedCheckedException e) {
			convertValidationException(e);
			throw (e);
		}
	}

	public static final String CREATE_PROFILE_URL = "signup";

	public Long createProfile(Profile profile) throws ValidationException {
		try {
			return new PostCall<Long>() {
				public Long getValue(ServiceCallResult result) {
					return result.getProfileId();
				}
			}.doCall(CREATE_PROFILE_URL, profile);
		} catch (WrappedCheckedException e) {
			convertValidationException(e);
			throw (e);
		}
	}

	public static final String USER_EMAIL_PARAM = "userEmail";
	public static final String INVITE_USER_TO_PROJECT_URL = "project/{" + PROJECT_IDENTIFIER_URLPARAM + "}/invite/{"
			+ USER_EMAIL_PARAM + "}";

	public String inviteUserForProject(String email, String projectIdentifier) throws EntityNotFoundException {
		try {
			return new GetCall<String>() {
				public String getValue(ServiceCallResult result) {
					return result.getToken();
				}
			}.doCall(INVITE_USER_TO_PROJECT_URL, projectIdentifier, email);
		} catch (WrappedCheckedException e) {
			convertEntityNotFoundException(e);
			throw (e);
		}
	}

	public static final String ACCEPT_INVITE_URL = "accept/{" + TOKEN_URLPARAM + "}";

	public void acceptInvitation(String invitationToken) throws EntityNotFoundException {
		try {
			delegate.getForObject(computeUrl(ACCEPT_INVITE_URL), Void.class, invitationToken);
		} catch (WrappedCheckedException e) {
			convertEntityNotFoundException(e);
			throw (e);
		}

	}

	public static final String GET_PROJECT_INVITATION_TOKEN_URL = "invitation/{" + TOKEN_URLPARAM + "}";

	public ProjectInvitationToken getProjectInvitationToken(String token) throws EntityNotFoundException {
		try {
			return new GetCall<ProjectInvitationToken>() {
				public ProjectInvitationToken getValue(ServiceCallResult result) {
					return result.getProjectInvitationToken();
				}
			}.doCall(GET_PROJECT_INVITATION_TOKEN_URL, token);
		} catch (WrappedCheckedException e) {
			throw (e);
		}
	}

	public static final String CREATE_SSH_PUBLIC_KEY_URL = "ssh-key";

	public SshPublicKey createSshPublicKey(SshPublicKeySpec publicKey) throws ValidationException {
		try {
			return new PostCall<SshPublicKey>() {
				public SshPublicKey getValue(ServiceCallResult result) {
					return result.getSshPublicKey();
				}
			}.doCall(CREATE_SSH_PUBLIC_KEY_URL, publicKey);
		} catch (WrappedCheckedException e) {
			convertValidationException(e);
			throw (e);
		}
	}

	public static final String LIST_SSH_PUBLIC_KEYS_URL = "ssh-keys";

	public List<SshPublicKey> listSshPublicKeys() {
		try {
			return new GetCall<List<SshPublicKey>>() {
				public List<SshPublicKey> getValue(ServiceCallResult result) {
					return result.getSshPublicKeyList();
				}
			}.doCall(LIST_SSH_PUBLIC_KEYS_URL);
		} catch (WrappedCheckedException e) {
			throw (e);
		}
	}

	public static final String SSH_KEY_ID_PARAM = "keyId";

	public static final String DELETE_SSH_PUBLIC_KEYS = "ssh-key/{" + SSH_KEY_ID_PARAM + "}";

	public void removeSshPublicKey(Long publicKeyId) throws EntityNotFoundException {
		// Default is not supported
		throw new UnsupportedOperationException();
	}

	public Boolean isProjectCreateAvailable() {
		throw new UnsupportedOperationException();
	}

	public List<ProjectServiceStatus> computeProjectServicesStatus(String projectId) throws EntityNotFoundException {
		try {
			return new GetCall<List<ProjectServiceStatus>>() {

				@Override
				public List<ProjectServiceStatus> getValue(ServiceCallResult result) {
					return result.getProjectServiceStatusList();
				}
			}.doCall("projects/{" + PROJECT_IDENTIFIER_URLPARAM + "}/status", projectId);
		} catch (WrappedCheckedException e) {
			convertEntityNotFoundException(e);
			throw e;
		}
	}

	public PasswordResetToken getPasswordResetToken(String token) {
		throw new UnsupportedOperationException();
	}

	public Organization createOrganization(Organization org) throws ValidationException {
		try {
			return new PostCall<Organization>() {
				public Organization getValue(ServiceCallResult result) {
					return result.getOrganization();
				}
			}.doCall("organization", org);
		} catch (WrappedCheckedException e) {
			convertValidationException(e);
			throw e;
		}
	}

	public Organization getOrganizationByIdentfier(String orgIdentifier) throws EntityNotFoundException {
		try {
			return new GetCall<Organization>() {
				public Organization getValue(ServiceCallResult result) {
					return result.getOrganization();
				}
			}.doCall("organization/{id}", orgIdentifier);
		} catch (WrappedCheckedException e) {
			convertEntityNotFoundException(e);
			throw e;
		}
	}

	public Organization updateOrganization(Organization organization) throws EntityNotFoundException,
			ValidationException {
		try {
			return new PostCall<Organization>() {
				public Organization getValue(ServiceCallResult result) {
					return result.getOrganization();
				}
			}.doCall("organization/update", organization);
		} catch (WrappedCheckedException e) {
			convertEntityNotFoundException(e);
			throw e;
		}
	}

	public List<PoolStatus> computeNodePoolStatus() {
		throw new UnsupportedOperationException();
	}

	public String getClientVersion() {
		return ProfileWebService.VERSION;
	}
}
