/*******************************************************************************
 * Copyright (c) 2010, 2013 Tasktop Technologies
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Tasktop Technologies - initial API and implementation
 ******************************************************************************/
package com.tasktop.c2c.server.common.service.web;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;

import org.apache.commons.httpclient.HttpStatus;
import org.codehaus.jackson.map.DeserializationConfig;
import org.codehaus.jackson.map.ObjectMapper;

import com.tasktop.c2c.server.common.service.AuthenticationException;
import com.tasktop.c2c.server.common.service.ConcurrentUpdateException;
import com.tasktop.c2c.server.common.service.EntityNotFoundException;
import com.tasktop.c2c.server.common.service.HttpStatusCodeException;
import com.tasktop.c2c.server.common.service.InsufficientPermissionsException;
import com.tasktop.c2c.server.common.service.ServerRuntimeException;
import com.tasktop.c2c.server.common.service.ValidationException;
import com.tasktop.c2c.server.common.service.WrappedCheckedException;

/**
 * Rest service call response error handler to be used with ApacheHttpRestClientDelegate. This class handles a subset of
 * DCS server side exceptions which don't have Spring dependencies
 * 
 * @See com.tasktop.c2c.server.common.service.web.spring.Error
 * @See com.tasktop.c2c.server.common.service.web.spring.ServiceResponseErrorHandler
 * 
 * @author Danny Ju (danny.ju@oracle.com)
 * 
 */
class ApacheResponseErrorHandler {
	private static Map<String, Class<? extends Throwable>> nameToExceptionType = new HashMap<String, Class<? extends Throwable>>();
	private static Set<Class<? extends Throwable>> exceptionTypes = new HashSet<Class<? extends Throwable>>();
	static {
		register(ValidationException.class);
		register(EntityNotFoundException.class);
		register(ConcurrentUpdateException.class);
		register(AuthenticationException.class);
		register(InsufficientPermissionsException.class);
		register(HttpStatusCodeException.class);
	}

	private static void register(Class<? extends Throwable> exceptionType) {
		if (exceptionType != ValidationException.class) {
			try {
				exceptionType.getConstructor(String.class);
			} catch (NoSuchMethodException e) {
				throw new IllegalStateException(e);
			}
		}

		Class<? extends Throwable> previous = nameToExceptionType.put(exceptionType.getSimpleName(), exceptionType);
		if (previous != null) {
			throw new IllegalStateException();
		}
		exceptionTypes.add(exceptionType);
	}

	public Throwable getException(String errorCode, String message) {
		Throwable e;
		Class<? extends Throwable> exceptionClass = nameToExceptionType.get(errorCode);
		if (exceptionClass == null) {
			exceptionClass = nameToExceptionType.get(message.substring(0, message.indexOf(':')));
		}
		if (exceptionClass != null) {
			if (exceptionClass.equals(ValidationException.class)) {
				e = new ValidationException(message);
			} else {
				try {
					e = exceptionClass.getConstructor(String.class).newInstance(message);
				} catch (Throwable t) {
					throw new IllegalStateException(message, t);
				}
			}

		} else {
			e = new ServerRuntimeException(message);
		}

		return e;
	}

	public boolean hasError(int statusCode) {
		return Series.valueOf(statusCode) == Series.CLIENT_ERROR || Series.valueOf(statusCode) == Series.SERVER_ERROR;
	}

	public void handleError(int statusCode, InputStream in) throws IOException {
		Throwable e = null;
		if (statusCode == HttpStatus.SC_UNAUTHORIZED) {
			e = new AuthenticationException("UNAUTHORIZED");
		} else {

			try {
				ObjectMapper m = new ObjectMapper();
				m.configure(DeserializationConfig.Feature.FAIL_ON_UNKNOWN_PROPERTIES, false);
				ErrorResult results = m.readValue(new InputStreamReader(in), ErrorResult.class);
				Object obj = results.getError();

				if (obj instanceof LinkedHashMap) {
					@SuppressWarnings("rawtypes")
					LinkedHashMap map = (LinkedHashMap) obj;
					String errorCode = map.get("errorCode") != null ? map.get("errorCode").toString() : null;
					if (errorCode != null) {
						String detail = map.get("detail").toString();
						e = getException(errorCode, detail);
					}

				} else {
					e = new WrappedCheckedException(new Exception(obj.toString()));
				}
			} finally {
				in.close();
			}
		}

		if (e instanceof RuntimeException) {
			throw (RuntimeException) e;
		}
		// Due to the declaration of this we can't throw checked exceptions here.
		throw new WrappedCheckedException(e);
	}

	/**
	 * Java 5 enumeration of HTTP status series.
	 * <p>
	 * Retrievable via {@link HttpStatus#series()}.
	 */
	public static enum Series {

		INFORMATIONAL(1), SUCCESSFUL(2), REDIRECTION(3), CLIENT_ERROR(4), SERVER_ERROR(5);

		private final int value;

		private Series(int value) {
			this.value = value;
		}

		/**
		 * Return the integer value of this status series. Ranges from 1 to 5.
		 */
		public int value() {
			return this.value;
		}

		private static Series valueOf(int code) {
			int seriesCode = code / 100;
			for (Series series : values()) {
				if (series.value == seriesCode) {
					return series;
				}
			}
			throw new IllegalArgumentException("No matching constant for [" + code + "]");
		}

	}

	private static final class ErrorResult {
		private Object error;

		public Object getError() {
			return error;
		}

		@SuppressWarnings("unused")
		public void setError(Object error) {
			this.error = error;
		}
	}

}
