"""
A GUI controller for a Tic-Tac-Toe game.

@author: Serita Nelesen
@date: Fall, 2014
@author: kvlinden
@date: Summer, 2015
@author: ka37
@date: Fall, 2019
@author: kvlinden
@date: Spring, 2021 - ported to GuiZero; removed exceptions
"""

from guizero import App, Drawing
from game import TicTacToe

class TicTacToeApp:

    def __init__(self, app):

        # Initialize a 3X3 Tic-Tac-Toe game.
        self.cell_width = 3
        self.game = TicTacToe(self.cell_width)

        self.box_size = 140

        # Configure the application GUI.
        app.title = 'Tic Tac Toe'
        app.width = self.cell_width * self.box_size
        app.height = app.width + 50
        app.font = 'Helvetica'
        app.text_size = 48
        self.drawing = Drawing(app, width=app.width, height=app.height)
        self.drawing.when_clicked = self.clicked

        self.set_message()

        # Draw the (empty) board.
        self.draw_board()

    def draw_board(self):
        """ Draw the current state of the board on the GUI using the game model state """
        self.drawing.clear()

        # Vertical lines
        for i in range(1, 3):
            self.drawing.line(i * self.box_size, 0, i * self.box_size, 3 * self.box_size)

        # Horizontal lines
        for i in range(1, 3):
            self.drawing.line(0, i * self.box_size, 3 * self.box_size, i * self.box_size)

        # Player data
        for row in range(3):
            for column in range(3):
                cell = self.game.get_cell(row, column)
                self.drawing.text((column + .5) * self.box_size, (row + .5) * self.box_size, text=cell)

        # Message
        self.drawing.text(5, 3 * self.box_size, text=self.message)

    def clicked(self, event):
        column = event.x // self.box_size
        row = event.y // self.box_size
        if 0 <= row < 3 and 0 <= column < 3 and self.game.get_winner() == None:
            self.process_click(row, column)

    def process_click(self, row, column):
        if self.game.get_cell(row, column) != ' ':
            # Tried to click somewhere that already had contents
            return
        self.game.make_move(row, column)
        winner = self.game.get_winner()
        if winner != None:
            self.message = '{} wins!'.format(winner)
        elif self.game.is_cat_game():
            self.message = 'Draw...'
        else:
            self.set_message()
        self.draw_board()

    def set_message(self):
        """ Set the message to which player's turn it is. """
        player = self.game.get_player()
        self.message = player + "'s turn"




app = App()
TicTacToeApp(app)
app.display()