"""
This module implements SimpleDraw, a canvas-based, figure-paint application.

@author: Keith VanderLinden (kvlinden)
@date: Summer, 2015
@date: Summer, 2021 - ported to GuiZero; cleaned up for new video demo
"""

from guizero import App, Drawing, PushButton, Box, ButtonGroup
from figure import *


class SimpleDraw:
    """Creates a GUI for the SimpleDraw figure editor"""

    def __init__(self, app):
        """Constructs a figure-drawing GUI"""

        app.title = 'SimpleDraw'
        unit = 300
        control_unit = 80
        app.width = unit
        app.height = unit + control_unit
        self.bg_color = 'white'

        # Create the main drawing canvas.
        self.drawing = Drawing(app, width=unit, height=unit)
        self.drawing.bg = self.bg_color
        self.drawing.when_left_button_pressed = self.process_mouse_press
        self.drawing.when_mouse_dragged = self.process_mouse_motion
        self.drawing.when_left_button_released = self.process_mouse_release

        # Create the control panel.
        self.figure_choice = ButtonGroup(
            app, ['line', 'rectangle', 'ellipse', 'squiggle'], selected='line',
            horizontal=True
        )
        self.color_choice = ButtonGroup(
            app, ['black', 'red', 'green', 'blue'], selected='black',
            horizontal = True
        )
        self.fill_choice = ButtonGroup(
            app, ['filled', 'not-filled'], selected='filled',
            horizontal=True
        )

        # Create an (empty) list of model objects and temporary figure and
        # position objects for use in drawing animation.
        self.figures = []
        self.temporary_figure = None
        self.saved_position = None

    def process_mouse_press(self, event):
        """Starts a new figure where the user presses the mouse based on the
        mode settings
        """
        self.saved_position = (event.x, event.y)
        self.temporary_figure = (self.create_figure(event))
        self.temporary_figure.draw(self.drawing)

    def process_mouse_motion(self, event):
        """Displays a temporary version of the figure and erase the previous
        temporary version
        """
        if self.figure_choice.value == 'squiggle':
            # Squiggles are special in that they don't have temporary versions.
            self.temporary_figure.add_point((event.x, event.y))
            self.temporary_figure.draw(self.drawing)
        else:
            # Erase the temporary version of the figure by redrawing it using
            # the drawing canvas background color.
            self.temporary_figure.color = self.bg_color
            self.temporary_figure.draw(self.drawing)
            self.temporary_figure = (self.create_figure(event))
            self.temporary_figure.draw(self.drawing)

    def process_mouse_release(self, event):
        """Create and save the final version of the figure"""
        if self.figure_choice.value == 'squiggle':
            self.temporary_figure.add_point((event.x, event.y))
            self.figures.append(self.temporary_figure)
        else:
            self.figures.append(self.create_figure(event))
        self.draw_figures()

    def get_fill_choice(self):
        """Converts the fill button choice to a boolean value"""
        return self.fill_choice.value == 'filled'
        
    def create_figure(self, event):
        """Creates a figure based on the given mode settings -
        The figure can be either temporary or permanent. The calls to the
        constructors for closed figures must convert upper-left and lower-right
        points into upper-left point and dimensions for the constructors.
       """
        if self.figure_choice.value == 'line':
            return Line(self.saved_position, (event.x, event.y),
                        color=self.color_choice.value)
        elif self.figure_choice.value == 'rectangle':
            return Rectangle(self.saved_position,
                             (event.x - self.saved_position[0],
                              event.y - self.saved_position[1]),
                             color=self.color_choice.value,
                             filled=self.get_fill_choice())
        elif self.figure_choice.value == 'ellipse':
            return Ellipse(self.saved_position,
                           (event.x - self.saved_position[0],
                            event.y - self.saved_position[1]),
                           color=self.color_choice.value,
                           filled=self.get_fill_choice())
        elif self.figure_choice.value == 'squiggle':
            return Squiggle(self.saved_position,
                            color=self.color_choice.value)
    
    def draw_figures(self):
        """Redraws all the stored figures on a fresh background"""
        self.drawing.clear()
        for figure in self.figures:
            figure.draw(self.drawing)

        
app = App()
SimpleDraw(app)
app.display()
