"""
This module implements the figure classe hierarchy for a drawing canvas-based
SimpleDraw application.

@author: Keith VanderLinden (kvlinden)
@date: Summer, 2015
@date: Summer, 2021 - ported to GuiZero; cleaned up for new video demo
"""

import math
from helpers import distance


class Figure:
    """The base class for all figures"""
    
    def __init__(self, start, color):
        """Instantiates a base figure object - not called directly"""
        self.start = start
        self.color = color 

    def __str__(self):
        """Creates a printable string for all figure objects"""
        return 'Figure: {0} {1}'.format(self.start, self.color)


class Line(Figure):
    """A figure class for simple lines"""
    
    def __init__(self, start, end, color='black'):
        """Instantiates a line object derived from the figure class
        :param start: starting point of the line (typle)
        :param end: ending point of the line (tuple)
        :param color: color of the line (color specification)
        """
        Figure.__init__(self, start, color)
        self.end = end

    def draw(self, drawing):
        """Renders the line
        :param drawing: the drawing canvas on which to do the rendering
        """
        drawing.line(self.start[0], self.start[1], self.end[0], self.end[1],
                     color=self.color)
    
    def get_length(self):
        """Returns the length of the line in pixels"""
        return distance(self.start[0], self.start[1], self.end[0], self.end[1])

    def __str__(self):
        """Creates a printable string for line objects"""
        return 'Line: {0} {1} {2}'.format(self.start, self.end, self.color)


class Squiggle(Figure):
    """A figure class for a user-drawn, potentially curved line"""
    
    def __init__(self, start, color='black'):
        """Instantiates a curved line object derived from the figure class
        :param start: starting point of the squiggle (tuple)
        :param color: color of the squiggle line (color specification)
        """
        Figure.__init__(self, start, color)
        self.points = [start]
        
    def add_point(self, point):
        """Adds a point to the squiggle - Points must be added sequentially.
        :param point: new point for the squiggle (tuple)
        """
        self.points.append(point)
        
    def get_length(self):
        """Returns the length of the squiggle in pixels defined as the length
        of each line segment defined by the points
        """
        result = 0
        if len(self.points) > 1:
            for i in range(1, len(self.points)):
                result += distance(self.points[i - 1][0], self.points[i - 1][1],
                                   self.points[i][0], self.points[i][1])
        return result

    def draw(self, drawing):
        """Renders the squiggle by connecting the points in order
        :param drawing: the drawing canvas on which to do the rendering
        """
        if len(self.points) > 1:
            for i in range(1, len(self.points)):
                drawing.line(self.points[i - 1][0], self.points[i - 1][1],
                             self.points[i][0], self.points[i][1],
                             color=self.color)

    def __str__(self):
        """Creates a printable string for squiggle objects"""
        return 'Squiggle: {0} {1}'.format(self.points, self.color)


class ClosedFigure(Figure):
    """A figure class for boxed, closed figures"""

    def __init__(self, start, dimensions, color, filled):
        """Instantiates a closed figure object derived from the figure class -
        not called directly.
        """
        Figure.__init__(self, start, color)
        self.dimensions = dimensions
        self.filled = filled

    def get_end_point(self):
        """Computes the end-point of the figure based on the start-point,
        width and height
        """
        return (self.start[0] + self.dimensions[0],
                self.start[1] + self.dimensions[1])
    
    def get_fill_color(self):
        """Returns the color for filled figures and None for non-filled figures -
        All figures are outlined in the figure color, filled figures are filled 
        with that same color as well.
        """
        if self.filled:
            return self.color
        else:
            return None
        


class Rectangle(ClosedFigure):
    """A figure class for a simple rectangle"""
    
    def __init__(self, start, dimensions, color='black', filled=True):
        """Instantiates a rectangle object derived from the figure class
        :param start: starting point of rectangle (tuple)
        :param dimensions: (width, height) of the rectangle (tuple)
        :param color: fill color of the rectangle (color specification)
        :param filled: whether or note the rectangle is filled in (boolean)
        """
        ClosedFigure.__init__(self, start, dimensions, color, filled)
    
    def draw(self, drawing):
        """Renders the rectangle
        :param drawing: the drawing canvas on which to do the rendering
        """
        end_point = self.get_end_point()
        drawing.rectangle(
            self.start[0], self.start[1],
            end_point[0], end_point[1],
            color=self.get_fill_color(),
            outline=True, outline_color=self.color
        )

    def get_area(self):
        """Returns the area of the rectangle in square pixels"""
        return self.dimensions[0] * self.dimensions[1]

    def __str__(self):
        """Creates a printable string for rectangle objects"""
        return 'Rectangle: {0} {1} {2} {3}'.format(self.start, self.dimensions,
                                                   self.color, self.filled)


class Ellipse(ClosedFigure):
    """A figure class for ellipses"""
    
    def __init__(self, start, dimensions, color='black', filled=True):
        """Instantiates an ellipse object derived from the figure class
        :param start: starting point of ellipse bounding box (tuple)
        :param dimensions: (width, height) of the ellipse bounding box (tuple)
        :param color: fill color of the ellipse (color specification)
        :param filled: whether or not the ellipse is outlined (boolean)
        """
        """Instantiates a ellipse object derived from the figure class"""
        ClosedFigure.__init__(self, start, dimensions, color, filled)
    
    def draw(self, drawing):
        """Renders the ellipse
        :param drawing: the drawing canvas on which to do the rendering
        """
        end_point = self.get_end_point()
        drawing.oval(
            self.start[0], self.start[1],
            end_point[0], end_point[1],
            color=self.get_fill_color(),
            outline=True, outline_color=self.color
        )

    def get_area(self):
        """Return the area of the ellipse in square pixels"""
        return math.pi * self.dimensions[0] * self.dimensions[1]

    def __str__(self):
        """Creates a printable string for ellipse objects"""
        return 'Ellipse: {0} {1} {2} {3}'.format(self.start, self.dimensions,
                                                 self.color, self.filled)
