import turtle
import random


class World:
    """
    A World object represents a grid of things. Each thing has a location in the grid. The
    world can be asked to move things around, add things, and remove things. The world can
    also be asked to simulate one time step, which involves asking one of the things to do
    something. The world can also be asked to draw itself.

    The world is drawn using the turtle graphics library. The world is drawn with the origin
    in the lower left corner. The x-axis increases to the right and the y-axis increases
    upwards. The world is drawn with a grid of lines, with each cell in the grid being a
    square. The world is drawn with the turtle graphics library, which is a bit slow. If you
    want to speed up the simulation, you can set the SHOW variable to False in the
    bearfishsimulation.py file.

    The world is also responsible for keeping track of the things (like bears and fish) that
    are in the world. It keeps them in a 2D data structure called `grid`, which is a list of
    lists. Each element of the grid is either None or a reference to a thing. The grid is
    used to quickly look up what is at a given location in the world.

    The world also keeps a list of all the things in the world. This list is used to quickly
    pick a thing to simulate at each time step. The list is shuffled at the beginning of each
    time step to ensure that the things are simulated in a random order.

    Attributes:

    - max_x (int): The width of the world.
    - max_y (int): The height of the world.
    - thing_list (list): A list of all the things in the world.
    - grid (list of lists): A 2D data structure that represents the world.
    - screen (turtle.Screen): The turtle screen that the world is drawn on.
    - pen (turtle.Turtle): The turtle pen that is used to draw the grid
    """
    def __init__(self, max_x, max_y):
        """
        Create a new world with the given dimensions.

        Arguments:
        - max_x: The width of the world.
        - max_y: The height of the world.
        """
        self.max_x = max_x
        self.max_y = max_y
        self.thing_list = []
        self.grid = []

        for i in range(self.max_y):
            row = []
            for j in range(self.max_x):
                row.append(None)
            self.grid.append(row)

        self.screen = turtle.Screen()
        self.pen = turtle.Turtle()
        self.screen.setworldcoordinates(0, 0, self.max_x - 1, self.max_y - 1)
        self.screen.addshape("Bear.gif")
        self.screen.addshape("Fish.gif")
        self.pen.hideturtle()

    def draw_grid(self):
        self.screen.tracer(False)
        self.pen.forward(self.max_x - 1)
        self.pen.left(90)
        self.pen.forward(self.max_y - 1)
        self.pen.left(90)
        self.pen.forward(self.max_x - 1)
        self.pen.left(90)
        self.pen.forward(self.max_y - 1)
        self.pen.left(90)
        for i in range(self.max_y - 1):
            self.pen.forward(self.max_x - 1)
            self.pen.backward(self.max_x - 1)
            self.pen.left(90)
            self.pen.forward(1)
            self.pen.right(90)
        self.pen.forward(1)
        self.pen.right(90)
        for i in range(self.max_x - 2):
            self.pen.forward(self.max_y - 1)
            self.pen.backward(self.max_y - 1)
            self.pen.left(90)
            self.pen.forward(1)
            self.pen.right(90)
        self.screen.tracer(True)

    def freeze_world(self):
        self.screen.exitonclick()

    def add_thing(self, thing, x, y):
        thing.x = x
        thing.y = y
        self.grid[y][x] = thing
        self.thing_list.append(thing)
        thing.appear()

    def del_thing(self, thing):
        # hide() is needed because there is no screen.remove(turtle)
        # see https://github.com/python/cpython/issues/93642
        thing.hide()
        self.grid[thing.y][thing.x] = None
        self.thing_list.remove(thing)

    def move_thing(self, old_x, old_y, new_x, new_y):
        thing = self.grid[old_y][old_x]
        self.grid[new_y][new_x] = thing
        self.grid[old_y][old_x] = None

    def live_one_step(self):
        if self.thing_list == []:
            return
        # It's important that we have only one thing do its thing at each time step. If we 
        # tried to simulate everything, suppose a bear eats a fish: we'd need to make sure
        # not to try to simulate that fish. In general we need to be careful going through
        # a list that can change as we're going through it.
        thing = random.randrange(len(self.thing_list))
        random_thing = self.thing_list[thing]
        random_thing.live_one_step()

    def get_neighbor_locations(self, x, y):
        offsets = [(-1,1) ,(0,1) ,(1,1),
                   (-1,0)        ,(1,0),
                   (-1,-1),(0,-1),(1,-1)]
        neighbor_locations = []
        for offset in offsets:
            new_x = x + offset[0]
            new_y = y + offset[1]
            if 0 <= new_x < self.max_x and 0 <= new_y < self.max_y:
                neighbor_locations.append((new_x, new_y))
        return neighbor_locations        

    def get_neighbors_of_kind(self, x, y, kind):
        neighbors_of_kind = []
        for loc in self.get_neighbor_locations(x, y):
            x = loc[0]
            y = loc[1]
            neighbor = self.look_at_location(x, y)
            if neighbor is not None and neighbor.kind == kind:
                neighbors_of_kind.append(neighbor)
        return neighbors_of_kind

    def get_empty_neighbor_locations(self, x, y):
        empty_neighbor_locs = []
        for loc in self.get_neighbor_locations(x, y):
            x = loc[0]
            y = loc[1]
            neighbor = self.look_at_location(x, y)
            if neighbor is None:
                empty_neighbor_locs.append((x, y))
        return empty_neighbor_locs

    def is_location_empty(self, x, y):
        return self.grid[y][x] == None

    def look_at_location(self, x, y):
        return self.grid[y][x]

    def count_items(self, kind):
        """Return the number of items in the grid for which thing.kind == kind."""
        return 0 # TODO: count, using a nested loop over grid
